// Copyright (c) 2012-2017 VideoStitch SAS
// Copyright (c) 2018 stitchEm

#include "testing_common.hpp"

#include "../include/metadataParser.hpp"

#include "libvideostitch/imuData.hpp"
#include "libvideostitch/logging.hpp"
#include "libvideostitch/orah/exposureData.hpp"

#include <vector>

namespace VideoStitch {

namespace Testing {

bool parseAndCompare(int cameraID, std::string input, std::pair<bool, IMU::Measure> imu,
                     std::map<videoreaderid_t, Metadata::Exposure> exposure,
                     std::map<videoreaderid_t, Metadata::WhiteBalance> whiteBalance,
                     std::map<videoreaderid_t, Metadata::ToneCurve> toneCurve) {
  std::map<videoreaderid_t, Metadata::Exposure> exposureParsed;
  std::map<videoreaderid_t, Metadata::WhiteBalance> whiteBalanceParsed;
  std::map<videoreaderid_t, Metadata::ToneCurve> toneCurveParsed;

  std::pair<bool, IMU::Measure> imuParsed;

  std::map<videoreaderid_t, Metadata::Exposure> exposureTest;
  std::map<videoreaderid_t, Metadata::WhiteBalance> whiteBalanceTest;
  std::map<videoreaderid_t, Metadata::ToneCurve> toneCurveTest;

  for (auto& kv : exposure) {
    exposureTest[kv.first + cameraID] = kv.second;
  }
  for (auto& kv : whiteBalance) {
    whiteBalanceTest[kv.first + cameraID] = kv.second;
  }
  for (auto& kv : toneCurve) {
    toneCurveTest[kv.first + cameraID] = kv.second;
  }

  bool success = MetadataParser::parse(input, cameraID, imuParsed, exposureParsed, whiteBalanceParsed, toneCurveParsed);

  ENSURE_EQ(imu, imuParsed);
  ENSURE_EQ(exposureTest, exposureParsed);
  ENSURE_EQ(whiteBalanceTest, whiteBalanceParsed);
  ENSURE_EQ(toneCurveTest, toneCurveParsed);
  return success;
}

bool parseAndCompare(std::string input, const std::pair<bool, IMU::Measure>& imu,
                     const std::map<videoreaderid_t, Metadata::Exposure>& exposure,
                     const std::map<videoreaderid_t, Metadata::WhiteBalance>& whiteBalance,
                     const std::map<videoreaderid_t, Metadata::ToneCurve>& toneCurve) {
  return parseAndCompare(0, input, imu, exposure, whiteBalance, toneCurve) &&
         parseAndCompare(2, input, imu, exposure, whiteBalance, toneCurve);
}

void testSyntheticData() {
  // incomplete parsing, but don't crash

  // start marker missing
  ENSURE(
      !parseAndCompare("idx=6175,ts=205832,"
                       "xgy=-368,ygy=19,zgy=1885,xac=-536,yac=736,zac=-16932,xma=-1805,yma=-1095,zma=5234,"
                       "temp=27,"
                       "sen=0,iso=357,sht=0.02000000,sht_mx=0.00000000,"
                       "r=5966,g=4096,b=9863,"
                       "sen=1,iso=322,sht=0.02000000,sht_mx=0.00000000,"
                       "r=5967,g=4096,b=9861",
                       {false, {}}, {}, {}, {}));

  // sensor missing
  ENSURE(
      !parseAndCompare("MetaStartMarker,idx=6175,ts=205832,"
                       "xgy=-368,ygy=19,zgy=1885,xac=-536,yac=736,zac=-16932,xma=-1805,yma=-1095,zma=5234,"
                       "temp=27,"
                       "iso=357,sht=0.02000000,sht_mx=0.00000000,"
                       "r=5966,g=4096,b=9863,",
                       {true, {205832000, -536, 736, -16932, -368, 19, 1885, -1805, -1095, 5234, 27}}, {}, {}, {}));

  // invalid sensor ID
  ENSURE(
      !parseAndCompare("MetaStartMarker,idx=6175,ts=205832,"
                       "xgy=-368,ygy=19,zgy=1885,xac=-536,yac=736,zac=-16932,xma=-1805,yma=-1095,zma=5234,"
                       "temp=27,"
                       "sen=2,iso=357,sht=0.02000000,sht_mx=0.00000000,"
                       "r=5966,g=4096,b=9863,",
                       {true, {205832000, -536, 736, -16932, -368, 19, 1885, -1805, -1095, 5234, 27}}, {}, {}, {}));

  // truncated tone curve
  ENSURE(!parseAndCompare(
      "MetaStartMarker,idx=5521,ts=184032,"
      "xgy=-344,ygy=-15,zgy=77,xac=-720,yac=132,zac=-16912,xma=-2616,yma=-1363,zma=6010,"
      "temp=30,"
      "sen=0,"
      "tcStart,0,4,8,13,17,22,26,31,35,40,45,49,54,58,63,67,72,76,81,86,90,95,99,104,108,113,118,122,127,131,136,140,"
      "145,150,154,159,164,168,173,177,182,187,192,197,202,206,211,215,220,225,230,234,239,244,248,253,258,262,267,272,"
      "276,281,286,290,295,300,304,309,314,319,323,328,332,337,342,347,351,356,360,365,370,374,379,384,389,393,398,402,"
      "407,412,416,421,426,430,435,439,444,449,454,458,463,468,472,477,482,487,491,496,500,505,510,515,519,524,528,533,"
      "538,542,547,552,556,561,566,570,575,580,585,589,594,599,603,608,612,617,621,626,630,634,639,643,647,651,656,660,"
      "664,668,673,677,681,686,690,694,698,702,706,710,714,718,722,726,730,734,738,742,745,749,753,756,760,764",
      {true, {184032000, -720, 132, -16912, -344, -15, 77, -2616, -1363, 6010, 30}}, {}, {}, {}));

  // empty values
  ENSURE(
      !parseAndCompare("MetaStartMarker,idx=5488,ts=182932,"
                       "xgy=,ygy=,zgy=,xac=,yac=,zac=,xma=,yma=,zma=,"
                       "temp=,"
                       "sen=0,iso=,sht=,sht_mx=,"
                       "sen=1",
                       {false, {}}, {}, {}, {}));

  // empty sensor
  ENSURE(
      !parseAndCompare("MetaStartMarker,idx=5488,ts=182932,"
                       "xgy=-355,ygy=-21,zgy=48,xac=-620,yac=128,zac=-17004,xma=-2554,yma=-1365,zma=5998,"
                       "temp=30,"
                       "sen=,iso=434,sht=0.02000000,sht_mx=0.00000000,"
                       "sen=",
                       {true, {182932000, -620, 128, -17004, -355, -21, 48, -2554, -1365, 5998, 30}}, {}, {}, {}));

  // out of range?
  ENSURE(
      !parseAndCompare("MetaStartMarker,idx=5488,ts=182932,"
                       "xgy=-355,ygy=-21,zgy=48,xac=-620,yac=128,zac=-17004,xma=-2554,yma=-1365,zma=5998,"
                       "temp=30,"
                       "sen=,iso=18446744073709551617,sht=0.02000000,sht_mx=0.00000000,"
                       "sen=",
                       {true, {182932000, -620, 128, -17004, -355, -21, 48, -2554, -1365, 5998, 30}}, {}, {}, {}));
}

void testRecordedDataFw0177() {
  // MetaStartMarker,idx=5055,ts=168499,xgy=-347,ygy=-40,zgy=25,xac=-688,yac=88,zac=-16932,xma=-2672,yma=-1428,zma=5876,temp=30,sen=0,sen=1

  ENSURE(
      parseAndCompare("MetaStartMarker,idx=5488,ts=182932,"
                      "xgy=-355,ygy=-21,zgy=48,xac=-620,yac=128,zac=-17004,xma=-2554,yma=-1365,zma=5998,"
                      "temp=30,"
                      "sen=0,iso=434,sht=0.02000000,sht_mx=0.00000000,"
                      "sen=1",
                      // IMU
                      {true, {182932000, -620, 128, -17004, -355, -21, 48, -2554, -1365, 5998, 30}},
                      // exposure
                      {{0, {182932000, 434, 0.02f, 0.0f}}},
                      // white balance
                      {},
                      // tone curve
                      {}));

  ENSURE(
      parseAndCompare("MetaStartMarker,idx=5054,ts=168466,"
                      "xgy=-298,ygy=-46,zgy=48,xac=-680,yac=176,zac=-16852,xma=-2672,yma=-1428,zma=5876,"
                      "temp=30,"
                      "sen=0,r=5935,g=4096,b=9899,"
                      "sen=1,r=5935,g=4096,b=9899",
                      // IMU
                      {true, {168466000, -680, 176, -16852, -298, -46, 48, -2672, -1428, 5876, 30}},
                      // exposure
                      {},
                      // white balance
                      {{0, {168466000, 5935, 4096, 9899}}, {1, {168466000, 5935, 4096, 9899}}},
                      // tone curve
                      {}));

  ENSURE(
      parseAndCompare("MetaStartMarker,idx=5057,ts=168566,"
                      "xgy=-322,ygy=-13,zgy=6,xac=-644,yac=40,zac=-16844,xma=-2672,yma=-1428,zma=5876,"
                      "temp=30,"
                      "sen=0,r=5936,g=4096,b=9899,sen=1",
                      // IMU
                      {true, {168566000, -644, 40, -16844, -322, -13, 6, -2672, -1428, 5876, 30}},
                      // exposure
                      {},
                      // white balance
                      {{0, {168566000, 5936, 4096, 9899}}},
                      // tone curve
                      {}));

  ENSURE(
      parseAndCompare("MetaStartMarker,idx=5494,ts=183132,"
                      "xgy=-275,ygy=-5,zgy=60,xac=-632,yac=40,zac=-16844,xma=-2592,yma=-1522,zma=5659,"
                      "temp=30,"
                      "sen=0,iso=436,sht=0.02000000,sht_mx=0.00000000,"
                      "r=5926,g=4096,b=9903,"
                      "sen=1,r=5926,g=4096,b=9903",
                      // IMU
                      {true, {183132000, -632, 40, -16844, -275, -5, 60, -2592, -1522, 5659, 30}},
                      // exposure
                      {{0, {183132000, 436, 0.02f, 0.0f}}},
                      // white balance
                      {{0, {183132000, 5926, 4096, 9903}}, {1, {183132000, 5926, 4096, 9903}}},
                      // tone curve
                      {}));

  ENSURE(parseAndCompare(
      "MetaStartMarker,idx=5521,ts=184032,"
      "xgy=-344,ygy=-15,zgy=77,xac=-720,yac=132,zac=-16912,xma=-2616,yma=-1363,zma=6010,"
      "temp=30,"
      "sen=0,"
      "tcStart,0,4,8,13,17,22,26,31,35,40,45,49,54,58,63,67,72,76,81,86,90,95,99,104,108,113,118,122,127,131,136,140,"
      "145,150,154,159,164,168,173,177,182,187,192,197,202,206,211,215,220,225,230,234,239,244,248,253,258,262,267,272,"
      "276,281,286,290,295,300,304,309,314,319,323,328,332,337,342,347,351,356,360,365,370,374,379,384,389,393,398,402,"
      "407,412,416,421,426,430,435,439,444,449,454,458,463,468,472,477,482,487,491,496,500,505,510,515,519,524,528,533,"
      "538,542,547,552,556,561,566,570,575,580,585,589,594,599,603,608,612,617,621,626,630,634,639,643,647,651,656,660,"
      "664,668,673,677,681,686,690,694,698,702,706,710,714,718,722,726,730,734,738,742,745,749,753,756,760,764,768,771,"
      "775,779,782,786,790,793,797,800,803,807,810,814,817,820,823,826,830,833,836,839,842,845,848,851,855,857,860,863,"
      "866,869,872,875,878,881,884,887,890,892,895,898,901,904,907,910,913,916,919,921,924,927,930,933,936,938,941,944,"
      "947,949,952,955,958,961,963,966,969,972,975,978,980,983,986,989,991,994,997,1000,1003,1005,1008,1011,1014,1017,"
      "1020,1023,"
      "sen=1",
      // IMU
      {true, {184032000, -720, 132, -16912, -344, -15, 77, -2616, -1363, 6010, 30}},
      // exposure
      {},
      // white balance
      {},
      // tone curve
      {{0,
        {184032000,
         {{0,   4,   8,   13,  17,  22,  26,  31,   35,   40,   45,   49,   54,   58,   63,   67,  72,  76,  81,  86,
           90,  95,  99,  104, 108, 113, 118, 122,  127,  131,  136,  140,  145,  150,  154,  159, 164, 168, 173, 177,
           182, 187, 192, 197, 202, 206, 211, 215,  220,  225,  230,  234,  239,  244,  248,  253, 258, 262, 267, 272,
           276, 281, 286, 290, 295, 300, 304, 309,  314,  319,  323,  328,  332,  337,  342,  347, 351, 356, 360, 365,
           370, 374, 379, 384, 389, 393, 398, 402,  407,  412,  416,  421,  426,  430,  435,  439, 444, 449, 454, 458,
           463, 468, 472, 477, 482, 487, 491, 496,  500,  505,  510,  515,  519,  524,  528,  533, 538, 542, 547, 552,
           556, 561, 566, 570, 575, 580, 585, 589,  594,  599,  603,  608,  612,  617,  621,  626, 630, 634, 639, 643,
           647, 651, 656, 660, 664, 668, 673, 677,  681,  686,  690,  694,  698,  702,  706,  710, 714, 718, 722, 726,
           730, 734, 738, 742, 745, 749, 753, 756,  760,  764,  768,  771,  775,  779,  782,  786, 790, 793, 797, 800,
           803, 807, 810, 814, 817, 820, 823, 826,  830,  833,  836,  839,  842,  845,  848,  851, 855, 857, 860, 863,
           866, 869, 872, 875, 878, 881, 884, 887,  890,  892,  895,  898,  901,  904,  907,  910, 913, 916, 919, 921,
           924, 927, 930, 933, 936, 938, 941, 944,  947,  949,  952,  955,  958,  961,  963,  966, 969, 972, 975, 978,
           980, 983, 986, 989, 991, 994, 997, 1000, 1003, 1005, 1008, 1011, 1014, 1017, 1020, 1023}}}}}));

  ENSURE(parseAndCompare(
      "MetaStartMarker,idx=5993,ts=199766,"
      "xgy=-368,ygy=64,zgy=1565,xac=-592,yac=-88,zac=-17012,xma=-2190,yma=-1448,zma=5079,temp=29,"
      "sen=0,r=5880,g=4096,b=9949,"
      "sen=1,r=5881,g=4096,b=9948,"
      "tcStart,0,4,8,13,17,22,26,31,35,40,44,49,53,58,62,67,71,76,80,84,89,93,98,102,107,111,116,121,125,130,134,139,"
      "143,148,152,157,161,166,170,175,180,185,189,194,199,204,208,213,217,222,227,231,236,240,245,250,254,259,263,268,"
      "273,277,282,286,291,295,300,305,309,314,318,323,328,332,337,342,346,351,355,360,364,369,374,378,383,388,392,397,"
      "401,406,410,415,419,424,429,433,438,442,447,452,456,461,466,470,475,479,484,488,493,498,502,507,511,516,521,525,"
      "530,534,539,543,548,553,557,562,567,571,576,580,586,590,595,599,603,608,612,616,621,625,629,633,638,642,646,651,"
      "655,659,663,667,672,676,680,684,688,692,697,701,705,709,713,717,720,724,728,732,736,740,743,747,751,755,758,762,"
      "766,769,773,777,780,784,787,791,795,798,801,805,808,811,815,818,821,825,828,831,834,838,841,844,847,850,853,856,"
      "859,862,865,868,871,874,877,880,883,887,889,892,895,899,902,905,908,911,914,917,920,923,925,928,931,935,937,940,"
      "943,946,949,952,955,958,961,964,967,969,972,975,978,981,984,987,990,993,996,999,1002,1005,1008,1011,1013,1016,"
      "1019,1023",
      // IMU
      {true, {199766000, -592, -88, -17012, -368, 64, 1565, -2190, -1448, 5079, 29}},
      // exposure
      {},
      // white balance
      {{0, {199766000, 5880, 4096, 9949}}, {1, {199766000, 5881, 4096, 9948}}},
      // tone curve
      {{1,
        {199766000,
         {{0,   4,   8,   13,  17,  22,  26,  31,  35,   40,   44,   49,   53,   58,   62,   67,  71,  76,  80,  84,
           89,  93,  98,  102, 107, 111, 116, 121, 125,  130,  134,  139,  143,  148,  152,  157, 161, 166, 170, 175,
           180, 185, 189, 194, 199, 204, 208, 213, 217,  222,  227,  231,  236,  240,  245,  250, 254, 259, 263, 268,
           273, 277, 282, 286, 291, 295, 300, 305, 309,  314,  318,  323,  328,  332,  337,  342, 346, 351, 355, 360,
           364, 369, 374, 378, 383, 388, 392, 397, 401,  406,  410,  415,  419,  424,  429,  433, 438, 442, 447, 452,
           456, 461, 466, 470, 475, 479, 484, 488, 493,  498,  502,  507,  511,  516,  521,  525, 530, 534, 539, 543,
           548, 553, 557, 562, 567, 571, 576, 580, 586,  590,  595,  599,  603,  608,  612,  616, 621, 625, 629, 633,
           638, 642, 646, 651, 655, 659, 663, 667, 672,  676,  680,  684,  688,  692,  697,  701, 705, 709, 713, 717,
           720, 724, 728, 732, 736, 740, 743, 747, 751,  755,  758,  762,  766,  769,  773,  777, 780, 784, 787, 791,
           795, 798, 801, 805, 808, 811, 815, 818, 821,  825,  828,  831,  834,  838,  841,  844, 847, 850, 853, 856,
           859, 862, 865, 868, 871, 874, 877, 880, 883,  887,  889,  892,  895,  899,  902,  905, 908, 911, 914, 917,
           920, 923, 925, 928, 931, 935, 937, 940, 943,  946,  949,  952,  955,  958,  961,  964, 967, 969, 972, 975,
           978, 981, 984, 987, 990, 993, 996, 999, 1002, 1005, 1008, 1011, 1013, 1016, 1019, 1023}}}}}));

  ENSURE(
      parseAndCompare("MetaStartMarker,idx=6175,ts=205832,"
                      "xgy=-368,ygy=19,zgy=1885,xac=-536,yac=736,zac=-16932,xma=-1805,yma=-1095,zma=5234,"
                      "temp=27,"
                      "sen=0,iso=357,sht=0.02000000,sht_mx=0.00000000,"
                      "r=5966,g=4096,b=9863,"
                      "sen=1,iso=322,sht=0.02000000,sht_mx=0.00000000,"
                      "r=5967,g=4096,b=9861",
                      // IMU
                      {true, {205832000, -536, 736, -16932, -368, 19, 1885, -1805, -1095, 5234, 27}},
                      // exposure
                      {{0, {205832000, 357, 0.02f, 0.0f}}, {1, {205832000, 322, 0.02f, 0.0f}}},
                      // white balance
                      {{0, {205832000, 5966, 4096, 9863}}, {1, {205832000, 5967, 4096, 9861}}},
                      // tone curve
                      {}));

  ENSURE(parseAndCompare(
      "MetaStartMarker,idx=6426,ts=214199,"
      "xgy=-325,ygy=-1,zgy=1128,xac=-476,yac=608,zac=-16988,xma=-2501,yma=-1485,zma=6000,temp=29,"
      "sen=0,iso=359,sht=0.02000000,sht_mx=0.00000000,r=5933,g=4096,b=9950,"
      "sen=1,"
      "tcStart,0,4,8,13,17,22,26,31,35,40,44,49,53,58,62,67,71,76,81,85,90,94,99,103,108,112,117,122,126,131,135,140,"
      "144,149,153,158,163,167,172,176,181,186,191,196,201,205,210,214,219,224,229,233,238,242,247,252,256,261,266,270,"
      "275,280,284,289,293,298,303,307,312,317,321,326,330,335,340,345,349,354,358,363,368,372,377,382,386,391,395,400,"
      "405,409,414,418,423,428,433,437,442,446,451,456,460,465,470,474,479,484,488,493,497,502,507,512,516,521,525,530,"
      "535,539,544,548,553,558,562,567,572,576,581,586,591,596,600,604,609,613,618,622,626,631,635,639,643,648,652,656,"
      "660,665,669,673,677,682,686,690,694,698,702,706,710,714,718,722,726,730,734,738,741,745,749,753,756,760,764,768,"
      "771,775,779,782,786,790,793,797,800,803,807,810,813,816,820,823,826,830,833,836,839,842,845,848,852,854,857,860,"
      "863,866,869,872,875,878,881,884,887,890,893,896,899,902,905,908,911,914,917,919,922,925,928,931,934,937,940,942,"
      "945,948,951,954,957,960,962,965,968,971,974,977,979,982,985,988,991,994,996,999,1002,1005,1008,1011,1014,1017,"
      "1020,1023",
      // IMU
      {true, {214199000, -476, 608, -16988, -325, -1, 1128, -2501, -1485, 6000, 29}},
      // exposure
      {{0, {214199000, 359, 0.02f, 0.0f}}},
      // white balance
      {{0, {214199000, 5933, 4096, 9950}}},
      // tone curve
      {{1,
        {214199000,
         {{0,   4,   8,   13,  17,  22,  26,  31,  35,   40,   44,   49,   53,   58,   62,   67,  71,  76,  81,  85,
           90,  94,  99,  103, 108, 112, 117, 122, 126,  131,  135,  140,  144,  149,  153,  158, 163, 167, 172, 176,
           181, 186, 191, 196, 201, 205, 210, 214, 219,  224,  229,  233,  238,  242,  247,  252, 256, 261, 266, 270,
           275, 280, 284, 289, 293, 298, 303, 307, 312,  317,  321,  326,  330,  335,  340,  345, 349, 354, 358, 363,
           368, 372, 377, 382, 386, 391, 395, 400, 405,  409,  414,  418,  423,  428,  433,  437, 442, 446, 451, 456,
           460, 465, 470, 474, 479, 484, 488, 493, 497,  502,  507,  512,  516,  521,  525,  530, 535, 539, 544, 548,
           553, 558, 562, 567, 572, 576, 581, 586, 591,  596,  600,  604,  609,  613,  618,  622, 626, 631, 635, 639,
           643, 648, 652, 656, 660, 665, 669, 673, 677,  682,  686,  690,  694,  698,  702,  706, 710, 714, 718, 722,
           726, 730, 734, 738, 741, 745, 749, 753, 756,  760,  764,  768,  771,  775,  779,  782, 786, 790, 793, 797,
           800, 803, 807, 810, 813, 816, 820, 823, 826,  830,  833,  836,  839,  842,  845,  848, 852, 854, 857, 860,
           863, 866, 869, 872, 875, 878, 881, 884, 887,  890,  893,  896,  899,  902,  905,  908, 911, 914, 917, 919,
           922, 925, 928, 931, 934, 937, 940, 942, 945,  948,  951,  954,  957,  960,  962,  965, 968, 971, 974, 977,
           979, 982, 985, 988, 991, 994, 996, 999, 1002, 1005, 1008, 1011, 1014, 1017, 1020, 1023}}}}}));

  ENSURE(parseAndCompare(
      "MetaStartMarker,idx=7040,ts=234666,"
      "xgy=-430,ygy=-16,zgy=22,xac=-640,yac=104,zac=-16944,xma=-2255,yma=-1428,zma=5843,temp=29,"
      "sen=0,"
      "tcStart,0,4,8,13,18,23,27,32,37,42,46,51,55,60,65,70,74,79,84,89,94,98,103,107,112,117,122,127,131,136,141,146,"
      "150,155,160,165,170,175,179,184,189,194,199,204,209,214,219,224,229,234,239,243,248,253,258,263,268,273,278,282,"
      "287,292,297,302,307,312,317,321,326,331,336,341,346,351,356,361,365,370,375,380,385,390,395,400,404,409,414,419,"
      "424,429,433,438,443,448,453,457,462,467,472,477,482,487,492,497,502,507,511,516,521,526,531,536,540,545,550,555,"
      "560,565,570,574,579,584,589,594,599,604,609,614,619,624,628,633,637,642,647,651,656,660,665,669,673,678,682,686,"
      "690,695,699,703,708,712,716,721,725,729,733,737,741,745,749,753,757,761,764,768,772,776,779,783,786,790,794,798,"
      "801,805,808,812,815,819,822,825,828,831,835,838,841,844,847,850,853,856,859,862,864,867,870,872,876,878,881,883,"
      "886,888,891,894,896,899,901,904,907,909,911,914,917,920,922,924,927,930,932,934,937,939,942,944,947,949,952,954,"
      "957,959,961,964,966,969,971,973,976,978,981,983,985,988,991,993,995,998,1000,1003,1005,1007,1010,1013,1015,1017,"
      "1020,1023,"
      "sen=1,"
      "tcStart,0,4,8,13,18,23,27,32,36,41,46,51,55,60,64,69,74,78,83,88,93,97,102,106,111,116,121,125,130,135,140,144,"
      "149,154,158,163,168,173,177,182,187,192,197,202,207,212,216,221,226,231,236,241,246,251,255,260,265,270,275,279,"
      "284,289,294,298,303,308,313,318,323,328,332,337,342,347,352,357,361,366,371,376,380,385,390,395,400,405,409,414,"
      "419,424,428,433,438,443,448,452,457,462,467,472,477,482,486,491,496,501,505,510,515,520,525,530,534,539,544,549,"
      "554,558,563,568,573,578,583,587,592,597,602,607,612,617,621,626,630,635,639,644,648,653,657,661,666,670,675,679,"
      "683,687,692,696,700,705,709,713,717,721,726,730,734,738,742,746,749,753,757,761,764,768,772,775,779,783,787,790,"
      "794,797,801,804,808,811,815,818,821,824,828,831,834,837,840,843,846,849,852,855,858,861,864,866,870,872,875,878,"
      "880,883,886,888,891,894,896,899,902,904,907,909,912,915,918,920,923,926,928,931,933,936,938,941,944,946,949,951,"
      "954,956,959,961,964,966,969,971,974,976,979,982,984,987,989,992,994,997,999,1002,1005,1007,1010,1012,1014,1017,"
      "1020,1023",
      // IMU
      {true, {234666000, -640, 104, -16944, -430, -16, 22, -2255, -1428, 5843, 29}},
      // exposure
      {},
      // white balance
      {},
      // tone curve
      {{0,
        {234666000,
         {{0,   4,   8,   13,  18,  23,  27,   32,   37,   42,   46,   51,   55,   60,   65,   70,  74,  79,  84,  89,
           94,  98,  103, 107, 112, 117, 122,  127,  131,  136,  141,  146,  150,  155,  160,  165, 170, 175, 179, 184,
           189, 194, 199, 204, 209, 214, 219,  224,  229,  234,  239,  243,  248,  253,  258,  263, 268, 273, 278, 282,
           287, 292, 297, 302, 307, 312, 317,  321,  326,  331,  336,  341,  346,  351,  356,  361, 365, 370, 375, 380,
           385, 390, 395, 400, 404, 409, 414,  419,  424,  429,  433,  438,  443,  448,  453,  457, 462, 467, 472, 477,
           482, 487, 492, 497, 502, 507, 511,  516,  521,  526,  531,  536,  540,  545,  550,  555, 560, 565, 570, 574,
           579, 584, 589, 594, 599, 604, 609,  614,  619,  624,  628,  633,  637,  642,  647,  651, 656, 660, 665, 669,
           673, 678, 682, 686, 690, 695, 699,  703,  708,  712,  716,  721,  725,  729,  733,  737, 741, 745, 749, 753,
           757, 761, 764, 768, 772, 776, 779,  783,  786,  790,  794,  798,  801,  805,  808,  812, 815, 819, 822, 825,
           828, 831, 835, 838, 841, 844, 847,  850,  853,  856,  859,  862,  864,  867,  870,  872, 876, 878, 881, 883,
           886, 888, 891, 894, 896, 899, 901,  904,  907,  909,  911,  914,  917,  920,  922,  924, 927, 930, 932, 934,
           937, 939, 942, 944, 947, 949, 952,  954,  957,  959,  961,  964,  966,  969,  971,  973, 976, 978, 981, 983,
           985, 988, 991, 993, 995, 998, 1000, 1003, 1005, 1007, 1010, 1013, 1015, 1017, 1020, 1023}}}},
       {1,
        {234666000,
         {{0,   4,   8,   13,  18,  23,  27,  32,   36,   41,   46,   51,   55,   60,   64,   69,  74,  78,  83,  88,
           93,  97,  102, 106, 111, 116, 121, 125,  130,  135,  140,  144,  149,  154,  158,  163, 168, 173, 177, 182,
           187, 192, 197, 202, 207, 212, 216, 221,  226,  231,  236,  241,  246,  251,  255,  260, 265, 270, 275, 279,
           284, 289, 294, 298, 303, 308, 313, 318,  323,  328,  332,  337,  342,  347,  352,  357, 361, 366, 371, 376,
           380, 385, 390, 395, 400, 405, 409, 414,  419,  424,  428,  433,  438,  443,  448,  452, 457, 462, 467, 472,
           477, 482, 486, 491, 496, 501, 505, 510,  515,  520,  525,  530,  534,  539,  544,  549, 554, 558, 563, 568,
           573, 578, 583, 587, 592, 597, 602, 607,  612,  617,  621,  626,  630,  635,  639,  644, 648, 653, 657, 661,
           666, 670, 675, 679, 683, 687, 692, 696,  700,  705,  709,  713,  717,  721,  726,  730, 734, 738, 742, 746,
           749, 753, 757, 761, 764, 768, 772, 775,  779,  783,  787,  790,  794,  797,  801,  804, 808, 811, 815, 818,
           821, 824, 828, 831, 834, 837, 840, 843,  846,  849,  852,  855,  858,  861,  864,  866, 870, 872, 875, 878,
           880, 883, 886, 888, 891, 894, 896, 899,  902,  904,  907,  909,  912,  915,  918,  920, 923, 926, 928, 931,
           933, 936, 938, 941, 944, 946, 949, 951,  954,  956,  959,  961,  964,  966,  969,  971, 974, 976, 979, 982,
           984, 987, 989, 992, 994, 997, 999, 1002, 1005, 1007, 1010, 1012, 1014, 1017, 1020, 1023}}}}}));
}

void testNewToneCurve() {
  ENSURE(parseAndCompare(
      "MetaStartMarker,idx=5521,ts=184032,"
      "xgy=-344,ygy=-15,zgy=77,xac=-720,yac=132,zac=-16912,xma=-2616,yma=-1363,zma=6010,"
      "temp=30,"
      "sen=0,"
      "tcStart=0;4;8;13;17;22;26;31;35;40;45;49;54;58;63;67;72;76;81;86;90;95;99;104;108;113;118;122;127;131;136;140;"
      "145;150;154;159;164;168;173;177;182;187;192;197;202;206;211;215;220;225;230;234;239;244;248;253;258;262;267;272;"
      "276;281;286;290;295;300;304;309;314;319;323;328;332;337;342;347;351;356;360;365;370;374;379;384;389;393;398;402;"
      "407;412;416;421;426;430;435;439;444;449;454;458;463;468;472;477;482;487;491;496;500;505;510;515;519;524;528;533;"
      "538;542;547;552;556;561;566;570;575;580;585;589;594;599;603;608;612;617;621;626;630;634;639;643;647;651;656;660;"
      "664;668;673;677;681;686;690;694;698;702;706;710;714;718;722;726;730;734;738;742;745;749;753;756;760;764;768;771;"
      "775;779;782;786;790;793;797;800;803;807;810;814;817;820;823;826;830;833;836;839;842;845;848;851;855;857;860;863;"
      "866;869;872;875;878;881;884;887;890;892;895;898;901;904;907;910;913;916;919;921;924;927;930;933;936;938;941;944;"
      "947;949;952;955;958;961;963;966;969;972;975;978;980;983;986;989;991;994;997;1000;1003;1005;1008;1011;1014;1017;"
      "1020;1023,"
      "sen=1",
      // IMU
      {true, {184032000, -720, 132, -16912, -344, -15, 77, -2616, -1363, 6010, 30}},
      // exposure
      {},
      // white balance
      {},
      // tone curve
      {{0,
        {184032000,
         {{0,   4,   8,   13,  17,  22,  26,  31,   35,   40,   45,   49,   54,   58,   63,   67,  72,  76,  81,  86,
           90,  95,  99,  104, 108, 113, 118, 122,  127,  131,  136,  140,  145,  150,  154,  159, 164, 168, 173, 177,
           182, 187, 192, 197, 202, 206, 211, 215,  220,  225,  230,  234,  239,  244,  248,  253, 258, 262, 267, 272,
           276, 281, 286, 290, 295, 300, 304, 309,  314,  319,  323,  328,  332,  337,  342,  347, 351, 356, 360, 365,
           370, 374, 379, 384, 389, 393, 398, 402,  407,  412,  416,  421,  426,  430,  435,  439, 444, 449, 454, 458,
           463, 468, 472, 477, 482, 487, 491, 496,  500,  505,  510,  515,  519,  524,  528,  533, 538, 542, 547, 552,
           556, 561, 566, 570, 575, 580, 585, 589,  594,  599,  603,  608,  612,  617,  621,  626, 630, 634, 639, 643,
           647, 651, 656, 660, 664, 668, 673, 677,  681,  686,  690,  694,  698,  702,  706,  710, 714, 718, 722, 726,
           730, 734, 738, 742, 745, 749, 753, 756,  760,  764,  768,  771,  775,  779,  782,  786, 790, 793, 797, 800,
           803, 807, 810, 814, 817, 820, 823, 826,  830,  833,  836,  839,  842,  845,  848,  851, 855, 857, 860, 863,
           866, 869, 872, 875, 878, 881, 884, 887,  890,  892,  895,  898,  901,  904,  907,  910, 913, 916, 919, 921,
           924, 927, 930, 933, 936, 938, 941, 944,  947,  949,  952,  955,  958,  961,  963,  966, 969, 972, 975, 978,
           980, 983, 986, 989, 991, 994, 997, 1000, 1003, 1005, 1008, 1011, 1014, 1017, 1020, 1023}}}}}));

  ENSURE(parseAndCompare(
      "MetaStartMarker,idx=5993,ts=199766,"
      "xgy=-368,ygy=64,zgy=1565,xac=-592,yac=-88,zac=-17012,xma=-2190,yma=-1448,zma=5079,temp=29,"
      "sen=0,r=5880,g=4096,b=9949,"
      "sen=1,r=5881,g=4096,b=9948,"
      "tcStart=0;4;8;13;17;22;26;31;35;40;44;49;53;58;62;67;71;76;80;84;89;93;98;102;107;111;116;121;125;130;134;139;"
      "143;148;152;157;161;166;170;175;180;185;189;194;199;204;208;213;217;222;227;231;236;240;245;250;254;259;263;268;"
      "273;277;282;286;291;295;300;305;309;314;318;323;328;332;337;342;346;351;355;360;364;369;374;378;383;388;392;397;"
      "401;406;410;415;419;424;429;433;438;442;447;452;456;461;466;470;475;479;484;488;493;498;502;507;511;516;521;525;"
      "530;534;539;543;548;553;557;562;567;571;576;580;586;590;595;599;603;608;612;616;621;625;629;633;638;642;646;651;"
      "655;659;663;667;672;676;680;684;688;692;697;701;705;709;713;717;720;724;728;732;736;740;743;747;751;755;758;762;"
      "766;769;773;777;780;784;787;791;795;798;801;805;808;811;815;818;821;825;828;831;834;838;841;844;847;850;853;856;"
      "859;862;865;868;871;874;877;880;883;887;889;892;895;899;902;905;908;911;914;917;920;923;925;928;931;935;937;940;"
      "943;946;949;952;955;958;961;964;967;969;972;975;978;981;984;987;990;993;996;999;1002;1005;1008;1011;1013;1016;"
      "1019;1023",
      // IMU
      {true, {199766000, -592, -88, -17012, -368, 64, 1565, -2190, -1448, 5079, 29}},
      // exposure
      {},
      // white balance
      {{0, {199766000, 5880, 4096, 9949}}, {1, {199766000, 5881, 4096, 9948}}},
      // tone curve
      {{1,
        {199766000,
         {{0,   4,   8,   13,  17,  22,  26,  31,  35,   40,   44,   49,   53,   58,   62,   67,  71,  76,  80,  84,
           89,  93,  98,  102, 107, 111, 116, 121, 125,  130,  134,  139,  143,  148,  152,  157, 161, 166, 170, 175,
           180, 185, 189, 194, 199, 204, 208, 213, 217,  222,  227,  231,  236,  240,  245,  250, 254, 259, 263, 268,
           273, 277, 282, 286, 291, 295, 300, 305, 309,  314,  318,  323,  328,  332,  337,  342, 346, 351, 355, 360,
           364, 369, 374, 378, 383, 388, 392, 397, 401,  406,  410,  415,  419,  424,  429,  433, 438, 442, 447, 452,
           456, 461, 466, 470, 475, 479, 484, 488, 493,  498,  502,  507,  511,  516,  521,  525, 530, 534, 539, 543,
           548, 553, 557, 562, 567, 571, 576, 580, 586,  590,  595,  599,  603,  608,  612,  616, 621, 625, 629, 633,
           638, 642, 646, 651, 655, 659, 663, 667, 672,  676,  680,  684,  688,  692,  697,  701, 705, 709, 713, 717,
           720, 724, 728, 732, 736, 740, 743, 747, 751,  755,  758,  762,  766,  769,  773,  777, 780, 784, 787, 791,
           795, 798, 801, 805, 808, 811, 815, 818, 821,  825,  828,  831,  834,  838,  841,  844, 847, 850, 853, 856,
           859, 862, 865, 868, 871, 874, 877, 880, 883,  887,  889,  892,  895,  899,  902,  905, 908, 911, 914, 917,
           920, 923, 925, 928, 931, 935, 937, 940, 943,  946,  949,  952,  955,  958,  961,  964, 967, 969, 972, 975,
           978, 981, 984, 987, 990, 993, 996, 999, 1002, 1005, 1008, 1011, 1013, 1016, 1019, 1023}}}}}));

  ENSURE(
      parseAndCompare("MetaStartMarker,idx=6175,ts=205832,"
                      "xgy=-368,ygy=19,zgy=1885,xac=-536,yac=736,zac=-16932,xma=-1805,yma=-1095,zma=5234,"
                      "temp=27,"
                      "sen=0,iso=357,sht=0.02000000,sht_mx=0.00000000,"
                      "r=5966,g=4096,b=9863,"
                      "sen=1,iso=322,sht=0.02000000,sht_mx=0.00000000,"
                      "r=5967,g=4096,b=9861",
                      // IMU
                      {true, {205832000, -536, 736, -16932, -368, 19, 1885, -1805, -1095, 5234, 27}},
                      // exposure
                      {{0, {205832000, 357, 0.02f, 0.0f}}, {1, {205832000, 322, 0.02f, 0.0f}}},
                      // white balance
                      {{0, {205832000, 5966, 4096, 9863}}, {1, {205832000, 5967, 4096, 9861}}},
                      // tone curve
                      {}));

  ENSURE(parseAndCompare(
      "MetaStartMarker,idx=6426,ts=214199,"
      "xgy=-325,ygy=-1,zgy=1128,xac=-476,yac=608,zac=-16988,xma=-2501,yma=-1485,zma=6000,temp=29,"
      "sen=0,iso=359,sht=0.02000000,sht_mx=0.00000000,r=5933,g=4096,b=9950,"
      "sen=1,"
      "tcStart=0;4;8;13;17;22;26;31;35;40;44;49;53;58;62;67;71;76;81;85;90;94;99;103;108;112;117;122;126;131;135;140;"
      "144;149;153;158;163;167;172;176;181;186;191;196;201;205;210;214;219;224;229;233;238;242;247;252;256;261;266;270;"
      "275;280;284;289;293;298;303;307;312;317;321;326;330;335;340;345;349;354;358;363;368;372;377;382;386;391;395;400;"
      "405;409;414;418;423;428;433;437;442;446;451;456;460;465;470;474;479;484;488;493;497;502;507;512;516;521;525;530;"
      "535;539;544;548;553;558;562;567;572;576;581;586;591;596;600;604;609;613;618;622;626;631;635;639;643;648;652;656;"
      "660;665;669;673;677;682;686;690;694;698;702;706;710;714;718;722;726;730;734;738;741;745;749;753;756;760;764;768;"
      "771;775;779;782;786;790;793;797;800;803;807;810;813;816;820;823;826;830;833;836;839;842;845;848;852;854;857;860;"
      "863;866;869;872;875;878;881;884;887;890;893;896;899;902;905;908;911;914;917;919;922;925;928;931;934;937;940;942;"
      "945;948;951;954;957;960;962;965;968;971;974;977;979;982;985;988;991;994;996;999;1002;1005;1008;1011;1014;1017;"
      "1020;1023",
      // IMU
      {true, {214199000, -476, 608, -16988, -325, -1, 1128, -2501, -1485, 6000, 29}},
      // exposure
      {{0, {214199000, 359, 0.02f, 0.0f}}},
      // white balance
      {{0, {214199000, 5933, 4096, 9950}}},
      // tone curve
      {{1,
        {214199000,
         {{0,   4,   8,   13,  17,  22,  26,  31,  35,   40,   44,   49,   53,   58,   62,   67,  71,  76,  81,  85,
           90,  94,  99,  103, 108, 112, 117, 122, 126,  131,  135,  140,  144,  149,  153,  158, 163, 167, 172, 176,
           181, 186, 191, 196, 201, 205, 210, 214, 219,  224,  229,  233,  238,  242,  247,  252, 256, 261, 266, 270,
           275, 280, 284, 289, 293, 298, 303, 307, 312,  317,  321,  326,  330,  335,  340,  345, 349, 354, 358, 363,
           368, 372, 377, 382, 386, 391, 395, 400, 405,  409,  414,  418,  423,  428,  433,  437, 442, 446, 451, 456,
           460, 465, 470, 474, 479, 484, 488, 493, 497,  502,  507,  512,  516,  521,  525,  530, 535, 539, 544, 548,
           553, 558, 562, 567, 572, 576, 581, 586, 591,  596,  600,  604,  609,  613,  618,  622, 626, 631, 635, 639,
           643, 648, 652, 656, 660, 665, 669, 673, 677,  682,  686,  690,  694,  698,  702,  706, 710, 714, 718, 722,
           726, 730, 734, 738, 741, 745, 749, 753, 756,  760,  764,  768,  771,  775,  779,  782, 786, 790, 793, 797,
           800, 803, 807, 810, 813, 816, 820, 823, 826,  830,  833,  836,  839,  842,  845,  848, 852, 854, 857, 860,
           863, 866, 869, 872, 875, 878, 881, 884, 887,  890,  893,  896,  899,  902,  905,  908, 911, 914, 917, 919,
           922, 925, 928, 931, 934, 937, 940, 942, 945,  948,  951,  954,  957,  960,  962,  965, 968, 971, 974, 977,
           979, 982, 985, 988, 991, 994, 996, 999, 1002, 1005, 1008, 1011, 1014, 1017, 1020, 1023}}}}}));

  ENSURE(parseAndCompare(
      "MetaStartMarker,idx=7040,ts=234666,"
      "xgy=-430,ygy=-16,zgy=22,xac=-640,yac=104,zac=-16944,xma=-2255,yma=-1428,zma=5843,temp=29,"
      "sen=0,"
      "tcStart=0;4;8;13;18;23;27;32;37;42;46;51;55;60;65;70;74;79;84;89;94;98;103;107;112;117;122;127;131;136;141;146;"
      "150;155;160;165;170;175;179;184;189;194;199;204;209;214;219;224;229;234;239;243;248;253;258;263;268;273;278;282;"
      "287;292;297;302;307;312;317;321;326;331;336;341;346;351;356;361;365;370;375;380;385;390;395;400;404;409;414;419;"
      "424;429;433;438;443;448;453;457;462;467;472;477;482;487;492;497;502;507;511;516;521;526;531;536;540;545;550;555;"
      "560;565;570;574;579;584;589;594;599;604;609;614;619;624;628;633;637;642;647;651;656;660;665;669;673;678;682;686;"
      "690;695;699;703;708;712;716;721;725;729;733;737;741;745;749;753;757;761;764;768;772;776;779;783;786;790;794;798;"
      "801;805;808;812;815;819;822;825;828;831;835;838;841;844;847;850;853;856;859;862;864;867;870;872;876;878;881;883;"
      "886;888;891;894;896;899;901;904;907;909;911;914;917;920;922;924;927;930;932;934;937;939;942;944;947;949;952;954;"
      "957;959;961;964;966;969;971;973;976;978;981;983;985;988;991;993;995;998;1000;1003;1005;1007;1010;1013;1015;1017;"
      "1020;1023,"
      "sen=1,"
      "tcStart=0;4;8;13;18;23;27;32;36;41;46;51;55;60;64;69;74;78;83;88;93;97;102;106;111;116;121;125;130;135;140;144;"
      "149;154;158;163;168;173;177;182;187;192;197;202;207;212;216;221;226;231;236;241;246;251;255;260;265;270;275;279;"
      "284;289;294;298;303;308;313;318;323;328;332;337;342;347;352;357;361;366;371;376;380;385;390;395;400;405;409;414;"
      "419;424;428;433;438;443;448;452;457;462;467;472;477;482;486;491;496;501;505;510;515;520;525;530;534;539;544;549;"
      "554;558;563;568;573;578;583;587;592;597;602;607;612;617;621;626;630;635;639;644;648;653;657;661;666;670;675;679;"
      "683;687;692;696;700;705;709;713;717;721;726;730;734;738;742;746;749;753;757;761;764;768;772;775;779;783;787;790;"
      "794;797;801;804;808;811;815;818;821;824;828;831;834;837;840;843;846;849;852;855;858;861;864;866;870;872;875;878;"
      "880;883;886;888;891;894;896;899;902;904;907;909;912;915;918;920;923;926;928;931;933;936;938;941;944;946;949;951;"
      "954;956;959;961;964;966;969;971;974;976;979;982;984;987;989;992;994;997;999;1002;1005;1007;1010;1012;1014;1017;"
      "1020;1023",
      // IMU
      {true, {234666000, -640, 104, -16944, -430, -16, 22, -2255, -1428, 5843, 29}},
      // exposure
      {},
      // white balance
      {},
      // tone curve
      {{0,
        {234666000,
         {{0,   4,   8,   13,  18,  23,  27,   32,   37,   42,   46,   51,   55,   60,   65,   70,  74,  79,  84,  89,
           94,  98,  103, 107, 112, 117, 122,  127,  131,  136,  141,  146,  150,  155,  160,  165, 170, 175, 179, 184,
           189, 194, 199, 204, 209, 214, 219,  224,  229,  234,  239,  243,  248,  253,  258,  263, 268, 273, 278, 282,
           287, 292, 297, 302, 307, 312, 317,  321,  326,  331,  336,  341,  346,  351,  356,  361, 365, 370, 375, 380,
           385, 390, 395, 400, 404, 409, 414,  419,  424,  429,  433,  438,  443,  448,  453,  457, 462, 467, 472, 477,
           482, 487, 492, 497, 502, 507, 511,  516,  521,  526,  531,  536,  540,  545,  550,  555, 560, 565, 570, 574,
           579, 584, 589, 594, 599, 604, 609,  614,  619,  624,  628,  633,  637,  642,  647,  651, 656, 660, 665, 669,
           673, 678, 682, 686, 690, 695, 699,  703,  708,  712,  716,  721,  725,  729,  733,  737, 741, 745, 749, 753,
           757, 761, 764, 768, 772, 776, 779,  783,  786,  790,  794,  798,  801,  805,  808,  812, 815, 819, 822, 825,
           828, 831, 835, 838, 841, 844, 847,  850,  853,  856,  859,  862,  864,  867,  870,  872, 876, 878, 881, 883,
           886, 888, 891, 894, 896, 899, 901,  904,  907,  909,  911,  914,  917,  920,  922,  924, 927, 930, 932, 934,
           937, 939, 942, 944, 947, 949, 952,  954,  957,  959,  961,  964,  966,  969,  971,  973, 976, 978, 981, 983,
           985, 988, 991, 993, 995, 998, 1000, 1003, 1005, 1007, 1010, 1013, 1015, 1017, 1020, 1023}}}},
       {1,
        {234666000,
         {{0,   4,   8,   13,  18,  23,  27,  32,   36,   41,   46,   51,   55,   60,   64,   69,  74,  78,  83,  88,
           93,  97,  102, 106, 111, 116, 121, 125,  130,  135,  140,  144,  149,  154,  158,  163, 168, 173, 177, 182,
           187, 192, 197, 202, 207, 212, 216, 221,  226,  231,  236,  241,  246,  251,  255,  260, 265, 270, 275, 279,
           284, 289, 294, 298, 303, 308, 313, 318,  323,  328,  332,  337,  342,  347,  352,  357, 361, 366, 371, 376,
           380, 385, 390, 395, 400, 405, 409, 414,  419,  424,  428,  433,  438,  443,  448,  452, 457, 462, 467, 472,
           477, 482, 486, 491, 496, 501, 505, 510,  515,  520,  525,  530,  534,  539,  544,  549, 554, 558, 563, 568,
           573, 578, 583, 587, 592, 597, 602, 607,  612,  617,  621,  626,  630,  635,  639,  644, 648, 653, 657, 661,
           666, 670, 675, 679, 683, 687, 692, 696,  700,  705,  709,  713,  717,  721,  726,  730, 734, 738, 742, 746,
           749, 753, 757, 761, 764, 768, 772, 775,  779,  783,  787,  790,  794,  797,  801,  804, 808, 811, 815, 818,
           821, 824, 828, 831, 834, 837, 840, 843,  846,  849,  852,  855,  858,  861,  864,  866, 870, 872, 875, 878,
           880, 883, 886, 888, 891, 894, 896, 899,  902,  904,  907,  909,  912,  915,  918,  920, 923, 926, 928, 931,
           933, 936, 938, 941, 944, 946, 949, 951,  954,  956,  959,  961,  964,  966,  969,  971, 974, 976, 979, 982,
           984, 987, 989, 992, 994, 997, 999, 1002, 1005, 1007, 1010, 1012, 1014, 1017, 1020, 1023}}}}}));
}

}  // namespace Testing
}  // namespace VideoStitch

int main(int /*argc*/, char** /*argv*/) {
  VideoStitch::Testing::initTest();
  VideoStitch::Testing::testRecordedDataFw0177();
  VideoStitch::Testing::testNewToneCurve();
  VideoStitch::Testing::testSyntheticData();

  return 0;
}
