/**
 * SSL/TLS utilities implementation for wssshd
 *
 * Copyright (C) 2024 Stefy Lanza <stefy@nexlab.net> and SexHack.me
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include "ssl.h"

SSL_CTX *ssl_create_context(void) {
    const SSL_METHOD *method;
    SSL_CTX *ctx;

    // Initialize OpenSSL
    SSL_library_init();
    OpenSSL_add_all_algorithms();
    SSL_load_error_strings();

    method = TLS_server_method();
    ctx = SSL_CTX_new(method);
    if (!ctx) {
        perror("Unable to create SSL context");
        ERR_print_errors_fp(stderr);
        return NULL;
    }

    return ctx;
}

void ssl_cleanup(void) {
    EVP_cleanup();
}

int ssl_load_certificates(SSL_CTX *ctx, const char *cert_file, const char *key_file) {
    // Load certificate
    if (SSL_CTX_use_certificate_file(ctx, cert_file, SSL_FILETYPE_PEM) <= 0) {
        ERR_print_errors_fp(stderr);
        return -1;
    }

    // Load private key
    if (SSL_CTX_use_PrivateKey_file(ctx, key_file, SSL_FILETYPE_PEM) <= 0) {
        ERR_print_errors_fp(stderr);
        return -1;
    }

    // Verify private key
    if (!SSL_CTX_check_private_key(ctx)) {
        fprintf(stderr, "Private key does not match the certificate\n");
        return -1;
    }

    return 0;
}

int ssl_generate_self_signed_cert(const char *cert_file, const char *key_file) {
    // This is a simplified implementation
    // In a real implementation, you would use OpenSSL command line tools
    // or the OpenSSL library to generate certificates programmatically

    char command[1024];
    snprintf(command, sizeof(command),
             "openssl req -x509 -newkey rsa:4096 -keyout %s -out %s -days 36500 -nodes -subj \"/C=US/ST=State/L=City/O=Organization/CN=localhost\"",
             key_file, cert_file);

    int result = system(command);
    return result == 0 ? 0 : -1;
}