/*
 * WSSSH Library - Shared utilities
 *
 * Copyright (C) 2024 Stefy Lanza <stefy@nexlab.net> and SexHack.me
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */

#ifndef WSSH_LIB_H
#define WSSH_LIB_H

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>
#include <openssl/ssl.h>
#include <openssl/err.h>
#include <getopt.h>
#include <sys/wait.h>
#include <fcntl.h>
#include <pthread.h>
#include <sys/select.h>
#include <signal.h>
#include <ifaddrs.h>
#include <net/if.h>
#include <linux/rtnetlink.h>
#include <linux/if.h>
#include <linux/route.h>

#define BUFFER_SIZE 1048576
#define MAX_CHUNK_SIZE 4096
#define DEFAULT_PORT 22

// Global signal flag
extern volatile sig_atomic_t sigint_received;

// SSL mutex for thread-safe SSL operations
extern pthread_mutex_t ssl_mutex;

// Operating modes
typedef enum {
    MODE_INTERACTIVE = 0,  // Default: current functionality
    MODE_SILENT,           // Same as interactive but no output
    MODE_BRIDGE,           // JSON stdin/stdout bridge
    MODE_SCRIPT,           // JSON protocol for scripting
    MODE_PIPE              // ProxyCommand mode: stdin/stdout proxy
} wsssh_mode_t;

// Data encoding modes
typedef enum {
    ENCODING_HEX = 0,     // Default: hex encoding
    ENCODING_BASE64,      // Base64 encoding
    ENCODING_BINARY       // Binary (no encoding)
} wsssh_encoding_t;

// Config structures
typedef struct {
    char *local_port;
    char *tunnel_host;  // Local IP address to bind tunnel to (default: 127.0.0.1)
    char *wssshd_host;  // wssshd server hostname
    char *client_id;    // Client ID for the tunnel
    int wssshd_port;  // wssshd server port
    int debug;
    int interval;  // Reconnection interval in seconds
    int dev_tunnel;  // Development mode - don't launch SSH, just setup tunnel
    char *tunnel;  // Transport types for data channel (comma-separated or "any")
    char *tunnel_control;  // Transport types for control channel (comma-separated or "any")
    char *service;  // Service type (default: "ssh")
    wsssh_mode_t mode;  // Operating mode
    wsssh_encoding_t encoding;  // Data encoding mode
    int daemon;  // Daemon mode: lazy initialization
} wsssh_config_t;

typedef struct {
    char *local_port;
    int wssshd_port;  // wssshd server port
    int debug;
    int interval;  // Connection retry interval in seconds
    int dev_tunnel;  // Development mode - don't launch SCP, just setup tunnel
    char *tunnel;  // Transport types for data channel (comma-separated or "any")
    char *tunnel_control;  // Transport types for control channel (comma-separated or "any")
    char *service;  // Service type (default: "ssh")
} wsscp_config_t;

// tunnel_t is defined in tunnel.h

// Thread arguments are defined in tunnel.h

// Transport definitions
#define SUPPORTED_TRANSPORTS "websocket"
#define RELAY_TRANSPORTS "websocket"

// Transport weights (lower number = higher priority, 0 = highest priority)
#define WEBSOCKET_WEIGHT 100

// Function declarations
char *read_config_value(const char *key);
char *read_config_value_from_file(const char *key, const char *config_file);
char *expand_transport_list(const char *transport_spec, int for_relay);
char *select_best_transport(const char *transport_list);
int get_transport_weight(const char *transport);
void print_trans_flag(void);
void print_palestinian_flag(void);
int find_available_port(void);
void generate_request_id(char *request_id, size_t size);
char *autodetect_local_ip(void);

#endif // WSSH_LIB_H