/**
 * JSON parsing utilities for wssshd
 *
 * Copyright (C) 2024 Stefy Lanza <stefy@nexlab.net> and SexHack.me
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */

#ifndef JSON_H
#define JSON_H

#include <stdbool.h>

// Simple JSON value types
typedef enum {
    JSON_NULL,
    JSON_BOOL,
    JSON_NUMBER,
    JSON_STRING,
    JSON_OBJECT,
    JSON_ARRAY
} json_type_t;

// Simple JSON value structure
typedef struct json_value {
    json_type_t type;
    union {
        bool boolean;
        double number;
        char *string;
        struct {
            struct json_pair *pairs;
            size_t count;
        } object;
        struct {
            struct json_value *values;
            size_t count;
        } array;
    } value;
} json_value_t;

// Key-value pair for objects
typedef struct json_pair {
    char *key;
    json_value_t *value;
} json_pair_t;

// Parsing functions
json_value_t *json_parse(const char *json_str);
void json_free(json_value_t *value);

// Access functions
json_value_t *json_get_object_value(const json_value_t *object, const char *key);
const char *json_get_string(const json_value_t *value);
double json_get_number(const json_value_t *value);
bool json_get_bool(const json_value_t *value);

// Building functions
json_value_t *json_create_object(void);
json_value_t *json_create_string(const char *str);
json_value_t *json_create_number(double num);
json_value_t *json_create_bool(bool b);
json_value_t *json_create_null(void);

bool json_add_to_object(json_value_t *object, const char *key, json_value_t *value);

// Serialization
char *json_serialize(const json_value_t *value);

#endif /* JSON_H */