#!/bin/bash

# configure.sh - Configuration script for wssshd2
# Generates Makefile based on system detection
#
# Copyright (C) 2024 Stefy Lanza <stefy@nexlab.net> and SexHack.me
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <https://www.gnu.org/licenses/>.

set -e

echo "Configuring wssshd2..."

# Detect system information
OS=$(uname -s)
ARCH=$(uname -m)

echo "Detected OS: $OS"
echo "Detected architecture: $ARCH"

# Check for required tools
check_command() {
    if ! command -v "$1" &> /dev/null; then
        echo "Error: $1 is required but not found"
        exit 1
    fi
}

check_command gcc
check_command make
check_command pkg-config

# Check for required libraries
if ! pkg-config --exists libssl libcrypto uuid sqlite3; then
    echo "Error: Required libraries (libssl, libcrypto, uuid, sqlite3) not found"
    echo "Please install development packages:"
    echo "  Ubuntu/Debian: sudo apt-get install libssl-dev uuid-dev libsqlite3-dev"
    echo "  CentOS/RHEL: sudo yum install openssl-devel libuuid-devel sqlite-devel"
    echo "  macOS: brew install openssl ossp-uuid sqlite"
    exit 1
fi

# Generate Makefile
cat > Makefile << 'EOF'
# Makefile for wssshd2 - Generated by configure.sh
# Do not edit manually, run ./configure.sh instead

CC = gcc
CFLAGS = -Wall -Wextra -O2 -I. -pthread
LDFLAGS = -lssl -lcrypto -lm -luuid -lsqlite3 -ldl
PREFIX = /usr/local
BINDIR = $(PREFIX)/bin
MANDIR = $(PREFIX)/share/man
CONFIGDIR = /etc

# Source files
SRCS = main.c config.c tunnel.c terminal.c vnc.c rdp.c websocket.c websocket_protocol.c web.c assets.c ssl.c plugin.c web_proxy.c
OBJS = $(SRCS:.c=.o)

# Target
TARGET = wssshd

.PHONY: all clean install uninstall

all: $(TARGET)

$(TARGET): $(OBJS)
	$(CC) $(CFLAGS) -o $@ $^ $(LDFLAGS)

%.o: %.c
	$(CC) $(CFLAGS) -c $< -o $@

# Dependencies
main.o: main.c config.h websocket.h web.h plugin.h
config.o: config.c config.h
tunnel.o: tunnel.c tunnel.h websocket.h
terminal.o: terminal.c terminal.h config.h
vnc.o: vnc.c vnc.h config.h
rdp.o: rdp.c rdp.h config.h
websocket.o: websocket.c websocket.h websocket_protocol.h config.h plugin.h
websocket_protocol.o: websocket_protocol.c websocket_protocol.h
web.o: web.c web.h terminal.h vnc.h rdp.h assets.h websocket.h plugin.h html_pages/index_page.h html_pages/login_page.h html_pages/terminal_page.h html_pages/vnc_page.h html_pages/rdp_page.h html_pages/users_page.h
assets.o: assets.c assets.h
ssl.o: ssl.c ssl.h
plugin.o: plugin.c plugin.h
web_proxy.o: web_proxy.c web_proxy.h websocket.h websocket_protocol.h

# Asset embedding (run before compilation)
assets.o: image_data.h

image_data.h: embed_assets.sh
	./embed_assets.sh

# HTML page generation
html_pages/index_page.h html_pages/login_page.h html_pages/terminal_page.h html_pages/vnc_page.h html_pages/rdp_page.h: embed_assets.sh
	./embed_assets.sh

clean:
	rm -f $(OBJS) $(TARGET) image_data.h favicon_data.h

install: $(TARGET)
	install -d $(DESTDIR)$(BINDIR)
	install -m 755 $(TARGET) $(DESTDIR)$(BINDIR)/
	install -d $(DESTDIR)$(CONFIGDIR)
	[ -f $(DESTDIR)$(CONFIGDIR)/wssshd.conf ] || install -m 644 wssshd.conf.example $(DESTDIR)$(CONFIGDIR)/wssshd.conf
	install -d $(DESTDIR)$(MANDIR)/man8
	install -m 644 wssshd.8 $(DESTDIR)$(MANDIR)/man8/

uninstall:
	rm -f $(DESTDIR)$(BINDIR)/$(TARGET)
	rm -f $(DESTDIR)$(MANDIR)/man8/wssshd.8

# Development targets
debug: CFLAGS += -g -DDEBUG
debug: clean all

test: $(TARGET)
	@echo "Running basic functionality test..."
	./$(TARGET) --help || true

distclean: clean
	rm -f Makefile

# Help target
help:
	@echo "Available targets:"
	@echo "  all        - Build wssshd2 (default)"
	@echo "  clean      - Remove build artifacts"
	@echo "  install    - Install wssshd2 to system"
	@echo "  uninstall  - Remove wssshd2 from system"
	@echo "  debug      - Build with debug symbols"
	@echo "  test       - Run basic tests"
	@echo "  distclean  - Remove all generated files"
	@echo "  help       - Show this help"
EOF

echo "Makefile generated successfully"
echo "Run 'make' to build wssshd2"
echo "Run 'make install' to install system-wide"