#!/bin/bash

# configure.sh - Configuration script for wssshtools C implementation
# Generates Makefile for building WebSocket SSH tools C components
#
# Copyright (C) 2024 Stefy Lanza <stefy@nexlab.net> and SexHack.me
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <https://www.gnu.org/licenses/>.

echo "Configuring wssshtools..."

# Check for required tools
echo "Checking for required tools..."

if ! command -v gcc &> /dev/null; then
    echo "Error: gcc not found. Please install gcc."
    exit 1
fi

if ! command -v make &> /dev/null; then
    echo "Error: make not found. Please install make."
    exit 1
fi

if ! command -v pkg-config &> /dev/null; then
    echo "Error: pkg-config not found. Please install pkg-config."
    exit 1
fi

# Check for OpenSSL development libraries
if ! pkg-config --exists openssl; then
    echo "Error: OpenSSL development libraries not found."
    echo "Please install openssl-dev or libssl-dev package."
    exit 1
fi

# Check for zlib development libraries
if ! pkg-config --exists zlib; then
    echo "Error: zlib development libraries not found."
    echo "Please install zlib-dev or zlib1g-dev package."
    exit 1
fi

echo "All required tools found."

# Generate Makefile
cat > Makefile << 'EOF'
# Makefile for wssshtools

CC = gcc
CFLAGS = -Wall -Wextra -O2 -D_GNU_SOURCE $(shell pkg-config --cflags openssl) $(shell pkg-config --cflags zlib)
LDFLAGS = $(shell pkg-config --libs openssl) $(shell pkg-config --libs zlib)

# Source files
LIB_SRCS = libwsssht/wssshlib.c libwsssht/websocket.c libwsssht/wssh_ssl.c libwsssht/tunnel.c libwsssht/utils.c libwsssht/modes.c libwsssht/threads.c libwsssht/control_messages.c libwsssht/data_messages.c
LIB_OBJS = $(LIB_SRCS:.c=.o)
SRCS = wssshc.c wsssht.c wsssh.c wsscp.c
OBJS = $(SRCS:.c=.o)
TEST_SRCS = test_vnc_client.c test_rdp_client.c
TEST_OBJS = $(TEST_SRCS:.c=.o)
TEST_TARGETS = test_vnc_client test_rdp_client
TARGETS = wssshc wsssht wsssh wsscp $(TEST_TARGETS)

# Man pages
MANPAGES = man/wssshc.1 man/wsssht.1 man/wsssh.1 man/wsscp.1

# Default target
all: $(TARGETS)

# Individual targets
wssshc: wssshc.o libwsssht/wssshlib.o libwsssht/websocket.o libwsssht/wssh_ssl.o libwsssht/tunnel.o libwsssht/control_messages.o libwsssht/data_messages.o
	$(CC) $(CFLAGS) -o $@ $^ $(LDFLAGS)

wsssht: wsssht.o $(LIB_OBJS)
	$(CC) $(CFLAGS) -o $@ $^ $(LDFLAGS)

wsssh: wsssh.o
	$(CC) $(CFLAGS) -o $@ $^ $(LDFLAGS)

wsscp: wsscp.o
	$(CC) $(CFLAGS) -o $@ $^ $(LDFLAGS)

test_vnc_client: test_vnc_client.o
	$(CC) $(CFLAGS) -o $@ $^ $(LDFLAGS)

test_rdp_client: test_rdp_client.o
	$(CC) $(CFLAGS) -o $@ $^ $(LDFLAGS)

# Object files
%.o: %.c
	$(CC) $(CFLAGS) -c $< -o $@

# Clean
clean:
	rm -f $(OBJS) $(LIB_OBJS) $(TEST_OBJS) $(TARGETS) wsscp.o libwsssht/control_messages.o libwsssht/data_messages.o

# Install (optional)
install: all
	# Use /usr/bin for Debian packaging, /usr/local/bin for manual installs
	if [ -n "$(DESTDIR)" ] && echo "$(DESTDIR)" | grep -q "debian/"; then \
		install -d $(DESTDIR)/usr/bin; \
		install -m 755 $(TARGETS) $(DESTDIR)/usr/bin/; \
		install -d $(DESTDIR)/usr/share/man/man1; \
		install -m 644 $(MANPAGES) $(DESTDIR)/usr/share/man/man1/; \
	else \
		install -d $(DESTDIR)/usr/local/bin; \
		install -m 755 $(TARGETS) $(DESTDIR)/usr/local/bin/; \
		install -d $(DESTDIR)/usr/local/share/man/man1; \
		install -m 644 $(MANPAGES) $(DESTDIR)/usr/local/share/man/man1/; \
	fi

# Uninstall (optional)
uninstall:
	# Remove from both possible locations
	rm -f $(DESTDIR)/usr/local/bin/wssshc
	rm -f $(DESTDIR)/usr/local/bin/wsssht
	rm -f $(DESTDIR)/usr/local/bin/wsssh
	rm -f $(DESTDIR)/usr/local/bin/wsscp
	rm -f $(DESTDIR)/usr/local/share/man/man1/wssshc.1
	rm -f $(DESTDIR)/usr/local/share/man/man1/wsssht.1
	rm -f $(DESTDIR)/usr/local/share/man/man1/wsssh.1
	rm -f $(DESTDIR)/usr/local/share/man/man1/wsscp.1
	rm -f $(DESTDIR)/usr/bin/wssshc
	rm -f $(DESTDIR)/usr/bin/wsssht
	rm -f $(DESTDIR)/usr/bin/wsssh
	rm -f $(DESTDIR)/usr/bin/wsscp
	rm -f $(DESTDIR)/usr/share/man/man1/wssshc.1
	rm -f $(DESTDIR)/usr/share/man/man1/wsssht.1
	rm -f $(DESTDIR)/usr/share/man/man1/wsssh.1
	rm -f $(DESTDIR)/usr/share/man/man1/wsscp.1

.PHONY: all clean install uninstall
EOF

echo "Makefile generated successfully."

# Create configure stamp
touch configure.sh.stamp

echo "Configuration complete. Run 'make' to build the C tools."