#!/bin/bash

# wssshc_watcher.sh - Wrapper script for wssshc with automatic restart functionality
# Copyright (C) 2024 Stefy Lanza <stefy@nexlab.net> and SexHack.me

PID_FILE="/tmp/wssshc_watcher.pid"
WSSSHC_CMD="./wssshc"  # Adjust if wssshc is in a different location

print_usage() {
    echo "Usage: $0 {start|stop|restart|status} [wssshc_args...]"
    echo "  start wssshc_args  - Start the wssshc watcher with specified arguments"
    echo "  stop               - Stop the wssshc watcher"
    echo "  restart            - Restart the wssshc watcher"
    echo "  status             - Show the status of the wssshc watcher"
    echo ""
    echo "Example: $0 start --server-ip 192.168.1.100 --port 9898 --id myclient --password mysecret"
}

start() {
    if [ -f "$PID_FILE" ] && kill -0 $(cat "$PID_FILE") 2>/dev/null; then
        echo "wssshc_watcher is already running (PID: $(cat "$PID_FILE"))"
        return 1
    fi

    # Get wssshc arguments from command line
    shift  # Remove 'start' from arguments
    WSSSHC_ARGS="$*"

    if [ -z "$WSSSHC_ARGS" ]; then
        echo "Error: No wssshc arguments provided."
        echo "Usage: $0 start wssshc_args..."
        echo "Example: $0 start --server-ip 192.168.1.100 --port 9898 --id myclient --password mysecret"
        return 1
    fi

    echo "Starting wssshc_watcher with arguments: $WSSSHC_ARGS"
    nohup "$0" watch "$WSSSHC_ARGS" > /tmp/wssshc_watcher.log 2>&1 &
    WATCHER_PID=$!
    echo $WATCHER_PID > "$PID_FILE"
    echo "wssshc_watcher started with PID $WATCHER_PID"
    echo "Log file: /tmp/wssshc_watcher.log"
}

stop() {
    if [ -f "$PID_FILE" ]; then
        WATCHER_PID=$(cat "$PID_FILE")
        if kill -0 $WATCHER_PID 2>/dev/null; then
            echo "Stopping wssshc_watcher (PID: $WATCHER_PID)..."
            kill $WATCHER_PID
            # Wait for graceful shutdown
            for i in {1..10}; do
                if ! kill -0 $WATCHER_PID 2>/dev/null; then
                    break
                fi
                sleep 1
            done
            # Force kill if still running
            if kill -0 $WATCHER_PID 2>/dev/null; then
                kill -9 $WATCHER_PID 2>/dev/null
            fi
        fi
        rm -f "$PID_FILE"
        echo "wssshc_watcher stopped"
    else
        echo "wssshc_watcher is not running"
    fi
}

restart() {
    echo "Restarting wssshc_watcher..."
    stop
    sleep 2
    start
}

status() {
    if [ -f "$PID_FILE" ]; then
        WATCHER_PID=$(cat "$PID_FILE")
        if kill -0 $WATCHER_PID 2>/dev/null; then
            echo "wssshc_watcher is running (PID: $WATCHER_PID)"
            # Check if wssshc is also running
            WSSSHC_PID=$(pgrep -f "$WSSSHC_CMD")
            if [ -n "$WSSSHC_PID" ]; then
                echo "wssshc is running (PID: $WSSSHC_PID)"
            else
                echo "wssshc is not running (watcher may be restarting it)"
            fi
        else
            echo "wssshc_watcher PID file exists but process is not running"
            rm -f "$PID_FILE"
        fi
    else
        echo "wssshc_watcher is not running"
    fi
}

watch() {
    WSSSHC_ARGS="$1"
    echo "wssshc_watcher started at $(date)"
    echo "Monitoring wssshc with arguments: $WSSSHC_ARGS"

    while true; do
        echo "$(date): Starting wssshc..."
        $WSSSHC_CMD $WSSSHC_ARGS
        EXIT_CODE=$?
        echo "$(date): wssshc exited with code $EXIT_CODE"

        if [ $EXIT_CODE -eq 0 ]; then
            echo "$(date): wssshc exited normally, stopping watcher"
            break
        else
            echo "$(date): wssshc crashed or exited with error, restarting in 5 seconds..."
            sleep 5
        fi
    done

    echo "$(date): wssshc_watcher exiting"
}

# Main script logic
case "$1" in
    start)
        start "$@"
        ;;
    stop)
        stop
        ;;
    restart)
        restart
        ;;
    status)
        status
        ;;
    watch)
        shift
        watch "$*"
        ;;
    *)
        print_usage
        exit 1
        ;;
esac