/**
 * Web Proxy server implementation for wssshd
 *
 * Copyright (C) 2024 Stefy Lanza <stefy@nexlab.net> and SexHack.me
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <pthread.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <fcntl.h>
#include <errno.h>
#include <time.h>
#include <ctype.h>
#include <signal.h>
#include <netdb.h>
#include <openssl/ssl.h>
#include <openssl/err.h>
#include "web_proxy.h"
#include "websocket.h"
#include "websocket_protocol.h"
#include "ssl.h"

// Service types for web proxy
typedef enum {
    SERVICE_NONE = 0,
    SERVICE_WEB,
    SERVICE_HTTP,
    SERVICE_HTTPS
} service_type_t;

// Structure to hold client and service info
typedef struct {
    client_t *client;
    service_type_t service_type;
    char service_host[256];
    int service_port;
} client_service_info_t;

// Global state for shutdown
static int server_socket = -1;
static volatile int server_running = 0;

// Active proxy connections
#define MAX_PROXY_CONNECTIONS 100
typedef struct {
    int client_fd;           // Connection from web proxy client
    void *tunnel_ws;         // WebSocket tunnel to the client
    char client_id[256];
    char hostname[256];
    bool active;
    pthread_t thread;
} proxy_connection_t;

static proxy_connection_t proxy_connections[MAX_PROXY_CONNECTIONS];
static int proxy_connections_count = 0;
static pthread_mutex_t proxy_mutex = PTHREAD_MUTEX_INITIALIZER;

// Active tunnels to client services
#define MAX_ACTIVE_TUNNELS 50
typedef struct {
    char client_id[256];
    char hostname[256];
    service_type_t service_type;
    char service_host[256];
    int service_port;
    int tunnel_fd;           // Socket to client's service
    SSL *ssl;                // SSL connection for HTTPS
    time_t last_used;
    time_t created_at;
    bool active;
    pthread_mutex_t mutex;
} service_tunnel_t;

static service_tunnel_t active_tunnels[MAX_ACTIVE_TUNNELS];
static int active_tunnels_count = 0;
static pthread_mutex_t tunnels_mutex = PTHREAD_MUTEX_INITIALIZER;

// Thread arguments structure
typedef struct {
    const wssshd_config_t *config;
    wssshd_state_t *state;
} proxy_thread_args_t;

// Forward declarations
static void cleanup_expired_tunnels(void);

// Cleanup thread
static volatile int cleanup_running = 0;
static void *cleanup_thread(void *arg) {
    (void)arg; // Unused

    cleanup_running = 1;
    while (cleanup_running) {
        cleanup_expired_tunnels();
        sleep(60); // Cleanup every minute
    }

    return NULL;
}

// Connection handler arguments
typedef struct {
    int client_fd;
    wssshd_state_t *state;
    const wssshd_config_t *config;
} connection_handler_args_t;

// Forward declaration for proxy connection handler
static void *proxy_connection_handler(void *arg);


// Parse service string to extract host and port
static int parse_service_info(const char *service_str, char *host, size_t host_size, int *port) {
    // Service format: "web:host:port" or just "web"
    char *colon1 = strchr(service_str, ':');
    if (!colon1) {
        // No host/port specified, use defaults
        if (host) strncpy(host, "127.0.0.1", host_size);
        if (port) *port = 80; // Default HTTP port
        return 0;
    }

    char *colon2 = strchr(colon1 + 1, ':');
    if (!colon2) {
        // Only host specified
        size_t host_len = colon1 - service_str;
        if (host_len >= host_size) host_len = host_size - 1;
        if (host) {
            strncpy(host, service_str, host_len);
            host[host_len] = '\0';
        }
        if (port) *port = 80; // Default HTTP port
        return 0;
    }

    // Both host and port specified
    size_t host_len = colon2 - colon1 - 1;
    if (host_len >= host_size) host_len = host_size - 1;
    if (host) {
        strncpy(host, colon1 + 1, host_len);
        host[host_len] = '\0';
    }
    if (port) *port = atoi(colon2 + 1);

    return 0;
}

// Find client by hostname and determine which service to use
static client_service_info_t *find_client_service_by_hostname(const char *hostname, wssshd_state_t *state, const wssshd_config_t *config) {
    static client_service_info_t result;
    memset(&result, 0, sizeof(result));

    if (!hostname || !state) return NULL;

    pthread_mutex_lock(&state->client_mutex);

    for (size_t i = 0; i < state->clients_count; i++) {
        client_t *client = &state->clients[i];
        if (!client->active) continue;

        // Check hostname match
        int hostname_match = 0;
        if (strcasecmp(hostname, client->client_id) == 0) {
            hostname_match = 1;
        } else if (config && config->domain) {
            char expected_hostname[512];
            snprintf(expected_hostname, sizeof(expected_hostname), "%s.%s",
                     client->client_id, config->domain);
            if (strcasecmp(hostname, expected_hostname) == 0) {
                hostname_match = 1;
            }
        }

        if (!hostname_match) continue;

        // Parse services and find the best one (web > http > https)
        char *services_copy = strdup(client->services);
        if (!services_copy) continue;

        char *service_token = strtok(services_copy, ",");
        service_type_t best_service = SERVICE_NONE;
        char best_host[256] = "127.0.0.1";
        int best_port = 80;

        while (service_token) {
            // Trim whitespace
            while (*service_token == ' ' || *service_token == '\t') service_token++;
            char *end = service_token + strlen(service_token) - 1;
            while (end > service_token && (*end == ' ' || *end == '\t')) {
                *end = '\0';
                end--;
            }

            service_type_t current_type = SERVICE_NONE;
            if (strncmp(service_token, "web", 3) == 0) {
                current_type = SERVICE_WEB;
            } else if (strncmp(service_token, "http", 4) == 0) {
                current_type = SERVICE_HTTP;
            } else if (strncmp(service_token, "https", 5) == 0) {
                current_type = SERVICE_HTTPS;
            }

            if (current_type != SERVICE_NONE) {
                // Check if this is better than current best
                if (best_service == SERVICE_NONE ||
                    (current_type == SERVICE_WEB) ||
                    (current_type == SERVICE_HTTP && best_service != SERVICE_WEB) ||
                    (current_type == SERVICE_HTTPS && best_service != SERVICE_WEB && best_service != SERVICE_HTTP)) {

                    best_service = current_type;
                    parse_service_info(service_token, best_host, sizeof(best_host), &best_port);
                }
            }

            service_token = strtok(NULL, ",");
        }

        free(services_copy);

        if (best_service != SERVICE_NONE) {
            result.client = client;
            result.service_type = best_service;
            strncpy(result.service_host, best_host, sizeof(result.service_host));
            result.service_port = best_port;
            pthread_mutex_unlock(&state->client_mutex);
            return &result;
        }
    }

    pthread_mutex_unlock(&state->client_mutex);
    return NULL;
}

// Parse Host header from HTTP request
static int parse_host_header(const char *request, char *hostname, size_t hostname_size) {
    if (!request || !hostname) return -1;

    const char *host_header = strstr(request, "Host:");
    if (!host_header) return -1;

    host_header += 5; // Skip "Host:"
    while (*host_header == ' ' || *host_header == '\t') host_header++;

    // Extract hostname (up to port or end of line)
    const char *end = host_header;
    while (*end && *end != '\r' && *end != '\n' && *end != ' ') {
        end++;
    }

    size_t len = end - host_header;
    if (len >= hostname_size) len = hostname_size - 1;

    strncpy(hostname, host_header, len);
    hostname[len] = '\0';

    // Remove port if present
    char *colon = strchr(hostname, ':');
    if (colon) {
        *colon = '\0';
    }

    return 0;
}

// Find existing tunnel for client service
static service_tunnel_t *find_service_tunnel(const char *client_id, const char *hostname, service_type_t service_type, const char *service_host, int service_port) {
    pthread_mutex_lock(&tunnels_mutex);
    for (int i = 0; i < active_tunnels_count; i++) {
        service_tunnel_t *tunnel = &active_tunnels[i];
        if (tunnel->active &&
            strcmp(tunnel->client_id, client_id) == 0 &&
            strcmp(tunnel->hostname, hostname) == 0 &&
            tunnel->service_type == service_type &&
            strcmp(tunnel->service_host, service_host) == 0 &&
            tunnel->service_port == service_port) {
            tunnel->last_used = time(NULL);
            pthread_mutex_unlock(&tunnels_mutex);
            return tunnel;
        }
    }
    pthread_mutex_unlock(&tunnels_mutex);
    return NULL;
}

// Create new tunnel to client service
static service_tunnel_t *create_service_tunnel(const char *client_id, const char *hostname, service_type_t service_type, const char *service_host, int service_port) {
    pthread_mutex_lock(&tunnels_mutex);

    // Find free slot
    int slot = -1;
    for (int i = 0; i < MAX_ACTIVE_TUNNELS; i++) {
        if (!active_tunnels[i].active) {
            slot = i;
            break;
        }
    }

    if (slot == -1 && active_tunnels_count < MAX_ACTIVE_TUNNELS) {
        slot = active_tunnels_count++;
    }

    if (slot == -1) {
        pthread_mutex_unlock(&tunnels_mutex);
        return NULL;
    }

    service_tunnel_t *tunnel = &active_tunnels[slot];
    memset(tunnel, 0, sizeof(service_tunnel_t));

    strncpy(tunnel->client_id, client_id, sizeof(tunnel->client_id));
    strncpy(tunnel->hostname, hostname, sizeof(tunnel->hostname));
    tunnel->service_type = service_type;
    strncpy(tunnel->service_host, service_host, sizeof(tunnel->service_host));
    tunnel->service_port = service_port;
    tunnel->created_at = time(NULL);
    tunnel->last_used = time(NULL);
    tunnel->active = true;
    pthread_mutex_init(&tunnel->mutex, NULL);

    pthread_mutex_unlock(&tunnels_mutex);
    return tunnel;
}

// Establish connection to client service
static int establish_service_connection(service_tunnel_t *tunnel) {
    pthread_mutex_lock(&tunnel->mutex);

    if (tunnel->tunnel_fd >= 0) {
        // Already connected
        pthread_mutex_unlock(&tunnel->mutex);
        return 0;
    }

    // Create socket
    tunnel->tunnel_fd = socket(AF_INET, SOCK_STREAM, 0);
    if (tunnel->tunnel_fd < 0) {
        pthread_mutex_unlock(&tunnel->mutex);
        return -1;
    }

    // Set non-blocking
    fcntl(tunnel->tunnel_fd, F_SETFL, O_NONBLOCK);

    // Resolve host
    struct hostent *he = gethostbyname(tunnel->service_host);
    if (!he) {
        close(tunnel->tunnel_fd);
        tunnel->tunnel_fd = -1;
        pthread_mutex_unlock(&tunnel->mutex);
        return -1;
    }

    // Connect
    struct sockaddr_in addr;
    memset(&addr, 0, sizeof(addr));
    addr.sin_family = AF_INET;
    addr.sin_port = htons(tunnel->service_port);
    addr.sin_addr = *((struct in_addr *)he->h_addr);

    if (connect(tunnel->tunnel_fd, (struct sockaddr *)&addr, sizeof(addr)) < 0) {
        if (errno != EINPROGRESS) {
            close(tunnel->tunnel_fd);
            tunnel->tunnel_fd = -1;
            pthread_mutex_unlock(&tunnel->mutex);
            return -1;
        }
    }

    // For HTTPS, establish SSL connection
    if (tunnel->service_type == SERVICE_HTTPS) {
        SSL_CTX *ctx = create_ssl_context();
        if (!ctx) {
            close(tunnel->tunnel_fd);
            tunnel->tunnel_fd = -1;
            pthread_mutex_unlock(&tunnel->mutex);
            return -1;
        }

        tunnel->ssl = create_ssl_connection(ctx, tunnel->tunnel_fd, 0); // debug = 0
        SSL_CTX_free(ctx);

        if (!tunnel->ssl) {
            close(tunnel->tunnel_fd);
            tunnel->tunnel_fd = -1;
            pthread_mutex_unlock(&tunnel->mutex);
            return -1;
        }

        // Set to accept invalid certificates
        SSL_set_verify(tunnel->ssl, SSL_VERIFY_NONE, NULL);
    }

    pthread_mutex_unlock(&tunnel->mutex);
    return 0;
}

// Close tunnel
static void close_service_tunnel(service_tunnel_t *tunnel) {
    pthread_mutex_lock(&tunnel->mutex);

    if (tunnel->ssl) {
        SSL_free(tunnel->ssl);
        tunnel->ssl = NULL;
    }

    if (tunnel->tunnel_fd >= 0) {
        close(tunnel->tunnel_fd);
        tunnel->tunnel_fd = -1;
    }

    tunnel->active = false;
    pthread_mutex_unlock(&tunnel->mutex);
}

// Cleanup expired tunnels (called periodically)
static void cleanup_expired_tunnels(void) {
    time_t now = time(NULL);
    const int timeout_seconds = 300; // 5 minutes

    pthread_mutex_lock(&tunnels_mutex);
    for (int i = 0; i < active_tunnels_count; i++) {
        service_tunnel_t *tunnel = &active_tunnels[i];
        if (tunnel->active && (now - tunnel->last_used) > timeout_seconds) {
            printf("[WEB-PROXY] Closing expired tunnel for %s\n", tunnel->hostname);
            close_service_tunnel(tunnel);
        }
    }
    pthread_mutex_unlock(&tunnels_mutex);
}

// Send HTTP error response
static void send_http_error(int client_fd, int status_code, const char *status_text) {
    const char *body_format = "<html><body><h1>%d %s</h1></body></html>";
    char body[512];
    snprintf(body, sizeof(body), body_format, status_code, status_text);

    char response[1024];
    int len = snprintf(response, sizeof(response),
        "HTTP/1.1 %d %s\r\n"
        "Content-Type: text/html\r\n"
        "Content-Length: %zu\r\n"
        "Connection: close\r\n"
        "\r\n"
        "%s",
        status_code, status_text, strlen(body), body);

    send(client_fd, response, len, 0);
}

// Proxy data between client socket and tunnel WebSocket
static void *proxy_data_forward(void *arg) __attribute__((unused));
static void *proxy_data_forward(void *arg) {
    proxy_connection_t *conn = (proxy_connection_t *)arg;
    if (!conn) return NULL;

    char buffer[8192];
    fd_set read_fds;
    int max_fd = conn->client_fd;
    struct timeval tv;

    printf("[WEB-PROXY] Started data forwarding for %s\n", conn->hostname);

    while (conn->active && server_running) {
        FD_ZERO(&read_fds);
        FD_SET(conn->client_fd, &read_fds);

        tv.tv_sec = 1;
        tv.tv_usec = 0;

        int activity = select(max_fd + 1, &read_fds, NULL, NULL, &tv);
        if (activity < 0) {
            if (errno == EINTR) continue;
            break;
        }

        if (activity == 0) {
            continue; // Timeout, check again
        }

        // Read from client socket
        if (FD_ISSET(conn->client_fd, &read_fds)) {
            ssize_t bytes_read = recv(conn->client_fd, buffer, sizeof(buffer), 0);
            if (bytes_read <= 0) {
                // Connection closed or error
                break;
            }

            // Forward to tunnel WebSocket
            if (conn->tunnel_ws) {
                ws_send_binary_frame((ws_connection_t *)conn->tunnel_ws, buffer, bytes_read, false);
            }
        }
    }

    printf("[WEB-PROXY] Stopped data forwarding for %s\n", conn->hostname);
    return NULL;
}

// Handle incoming HTTP request and establish tunnel
static int handle_proxy_request(int client_fd, wssshd_state_t *state, const wssshd_config_t *config) {
    char request[8192];
    ssize_t bytes_read = recv(client_fd, request, sizeof(request) - 1, 0);
    if (bytes_read <= 0) {
        return -1;
    }
    request[bytes_read] = '\0';

    // Parse Host header
    char hostname[256];
    if (parse_host_header(request, hostname, sizeof(hostname)) != 0) {
        send_http_error(client_fd, 400, "Bad Request");
        return -1;
    }

    printf("[WEB-PROXY] Received request for hostname: %s\n", hostname);

    // Find client and service by hostname
    client_service_info_t *service_info = find_client_service_by_hostname(hostname, state, config);
    if (!service_info || !service_info->client) {
        printf("[WEB-PROXY] No client/service found for hostname: %s\n", hostname);
        send_http_error(client_fd, 404, "Not Found");
        return -1;
    }

    printf("[WEB-PROXY] Found client: %s, service: %s:%d\n",
           service_info->client->client_id, service_info->service_host, service_info->service_port);

    // Find or create tunnel
    service_tunnel_t *tunnel = find_service_tunnel(service_info->client->client_id, hostname,
                                                   service_info->service_type,
                                                   service_info->service_host,
                                                   service_info->service_port);

    if (!tunnel) {
        tunnel = create_service_tunnel(service_info->client->client_id, hostname,
                                       service_info->service_type,
                                       service_info->service_host,
                                       service_info->service_port);
        if (!tunnel) {
            printf("[WEB-PROXY] Failed to create tunnel for %s\n", hostname);
            send_http_error(client_fd, 503, "Service Unavailable");
            return -1;
        }
    }

    // Establish connection if not already connected
    if (establish_service_connection(tunnel) != 0) {
        printf("[WEB-PROXY] Failed to establish connection to %s:%d\n",
               service_info->service_host, service_info->service_port);
        send_http_error(client_fd, 502, "Bad Gateway");
        close_service_tunnel(tunnel);
        return -1;
    }

    // For WEB service, try HTTP first, then HTTPS if it fails
    if (service_info->service_type == SERVICE_WEB) {
        // Try HTTP first
        service_tunnel_t *http_tunnel = find_service_tunnel(service_info->client->client_id, hostname,
                                                            SERVICE_HTTP,
                                                            service_info->service_host,
                                                            80);
        if (!http_tunnel) {
            http_tunnel = create_service_tunnel(service_info->client->client_id, hostname,
                                                SERVICE_HTTP,
                                                service_info->service_host,
                                                80);
        }

        if (http_tunnel && establish_service_connection(http_tunnel) == 0) {
            tunnel = http_tunnel;
        } else {
            // Try HTTPS
            service_tunnel_t *https_tunnel = find_service_tunnel(service_info->client->client_id, hostname,
                                                                 SERVICE_HTTPS,
                                                                 service_info->service_host,
                                                                 443);
            if (!https_tunnel) {
                https_tunnel = create_service_tunnel(service_info->client->client_id, hostname,
                                                     SERVICE_HTTPS,
                                                     service_info->service_host,
                                                     443);
            }

            if (https_tunnel && establish_service_connection(https_tunnel) == 0) {
                tunnel = https_tunnel;
            } else {
                printf("[WEB-PROXY] Failed to establish HTTP or HTTPS connection for WEB service\n");
                send_http_error(client_fd, 502, "Bad Gateway");
                return -1;
            }
        }
    }

    // Forward the request to the service
    pthread_mutex_lock(&tunnel->mutex);

    ssize_t sent;
    if (tunnel->ssl) {
        sent = SSL_write(tunnel->ssl, request, bytes_read);
    } else {
        sent = send(tunnel->tunnel_fd, request, bytes_read, 0);
    }

    if (sent != bytes_read) {
        printf("[WEB-PROXY] Failed to forward request to service\n");
        pthread_mutex_unlock(&tunnel->mutex);
        send_http_error(client_fd, 502, "Bad Gateway");
        return -1;
    }

    // Read response from service and forward to client
    char buffer[8192];
    while (1) {
        ssize_t response_bytes;
        if (tunnel->ssl) {
            response_bytes = SSL_read(tunnel->ssl, buffer, sizeof(buffer));
        } else {
            response_bytes = recv(tunnel->tunnel_fd, buffer, sizeof(buffer), 0);
        }

        if (response_bytes <= 0) {
            break;
        }

        ssize_t client_sent = send(client_fd, buffer, response_bytes, 0);
        if (client_sent != response_bytes) {
            break;
        }
    }

    pthread_mutex_unlock(&tunnel->mutex);

    printf("[WEB-PROXY] Request handled for %s\n", hostname);
    return 0;
}

// Proxy connection handler thread
static void *proxy_connection_handler(void *arg) {
    connection_handler_args_t *args = (connection_handler_args_t *)arg;
    int client_fd = args->client_fd;
    wssshd_state_t *state = args->state;
    const wssshd_config_t *config = args->config;
    free(args);

    printf("[WEB-PROXY] New connection received\n");

    handle_proxy_request(client_fd, state, config);

    close(client_fd);
    printf("[WEB-PROXY] Connection closed\n");

    return NULL;
}

static void *http_proxy_thread(void *arg) {
    proxy_thread_args_t *args = (proxy_thread_args_t *)arg;
    const wssshd_config_t *config = args->config;
    wssshd_state_t *state = args->state;
    free(args);
    struct sockaddr_in server_addr, client_addr;
    socklen_t client_len = sizeof(client_addr);

    printf("[WEB-PROXY] Creating socket\n");
    server_socket = socket(AF_INET, SOCK_STREAM, 0);
    if (server_socket < 0) {
        perror("[WEB-PROXY] Failed to create server socket");
        return NULL;
    }
    printf("[WEB-PROXY] Socket created\n");

    // Set socket options
    int opt = 1;
    setsockopt(server_socket, SOL_SOCKET, SO_REUSEADDR, &opt, sizeof(opt));

    memset(&server_addr, 0, sizeof(server_addr));
    server_addr.sin_family = AF_INET;
    server_addr.sin_addr.s_addr = htonl(INADDR_LOOPBACK);  // Listen on localhost only
    server_addr.sin_port = htons(config->web_proxy_port);
    printf("[WEB-PROXY] Binding to 127.0.0.1:%d\n", config->web_proxy_port);

    if (bind(server_socket, (struct sockaddr *)&server_addr, sizeof(server_addr)) < 0) {
        perror("[WEB-PROXY] Failed to bind server socket");
        close(server_socket);
        return NULL;
    }
    printf("[WEB-PROXY] Bound successfully\n");

    if (listen(server_socket, 10) < 0) {
        perror("[WEB-PROXY] Failed to listen on server socket");
        close(server_socket);
        return NULL;
    }
    printf("[WEB-PROXY] Listening successfully\n");

    printf("[WEB-PROXY] Web proxy server starting on 127.0.0.1:%d\n", config->web_proxy_port);
    server_running = 1;

    // Ignore SIGPIPE to prevent crashes on broken connections
    signal(SIGPIPE, SIG_IGN);

    while (server_running) {
        int client_fd = accept(server_socket, (struct sockaddr *)&client_addr, &client_len);
        if (client_fd < 0) {
            if (server_running) perror("[WEB-PROXY] Failed to accept client connection");
            continue;
        }

        // Create a new thread to handle the connection
        connection_handler_args_t *handler_args = malloc(sizeof(connection_handler_args_t));
        if (!handler_args) {
            close(client_fd);
            continue;
        }
        handler_args->client_fd = client_fd;
        handler_args->state = state;
        handler_args->config = config;

        pthread_t thread;
        if (pthread_create(&thread, NULL, proxy_connection_handler, handler_args) != 0) {
            perror("[WEB-PROXY] Failed to create connection handler thread");
            free(handler_args);
            close(client_fd);
            continue;
        }

        pthread_detach(thread);
    }

    close(server_socket);
    server_socket = -1;
    printf("[WEB-PROXY] Web proxy server stopped\n");
    return NULL;
}

int web_proxy_start_server(const wssshd_config_t *config, wssshd_state_t *state) {
    if (!config->web_proxy_enabled || config->web_proxy_port == 0) {
        return 0; // Web proxy not enabled
    }

    // Start cleanup thread
    pthread_t cleanup_tid;
    if (pthread_create(&cleanup_tid, NULL, cleanup_thread, NULL) != 0) {
        perror("[WEB-PROXY] Failed to create cleanup thread");
        return -1;
    }
    pthread_detach(cleanup_tid);

    // Allocate thread arguments
    proxy_thread_args_t *args = malloc(sizeof(proxy_thread_args_t));
    if (!args) {
        perror("[WEB-PROXY] Failed to allocate thread arguments");
        return -1;
    }
    args->config = config;
    args->state = state;

    // Start HTTP proxy server thread
    pthread_t thread;
    if (pthread_create(&thread, NULL, http_proxy_thread, args) != 0) {
        perror("[WEB-PROXY] Failed to create HTTP proxy server thread");
        free(args);
        return -1;
    }

    pthread_detach(thread);
    return 0;
}

void web_proxy_stop_server(void) {
    server_running = 0;
    cleanup_running = 0;

    // Close server socket to unblock accept()
    if (server_socket >= 0) {
        close(server_socket);
        server_socket = -1;
    }

    // Close all active proxy connections
    pthread_mutex_lock(&proxy_mutex);
    for (int i = 0; i < proxy_connections_count; i++) {
        if (proxy_connections[i].active) {
            proxy_connections[i].active = false;
            if (proxy_connections[i].client_fd >= 0) {
                close(proxy_connections[i].client_fd);
            }
        }
    }
    proxy_connections_count = 0;
    pthread_mutex_unlock(&proxy_mutex);

    // Close all active tunnels
    pthread_mutex_lock(&tunnels_mutex);
    for (int i = 0; i < active_tunnels_count; i++) {
        if (active_tunnels[i].active) {
            close_service_tunnel(&active_tunnels[i]);
        }
    }
    active_tunnels_count = 0;
    pthread_mutex_unlock(&tunnels_mutex);

    printf("[WEB-PROXY] Web proxy server stopping\n");
}
