/*
 * WSSSH Library - Data Channel Messages
 *
 * Copyright (C) 2024 Stefy Lanza <stefy@nexlab.net> and SexHack.me
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */

#ifndef DATA_MESSAGES_H
#define DATA_MESSAGES_H

#include <openssl/ssl.h>
#include "wssshlib.h"

// Retransmission buffer entry
typedef struct {
    uint32_t frame_id;
    char *message;
    size_t message_len;
    time_t timestamp;
    int retries;
} retransmission_entry_t;

// Retransmission buffer
#define RETRANSMISSION_BUFFER_SIZE 10
#define MAX_FRAME_SIZE 1048576  // 1MB max frame size
#define RETRANSMISSION_TIMEOUT 30  // 30 seconds timeout
#define MAX_RETRIES 3

typedef struct {
    retransmission_entry_t entries[RETRANSMISSION_BUFFER_SIZE];
    int count;
    uint32_t next_frame_id;
    pthread_mutex_t mutex;
} retransmission_buffer_t;

// Function declarations for data channel messages
int send_tunnel_data_message(SSL *ssl, const char *request_id, const char *data_hex, int debug);
int send_tunnel_data_binary_message(SSL *ssl, const char *request_id, const unsigned char *data, size_t data_len, int debug);
int send_tunnel_data_reliable_message(SSL *ssl, const char *request_id, const unsigned char *data, size_t data_len, retransmission_buffer_t *buffer, wsssh_encoding_t encoding, int debug);
int send_tunnel_response_message(SSL *ssl, const char *request_id, const char *data_hex, int debug);
int send_tunnel_ack_message(SSL *ssl, const char *request_id, uint32_t frame_id, int debug);
int send_tunnel_ko_message(SSL *ssl, const char *request_id, uint32_t frame_id, int debug);

// Retransmission buffer functions
retransmission_buffer_t *retransmission_buffer_init(void);
void retransmission_buffer_free(retransmission_buffer_t *buffer);
int retransmission_buffer_add(retransmission_buffer_t *buffer, uint32_t frame_id, const char *message, size_t message_len);
void retransmission_buffer_ack(retransmission_buffer_t *buffer, uint32_t frame_id);
void retransmission_buffer_ko(retransmission_buffer_t *buffer, uint32_t frame_id);
void retransmission_buffer_gc(retransmission_buffer_t *buffer);
uint32_t retransmission_buffer_get_next_frame_id(retransmission_buffer_t *buffer);

// Checksum functions
uint32_t crc32_checksum(const unsigned char *data, size_t len);
int send_tunnel_data_binary_message(SSL *ssl, const char *request_id, const unsigned char *data, size_t data_len, int debug);

#endif // DATA_MESSAGES_H