/**
 * Web Proxy server implementation for wssshd
 *
 * Copyright (C) 2024 Stefy Lanza <stefy@nexlab.net> and SexHack.me
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <pthread.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <fcntl.h>
#include <errno.h>
#include <time.h>
#include <ctype.h>
#include <signal.h>
#include "web_proxy.h"
#include "websocket.h"
#include "websocket_protocol.h"

// Global state for shutdown
static int server_socket = -1;
static volatile int server_running = 0;

// Active proxy connections
#define MAX_PROXY_CONNECTIONS 100
typedef struct {
    int client_fd;           // Connection from web proxy client
    void *tunnel_ws;         // WebSocket tunnel to the client
    char client_id[256];
    char hostname[256];
    bool active;
    pthread_t thread;
} proxy_connection_t;

static proxy_connection_t proxy_connections[MAX_PROXY_CONNECTIONS];
static int proxy_connections_count = 0;
static pthread_mutex_t proxy_mutex = PTHREAD_MUTEX_INITIALIZER;

// Thread arguments structure
typedef struct {
    const wssshd_config_t *config;
    wssshd_state_t *state;
} proxy_thread_args_t;

// Connection handler arguments
typedef struct {
    int client_fd;
    wssshd_state_t *state;
    const wssshd_config_t *config;
} connection_handler_args_t;

// Forward declaration for proxy connection handler
static void *proxy_connection_handler(void *arg);

// Find client by hostname (matching subdomain or client_id)
static client_t *find_client_by_hostname(const char *hostname, wssshd_state_t *state, const wssshd_config_t *config) {
    if (!hostname || !state) return NULL;

    pthread_mutex_lock(&state->client_mutex);

    for (size_t i = 0; i < state->clients_count; i++) {
        client_t *client = &state->clients[i];
        if (!client->active) continue;

        // Check if client has web service
        if (strstr(client->services, "web") == NULL &&
            strstr(client->services, "http") == NULL) {
            continue;
        }

        // Try exact match with client_id
        if (strcasecmp(hostname, client->client_id) == 0) {
            pthread_mutex_unlock(&state->client_mutex);
            return client;
        }

        // Try subdomain match: hostname.client_id.domain
        if (config && config->domain) {
            char expected_hostname[512];
            snprintf(expected_hostname, sizeof(expected_hostname), "%s.%s",
                     client->client_id, config->domain);
            if (strcasecmp(hostname, expected_hostname) == 0) {
                pthread_mutex_unlock(&state->client_mutex);
                return client;
            }
        }
    }

    pthread_mutex_unlock(&state->client_mutex);
    return NULL;
}

// Parse Host header from HTTP request
static int parse_host_header(const char *request, char *hostname, size_t hostname_size) {
    if (!request || !hostname) return -1;

    const char *host_header = strstr(request, "Host:");
    if (!host_header) return -1;

    host_header += 5; // Skip "Host:"
    while (*host_header == ' ' || *host_header == '\t') host_header++;

    // Extract hostname (up to port or end of line)
    const char *end = host_header;
    while (*end && *end != '\r' && *end != '\n' && *end != ' ') {
        end++;
    }

    size_t len = end - host_header;
    if (len >= hostname_size) len = hostname_size - 1;

    strncpy(hostname, host_header, len);
    hostname[len] = '\0';

    // Remove port if present
    char *colon = strchr(hostname, ':');
    if (colon) {
        *colon = '\0';
    }

    return 0;
}

// Send HTTP error response
static void send_http_error(int client_fd, int status_code, const char *status_text) {
    const char *body_format = "<html><body><h1>%d %s</h1></body></html>";
    char body[512];
    snprintf(body, sizeof(body), body_format, status_code, status_text);

    char response[1024];
    int len = snprintf(response, sizeof(response),
        "HTTP/1.1 %d %s\r\n"
        "Content-Type: text/html\r\n"
        "Content-Length: %zu\r\n"
        "Connection: close\r\n"
        "\r\n"
        "%s",
        status_code, status_text, strlen(body), body);

    send(client_fd, response, len, 0);
}

// Proxy data between client socket and tunnel WebSocket
static void *proxy_data_forward(void *arg) __attribute__((unused));
static void *proxy_data_forward(void *arg) {
    proxy_connection_t *conn = (proxy_connection_t *)arg;
    if (!conn) return NULL;

    char buffer[8192];
    fd_set read_fds;
    int max_fd = conn->client_fd;
    struct timeval tv;

    printf("[WEB-PROXY] Started data forwarding for %s\n", conn->hostname);

    while (conn->active && server_running) {
        FD_ZERO(&read_fds);
        FD_SET(conn->client_fd, &read_fds);

        tv.tv_sec = 1;
        tv.tv_usec = 0;

        int activity = select(max_fd + 1, &read_fds, NULL, NULL, &tv);
        if (activity < 0) {
            if (errno == EINTR) continue;
            break;
        }

        if (activity == 0) {
            continue; // Timeout, check again
        }

        // Read from client socket
        if (FD_ISSET(conn->client_fd, &read_fds)) {
            ssize_t bytes_read = recv(conn->client_fd, buffer, sizeof(buffer), 0);
            if (bytes_read <= 0) {
                // Connection closed or error
                break;
            }

            // Forward to tunnel WebSocket
            if (conn->tunnel_ws) {
                ws_send_binary_frame((ws_connection_t *)conn->tunnel_ws, buffer, bytes_read, false);
            }
        }
    }

    printf("[WEB-PROXY] Stopped data forwarding for %s\n", conn->hostname);
    return NULL;
}

// Handle incoming HTTP request and establish tunnel
static int handle_proxy_request(int client_fd, wssshd_state_t *state, const wssshd_config_t *config) {
    char request[8192];
    ssize_t bytes_read = recv(client_fd, request, sizeof(request) - 1, 0);
    if (bytes_read <= 0) {
        return -1;
    }
    request[bytes_read] = '\0';

    // Parse Host header
    char hostname[256];
    if (parse_host_header(request, hostname, sizeof(hostname)) != 0) {
        send_http_error(client_fd, 400, "Bad Request");
        return -1;
    }

    printf("[WEB-PROXY] Received request for hostname: %s\n", hostname);

    // Find client by hostname
    client_t *client = find_client_by_hostname(hostname, state, config);
    if (!client) {
        printf("[WEB-PROXY] No client found for hostname: %s\n", hostname);
        send_http_error(client_fd, 404, "Not Found");
        return -1;
    }

    printf("[WEB-PROXY] Found client: %s\n", client->client_id);

    // TODO: Establish tunnel to the client's web service
    // For now, we'll return a simple response indicating the proxy is working
    const char *response =
        "HTTP/1.1 200 OK\r\n"
        "Content-Type: text/plain\r\n"
        "Content-Length: 50\r\n"
        "Connection: close\r\n"
        "\r\n"
        "Web proxy tunnel established for client: ";

    char response_body[512];
    snprintf(response_body, sizeof(response_body), "%s%s", response, client->client_id);

    send(client_fd, response_body, strlen(response_body), 0);

    // Close the connection for now (full tunnel implementation requires more work)
    return 0;
}

// Proxy connection handler thread
static void *proxy_connection_handler(void *arg) {
    connection_handler_args_t *args = (connection_handler_args_t *)arg;
    int client_fd = args->client_fd;
    wssshd_state_t *state = args->state;
    const wssshd_config_t *config = args->config;
    free(args);

    printf("[WEB-PROXY] New connection received\n");

    handle_proxy_request(client_fd, state, config);

    close(client_fd);
    printf("[WEB-PROXY] Connection closed\n");

    return NULL;
}

static void *http_proxy_thread(void *arg) {
    proxy_thread_args_t *args = (proxy_thread_args_t *)arg;
    const wssshd_config_t *config = args->config;
    wssshd_state_t *state = args->state;
    free(args);
    struct sockaddr_in server_addr, client_addr;
    socklen_t client_len = sizeof(client_addr);

    printf("[WEB-PROXY] Creating socket\n");
    server_socket = socket(AF_INET, SOCK_STREAM, 0);
    if (server_socket < 0) {
        perror("[WEB-PROXY] Failed to create server socket");
        return NULL;
    }
    printf("[WEB-PROXY] Socket created\n");

    // Set socket options
    int opt = 1;
    setsockopt(server_socket, SOL_SOCKET, SO_REUSEADDR, &opt, sizeof(opt));

    memset(&server_addr, 0, sizeof(server_addr));
    server_addr.sin_family = AF_INET;
    server_addr.sin_addr.s_addr = htonl(INADDR_LOOPBACK);  // Listen on localhost only
    server_addr.sin_port = htons(config->web_proxy_port);
    printf("[WEB-PROXY] Binding to 127.0.0.1:%d\n", config->web_proxy_port);

    if (bind(server_socket, (struct sockaddr *)&server_addr, sizeof(server_addr)) < 0) {
        perror("[WEB-PROXY] Failed to bind server socket");
        close(server_socket);
        return NULL;
    }
    printf("[WEB-PROXY] Bound successfully\n");

    if (listen(server_socket, 10) < 0) {
        perror("[WEB-PROXY] Failed to listen on server socket");
        close(server_socket);
        return NULL;
    }
    printf("[WEB-PROXY] Listening successfully\n");

    printf("[WEB-PROXY] Web proxy server starting on 127.0.0.1:%d\n", config->web_proxy_port);
    server_running = 1;

    // Ignore SIGPIPE to prevent crashes on broken connections
    signal(SIGPIPE, SIG_IGN);

    while (server_running) {
        int client_fd = accept(server_socket, (struct sockaddr *)&client_addr, &client_len);
        if (client_fd < 0) {
            if (server_running) perror("[WEB-PROXY] Failed to accept client connection");
            continue;
        }

        // Create a new thread to handle the connection
        connection_handler_args_t *handler_args = malloc(sizeof(connection_handler_args_t));
        if (!handler_args) {
            close(client_fd);
            continue;
        }
        handler_args->client_fd = client_fd;
        handler_args->state = state;
        handler_args->config = config;

        pthread_t thread;
        if (pthread_create(&thread, NULL, proxy_connection_handler, handler_args) != 0) {
            perror("[WEB-PROXY] Failed to create connection handler thread");
            free(handler_args);
            close(client_fd);
            continue;
        }

        pthread_detach(thread);
    }

    close(server_socket);
    server_socket = -1;
    printf("[WEB-PROXY] Web proxy server stopped\n");
    return NULL;
}

int web_proxy_start_server(const wssshd_config_t *config, wssshd_state_t *state) {
    if (!config->web_proxy_enabled || config->web_proxy_port == 0) {
        return 0; // Web proxy not enabled
    }

    // Allocate thread arguments
    proxy_thread_args_t *args = malloc(sizeof(proxy_thread_args_t));
    if (!args) {
        perror("[WEB-PROXY] Failed to allocate thread arguments");
        return -1;
    }
    args->config = config;
    args->state = state;

    // Start HTTP proxy server thread
    pthread_t thread;
    if (pthread_create(&thread, NULL, http_proxy_thread, args) != 0) {
        perror("[WEB-PROXY] Failed to create HTTP proxy server thread");
        free(args);
        return -1;
    }

    pthread_detach(thread);
    return 0;
}

void web_proxy_stop_server(void) {
    server_running = 0;

    // Close server socket to unblock accept()
    if (server_socket >= 0) {
        close(server_socket);
        server_socket = -1;
    }

    // Close all active proxy connections
    pthread_mutex_lock(&proxy_mutex);
    for (int i = 0; i < proxy_connections_count; i++) {
        if (proxy_connections[i].active) {
            proxy_connections[i].active = false;
            if (proxy_connections[i].client_fd >= 0) {
                close(proxy_connections[i].client_fd);
            }
        }
    }
    proxy_connections_count = 0;
    pthread_mutex_unlock(&proxy_mutex);

    printf("[WEB-PROXY] Web proxy server stopping\n");
}
