/**
 * WebSocket handling for wssshd
 *
 * Copyright (C) 2024 Stefy Lanza <stefy@nexlab.net> and SexHack.me
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */

#ifndef WEBSOCKET_H
#define WEBSOCKET_H

#include <stdbool.h>
#include <time.h>
#include <pthread.h>
#include "config.h"
#include "tunnel.h"
#include "terminal.h"
#include "websocket_protocol.h"

// Client information
typedef struct {
    char client_id[256];
    void *websocket;  // WebSocket connection handle
    time_t last_seen;
    bool active;
    char tunnel[16];
    char tunnel_control[16];
    char wssshd_private_ip[64];
    char services[256];  // Comma-separated list of services (e.g., "ssh,rdp,vnc")
    char ip[INET_ADDRSTRLEN];
} client_t;

// Terminal session (defined in terminal.h)

// Global state
typedef struct {
    client_t *clients;
    size_t clients_count;
    size_t clients_capacity;

    tunnel_t **tunnels;
    size_t tunnels_count;
    size_t tunnels_capacity;

    terminal_session_t *terminals;
    size_t terminals_count;
    size_t terminals_capacity;

    bool debug;
    const char *server_password;
    time_t start_time;

    pthread_mutex_t client_mutex;
    pthread_mutex_t tunnel_mutex;
    pthread_mutex_t terminal_mutex;
} wssshd_state_t;

// Function declarations
wssshd_state_t *websocket_init_state(bool debug, const char *server_password);
void websocket_free_state(wssshd_state_t *state);

// Client management
client_t *websocket_find_client(wssshd_state_t *state, const char *client_id);
client_t *websocket_add_client(wssshd_state_t *state, const char *client_id, void *websocket, const char *services);
void websocket_remove_client(wssshd_state_t *state, const char *client_id);
void websocket_update_client_activity(wssshd_state_t *state, const char *client_id);

// Tunnel management
tunnel_t *websocket_find_tunnel(wssshd_state_t *state, const char *request_id);
tunnel_t *websocket_add_tunnel(wssshd_state_t *state, const char *request_id, const char *client_id);
void websocket_remove_tunnel(wssshd_state_t *state, const char *request_id);

// Terminal management
terminal_session_t *websocket_find_terminal(wssshd_state_t *state, const char *request_id);
terminal_session_t *websocket_add_terminal(wssshd_state_t *state, const char *request_id, const char *client_id, const char *username, pid_t proc_pid, int master_fd);
void websocket_remove_terminal(wssshd_state_t *state, const char *request_id);

// Message handling
int websocket_handle_message(wssshd_state_t *state, ws_connection_t *conn, const char *message, size_t message_len);

// Cleanup functions
void websocket_cleanup_expired_clients(wssshd_state_t *state);
void websocket_check_keepalive_timeouts(wssshd_state_t *state);

// WebSocket server functions (to be implemented with libwebsockets or similar)
int websocket_start_server(const wssshd_config_t *config, wssshd_state_t *state);
void websocket_stop_server(void);

#endif /* WEBSOCKET_H */