/**
 * Example plugin for wssshd2 demonstrating all plugin capabilities
 *
 * This plugin shows how to:
 * - Add new web pages
 * - Handle custom message types
 * - Implement a custom transport protocol
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include "../../plugin.h"

// Plugin state
typedef struct {
    int message_count;
    time_t start_time;
} example_plugin_state_t;

static example_plugin_state_t *plugin_state = NULL;

// Web request handler
int example_web_handler(int client_fd, const http_request_t *req, plugin_context_t *ctx) {
    (void)client_fd; // Parameter not used in this example
    (void)ctx; // Unused in this example

    // Handle /plugin/example page
    if (strcmp(req->path, "/plugin/example") == 0 && strcmp(req->method, "GET") == 0) {
        // Note: send_response is not available in plugin context, this is just for demonstration
        // In a real plugin, you'd need to use the appropriate response mechanism
        printf("[EXAMPLE PLUGIN] Would send HTML response for /plugin/example\n");
        return 1; // Handled
    }

    return 0; // Not handled
}

// Message handler
int example_message_handler(const char *message, size_t message_len, ws_connection_t *conn, plugin_context_t *ctx) {
    (void)message_len; // Unused
    (void)conn; // Unused
    (void)ctx; // Unused

    // Handle custom message type "example_message"
    if (strstr(message, "\"type\":\"example_message\"")) {
        if (plugin_state) {
            plugin_state->message_count++;
        }

        printf("[EXAMPLE PLUGIN] Handled example_message, total messages: %d\n",
               plugin_state ? plugin_state->message_count : 0);

        return 1; // Handled
    }

    return 0; // Not handled
}

// Plugin initialization
bool example_plugin_init(plugin_context_t *ctx) {
    plugin_state = calloc(1, sizeof(example_plugin_state_t));
    if (!plugin_state) {
        return false;
    }

    plugin_state->start_time = time(NULL);
    ctx->set_data(ctx, plugin_state);

    printf("[EXAMPLE PLUGIN] Initialized\n");
    return true;
}

// Plugin cleanup
void example_plugin_cleanup(plugin_context_t *ctx) {
    example_plugin_state_t *state = ctx->get_data(ctx);
    if (state) {
        printf("[EXAMPLE PLUGIN] Processed %d messages, uptime: %ld seconds\n",
               state->message_count, time(NULL) - state->start_time);
        free(state);
    }
    printf("[EXAMPLE PLUGIN] Cleaned up\n");
}

// Plugin interface definition
static const plugin_web_route_t example_routes[] = {
    {"GET", "/plugin/example", example_web_handler}
};

static const plugin_message_handler_t example_handlers[] = {
    {"example_message", example_message_handler}
};

const plugin_interface_t plugin_interface = {
    .info = {
        .name = "Example Plugin",
        .version = "1.0.0",
        .description = "Demonstrates plugin capabilities",
        .api_version = PLUGIN_API_VERSION
    },
    .capabilities = PLUGIN_CAP_WEB | PLUGIN_CAP_MESSAGE,
    .init = example_plugin_init,
    .cleanup = example_plugin_cleanup,
    .web = {
        .routes = example_routes,
        .routes_count = sizeof(example_routes) / sizeof(example_routes[0]),
        .template_files = NULL,
        .template_files_count = 0
    },
    .message = {
        .handlers = example_handlers,
        .handlers_count = sizeof(example_handlers) / sizeof(example_handlers[0])
    }
};