/**
 * Plugin system for wssshd2
 *
 * Copyright (C) 2024 Stefy Lanza <stefy@nexlab.net> and SexHack.me
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */

#ifndef PLUGIN_H
#define PLUGIN_H

#include <stdbool.h>
#include <stddef.h>
#include "websocket.h"
#include "config.h"
#include "tunnel.h"   // For tunnel_t

// Type aliases for plugin interface
typedef tunnel_t tunnel_session_t;

// Plugin version
#define PLUGIN_API_VERSION 1

// Forward declarations
typedef struct {
    char method[10];
    char path[1024];
    char query[1024];
    char headers[4096];
    char body[4096];
    int content_length;
} http_request_t;

// Plugin information
typedef struct {
    const char *name;
    const char *version;
    const char *description;
    int api_version;
} plugin_info_t;

// Forward declaration
typedef struct plugin_context plugin_context_t;

// Plugin context passed to all plugin functions
struct plugin_context {
    wssshd_state_t *state;
    const wssshd_config_t *config;
    void *plugin_data; // Plugin-specific data

    // Utility functions
    void *(*get_data)(plugin_context_t *ctx);
    void (*set_data)(plugin_context_t *ctx, void *data);
};

// Web route handler function type
typedef int (*web_route_handler_t)(int client_fd, const http_request_t *req, plugin_context_t *ctx);

// Web route definition
typedef struct {
    const char *method;     // "GET", "POST", etc.
    const char *path;       // Route path (e.g., "/myplugin")
    web_route_handler_t handler;
} plugin_web_route_t;

// Message handler function type
typedef int (*message_handler_t)(const char *message, size_t message_len, ws_connection_t *conn, plugin_context_t *ctx);

// Message handler definition
typedef struct {
    const char *message_type;   // Message type to handle (e.g., "my_message")
    message_handler_t handler;
} plugin_message_handler_t;

// Transport protocol handler

// Transport handler function types
typedef tunnel_session_t *(*transport_create_session_t)(plugin_context_t *ctx, const char *client_id, const char *username, bool debug);
typedef void (*transport_free_session_t)(tunnel_session_t *session);
typedef bool (*transport_is_running_t)(tunnel_session_t *session);
typedef char *(*transport_get_output_t)(tunnel_session_t *session, size_t *output_len);
typedef bool (*transport_send_data_t)(tunnel_session_t *session, const char *data, size_t len);

// Transport protocol definition
typedef struct {
    const char *protocol_name;   // Protocol name (e.g., "custom")
    transport_create_session_t create_session;
    transport_free_session_t free_session;
    transport_is_running_t is_running;
    transport_get_output_t get_output;
    transport_send_data_t send_data;
} plugin_transport_protocol_t;

// Plugin capabilities flags
#define PLUGIN_CAP_WEB        (1 << 0)  // Can add web routes/pages
#define PLUGIN_CAP_MESSAGE    (1 << 1)  // Can handle messages
#define PLUGIN_CAP_TRANSPORT  (1 << 2)  // Can provide transport protocols

// Plugin interface structure
typedef struct {
    // Plugin information
    plugin_info_t info;

    // Capabilities (bitmask of PLUGIN_CAP_*)
    unsigned int capabilities;

    // Initialization and cleanup
    bool (*init)(plugin_context_t *ctx);
    void (*cleanup)(plugin_context_t *ctx);

    // Web interface (optional)
    struct {
        const plugin_web_route_t *routes;
        size_t routes_count;
        const char **template_files;  // Array of template file paths
        size_t template_files_count;
    } web;

    // Message bus (optional)
    struct {
        const plugin_message_handler_t *handlers;
        size_t handlers_count;
    } message;

    // Transport protocols (optional)
    struct {
        const plugin_transport_protocol_t *protocols;
        size_t protocols_count;
    } transport;

} plugin_interface_t;

// Plugin loading functions
bool plugin_load(const char *plugin_path, plugin_context_t *ctx);
void plugin_unload(plugin_context_t *ctx);

// Plugin management
bool plugin_system_init(wssshd_state_t *state, const wssshd_config_t *config);
void plugin_system_cleanup(void);
bool plugin_load_from_directory(const char *plugin_dir);

// Plugin utility functions
void *plugin_get_data(plugin_context_t *ctx);
void plugin_set_data(plugin_context_t *ctx, void *data);

// Web interface integration
int plugin_handle_web_request(int client_fd, const http_request_t *req);

// Message bus integration
int plugin_handle_message(wssshd_state_t *state, ws_connection_t *conn, const char *message, size_t message_len);

// Transport protocol integration
tunnel_session_t *plugin_create_transport_session(const char *protocol, plugin_context_t *ctx, const char *client_id, const char *username, bool debug);

#endif /* PLUGIN_H */