#!/usr/bin/env python3
"""
VideoGen Web Interface
======================

A web interface for VideoGen that provides access to all features
including video generation, audio, dubbing, and more.

Usage:
    python webapp.py --port 5000 --host 0.0.0.0

Copyleft © 2026 Stefy <stefy@nexlab.net>
Licensed under GNU General Public License v3.0 or later
"""

import os
import sys
import json
import uuid
import subprocess
import threading
import queue
import time
import re
import shutil
from pathlib import Path
from datetime import datetime
from typing import Dict, List, Optional, Any
from dataclasses import dataclass, asdict
import argparse

# Flask and extensions
from flask import Flask, request, jsonify, send_file, send_from_directory, Response
from flask_cors import CORS
from flask_socketio import SocketIO, emit
from werkzeug.utils import secure_filename

# Note: Using threading mode instead of eventlet for Python 3.13+ compatibility
# eventlet is deprecated and has issues with newer Python versions

# Configuration
UPLOAD_FOLDER = Path.home() / ".config" / "videogen" / "webapp" / "uploads"
OUTPUT_FOLDER = Path.home() / ".config" / "videogen" / "webapp" / "outputs"
JOBS_FILE = Path.home() / ".config" / "videogen" / "webapp" / "jobs.json"
ALLOWED_EXTENSIONS = {
    'image': {'png', 'jpg', 'jpeg', 'gif', 'webp', 'bmp'},
    'video': {'mp4', 'avi', 'mov', 'mkv', 'webm', 'gif'},
    'audio': {'mp3', 'wav', 'ogg', 'flac', 'aac'},
    'subtitle': {'srt', 'vtt', 'ass', 'ssa'},
}

# Ensure directories exist
UPLOAD_FOLDER.mkdir(parents=True, exist_ok=True)
OUTPUT_FOLDER.mkdir(parents=True, exist_ok=True)

# Flask app setup
app = Flask(__name__, static_folder='static', template_folder='templates')
app.config['UPLOAD_FOLDER'] = str(UPLOAD_FOLDER)
app.config['OUTPUT_FOLDER'] = str(OUTPUT_FOLDER)
app.config['MAX_CONTENT_LENGTH'] = 500 * 1024 * 1024  # 500MB max upload
app.config['SECRET_KEY'] = os.environ.get('SECRET_KEY', 'videogen-webapp-secret-key')

CORS(app)
# Use threading mode for Python 3.13+ compatibility (eventlet is deprecated)
socketio = SocketIO(app, cors_allowed_origins="*", async_mode='threading')

# Job storage
@dataclass
class Job:
    id: str
    status: str  # pending, running, completed, failed, cancelled
    command: str
    created_at: str
    started_at: Optional[str] = None
    completed_at: Optional[str] = None
    progress: float = 0.0
    progress_text: str = ""
    output_files: List[str] = None
    error: Optional[str] = None
    logs: List[str] = None
    
    def __post_init__(self):
        if self.output_files is None:
            self.output_files = []
        if self.logs is None:
            self.logs = []

# In-memory job storage (could be replaced with database)
jobs: Dict[str, Job] = {}
job_queues: Dict[str, queue.Queue] = {}

# Load existing jobs from file
def load_jobs():
    global jobs
    if JOBS_FILE.exists():
        try:
            with open(JOBS_FILE, 'r') as f:
                data = json.load(f)
                for job_id, job_data in data.items():
                    jobs[job_id] = Job(**job_data)
        except Exception as e:
            print(f"Error loading jobs: {e}")

def save_jobs():
    try:
        with open(JOBS_FILE, 'w') as f:
            json.dump({k: asdict(v) for k, v in jobs.items()}, f, indent=2)
    except Exception as e:
        print(f"Error saving jobs: {e}")

# Helper functions
def allowed_file(filename: str, file_type: str) -> bool:
    """Check if file extension is allowed for the given type"""
    if '.' not in filename:
        return False
    ext = filename.rsplit('.', 1)[1].lower()
    return ext in ALLOWED_EXTENSIONS.get(file_type, set())

def get_model_list() -> List[Dict]:
    """Get list of available models from videogen"""
    try:
        result = subprocess.run(
            ['python3', 'videogen', '--model-list', '--json'],
            capture_output=True,
            text=True,
            timeout=30
        )
        if result.returncode == 0:
            return json.loads(result.stdout)
    except Exception as e:
        print(f"Error getting model list: {e}")
    return []

def get_tts_voices() -> List[Dict]:
    """Get list of available TTS voices"""
    return [
        {"id": "edge_female_us", "name": "Edge TTS - Female (US)", "engine": "edge"},
        {"id": "edge_male_us", "name": "Edge TTS - Male (US)", "engine": "edge"},
        {"id": "edge_female_uk", "name": "Edge TTS - Female (UK)", "engine": "edge"},
        {"id": "edge_male_uk", "name": "Edge TTS - Male (UK)", "engine": "edge"},
        {"id": "edge_female_es", "name": "Edge TTS - Female (Spanish)", "engine": "edge"},
        {"id": "edge_male_es", "name": "Edge TTS - Male (Spanish)", "engine": "edge"},
        {"id": "edge_female_fr", "name": "Edge TTS - Female (French)", "engine": "edge"},
        {"id": "edge_male_fr", "name": "Edge TTS - Male (French)", "engine": "edge"},
        {"id": "edge_female_de", "name": "Edge TTS - Female (German)", "engine": "edge"},
        {"id": "edge_male_de", "name": "Edge TTS - Male (German)", "engine": "edge"},
        {"id": "edge_female_it", "name": "Edge TTS - Female (Italian)", "engine": "edge"},
        {"id": "edge_male_it", "name": "Edge TTS - Male (Italian)", "engine": "edge"},
        {"id": "edge_female_ja", "name": "Edge TTS - Female (Japanese)", "engine": "edge"},
        {"id": "edge_male_ja", "name": "Edge TTS - Male (Japanese)", "engine": "edge"},
        {"id": "edge_female_zh", "name": "Edge TTS - Female (Chinese)", "engine": "edge"},
        {"id": "edge_male_zh", "name": "Edge TTS - Male (Chinese)", "engine": "edge"},
        {"id": "bark_speaker_0", "name": "Bark - Speaker 0", "engine": "bark"},
        {"id": "bark_speaker_1", "name": "Bark - Speaker 1", "engine": "bark"},
        {"id": "bark_speaker_2", "name": "Bark - Speaker 2", "engine": "bark"},
        {"id": "bark_speaker_3", "name": "Bark - Speaker 3", "engine": "bark"},
    ]

def get_languages() -> List[Dict]:
    """Get list of supported languages for translation"""
    return [
        {"code": "en", "name": "English"},
        {"code": "es", "name": "Spanish"},
        {"code": "fr", "name": "French"},
        {"code": "de", "name": "German"},
        {"code": "it", "name": "Italian"},
        {"code": "pt", "name": "Portuguese"},
        {"code": "ru", "name": "Russian"},
        {"code": "zh", "name": "Chinese"},
        {"code": "ja", "name": "Japanese"},
        {"code": "ko", "name": "Korean"},
        {"code": "ar", "name": "Arabic"},
        {"code": "hi", "name": "Hindi"},
        {"code": "nl", "name": "Dutch"},
        {"code": "pl", "name": "Polish"},
        {"code": "tr", "name": "Turkish"},
        {"code": "vi", "name": "Vietnamese"},
        {"code": "th", "name": "Thai"},
        {"code": "id", "name": "Indonesian"},
        {"code": "sv", "name": "Swedish"},
        {"code": "uk", "name": "Ukrainian"},
    ]

def build_command(params: Dict) -> List[str]:
    """Build videogen command from parameters"""
    cmd = ['python3', 'videogen']
    
    # Mode selection
    mode = params.get('mode', 't2v')
    
    # Prompt
    if params.get('prompt'):
        cmd.extend(['--prompt', params['prompt']])
    
    # Model selection
    if params.get('model'):
        cmd.extend(['--model', params['model']])
    
    if params.get('image_model'):
        cmd.extend(['--image-model', params['image_model']])
    
    # Output
    output_name = params.get('output', f"output_{uuid.uuid4().hex[:8]}")
    cmd.extend(['--output', str(OUTPUT_FOLDER / output_name)])
    
    # Resolution
    if params.get('width'):
        cmd.extend(['--width', str(params['width'])])
    if params.get('height'):
        cmd.extend(['--height', str(params['height'])])
    
    # Duration and FPS
    if params.get('length'):
        cmd.extend(['--length', str(params['length'])])
    if params.get('fps'):
        cmd.extend(['--fps', str(params['fps'])])
    
    # Seed
    if params.get('seed'):
        cmd.extend(['--seed', str(params['seed'])])
    
    # Mode-specific options
    if mode == 'i2v':
        cmd.append('--image-to-video')
        if params.get('prompt_image'):
            cmd.extend(['--prompt-image', params['prompt_image']])
        if params.get('prompt_animation'):
            cmd.extend(['--prompt-animation', params['prompt_animation']])
    
    elif mode == 't2i':
        cmd.append('--generate-image')
        if params.get('image_steps'):
            cmd.extend(['--image-steps', str(params['image_steps'])])
    
    elif mode == 'i2i':
        cmd.append('--image-to-image')
        if params.get('strength'):
            cmd.extend(['--strength', str(params['strength'])])
    
    # Input files
    if params.get('input_image'):
        cmd.extend(['--image', params['input_image']])
    if params.get('input_video'):
        cmd.extend(['--video', params['input_video']])
    if params.get('input_audio'):
        cmd.extend(['--audio-file', params['input_audio']])
    
    # Audio options
    if params.get('generate_audio'):
        cmd.append('--generate-audio')
        if params.get('audio_type'):
            cmd.extend(['--audio-type', params['audio_type']])
        if params.get('audio_text'):
            cmd.extend(['--audio-text', params['audio_text']])
        if params.get('tts_voice'):
            cmd.extend(['--tts-voice', params['tts_voice']])
        if params.get('music_prompt'):
            cmd.extend(['--music-prompt', params['music_prompt']])
    
    if params.get('sync_audio'):
        cmd.append('--sync-audio')
        if params.get('audio_sync_mode'):
            cmd.extend(['--audio-sync-mode', params['audio_sync_mode']])
    
    if params.get('lip_sync'):
        cmd.append('--lip-sync')
        if params.get('lip_sync_method'):
            cmd.extend(['--lip-sync-method', params['lip_sync_method']])
    
    # Dubbing/Translation
    if params.get('dub_video'):
        cmd.append('--dub-video')
        if params.get('target_lang'):
            cmd.extend(['--target-lang', params['target_lang']])
        if params.get('source_lang'):
            cmd.extend(['--source-lang', params['source_lang']])
        if params.get('voice_clone'):
            cmd.append('--voice-clone')
        else:
            cmd.append('--no-voice-clone')
    
    # Subtitles
    if params.get('create_subtitles'):
        cmd.append('--create-subtitles')
    if params.get('translate_subtitles'):
        cmd.append('--translate-subtitles')
    if params.get('burn_subtitles'):
        cmd.append('--burn-subtitles')
    if params.get('subtitle_style'):
        cmd.extend(['--subtitle-style', params['subtitle_style']])
    
    # Transcription
    if params.get('transcribe'):
        cmd.append('--transcribe')
        if params.get('whisper_model'):
            cmd.extend(['--whisper-model', params['whisper_model']])
    
    # V2V options
    if params.get('video_to_video'):
        cmd.append('--video-to-video')
        if params.get('v2v_strength'):
            cmd.extend(['--v2v-strength', str(params['v2v_strength'])])
    
    # 2D to 3D
    if params.get('convert_3d'):
        cmd.append('--convert-3d')
        if params.get('depth_method'):
            cmd.extend(['--depth-method', params['depth_method']])
    
    # Upscale
    if params.get('upscale'):
        cmd.append('--upscale')
        if params.get('upscale_factor'):
            cmd.extend(['--upscale-factor', str(params['upscale_factor'])])
    
    # Character consistency
    if params.get('character_reference'):
        cmd.extend(['--character-reference', params['character_reference']])
    if params.get('character_strength'):
        cmd.extend(['--character-strength', str(params['character_strength'])])
    
    # NSFW
    if params.get('no_filter'):
        cmd.append('--no-filter')
    
    # Auto mode
    if params.get('auto'):
        cmd.append('--auto')
    
    # Offloading
    if params.get('offload_strategy'):
        cmd.extend(['--offload-strategy', params['offload_strategy']])
    
    # VRAM limit
    if params.get('vram_limit'):
        cmd.extend(['--vram-limit', str(params['vram_limit'])])
    
    # Debug
    if params.get('debug'):
        cmd.append('--debug')
    
    return cmd

def run_job(job_id: str, cmd: List[str]):
    """Run a videogen job in background"""
    job = jobs.get(job_id)
    if not job:
        return
    
    job.status = 'running'
    job.started_at = datetime.now().isoformat()
    save_jobs()
    socketio.emit('job_update', asdict(job), room=job_id)
    
    try:
        # Start subprocess
        process = subprocess.Popen(
            cmd,
            stdout=subprocess.PIPE,
            stderr=subprocess.STDOUT,
            text=True,
            bufsize=1,
            cwd=os.getcwd()
        )
        
        # Store process for cancellation
        job_queues[job_id] = queue.Queue()
        job_queues[job_id].put(process)
        
        # Read output line by line
        for line in iter(process.stdout.readline, ''):
            if jobs.get(job_id, Job(id='', status='', command='', created_at='')).status == 'cancelled':
                process.terminate()
                break
            
            line = line.rstrip()
            job.logs.append(line)
            
            # Parse progress from output
            progress_match = re.search(r'(\d+(?:\.\d+)?)\s*%', line)
            if progress_match:
                job.progress = float(progress_match.group(1))
            
            # Update progress text
            if 'Loading' in line or 'loading' in line:
                job.progress_text = line.strip()
            elif 'Generating' in line or 'generating' in line:
                job.progress_text = line.strip()
            elif 'Saving' in line or 'saving' in line:
                job.progress_text = line.strip()
            elif '✅' in line or '✨' in line:
                job.progress_text = line.strip()
            
            # Emit update
            socketio.emit('job_log', {'job_id': job_id, 'line': line}, room=job_id)
            socketio.emit('job_update', asdict(job), room=job_id)
        
        process.wait()
        
        if process.returncode == 0:
            job.status = 'completed'
            job.progress = 100.0
            job.progress_text = 'Generation completed successfully!'
            
            # Find output files
            output_base = cmd[cmd.index('--output') + 1] if '--output' in cmd else str(OUTPUT_FOLDER / 'output')
            for ext in ['.mp4', '.png', '.wav', '.srt', '_dubbed.mp4', '_upscaled.mp4']:
                potential_file = output_base + ext if not ext.startswith('_') else output_base.replace('.mp4', '') + ext
                if os.path.exists(potential_file):
                    job.output_files.append(os.path.basename(potential_file))
                # Also check for numbered outputs
                for f in OUTPUT_FOLDER.glob(f"{os.path.basename(output_base)}*{ext}"):
                    if f.name not in job.output_files:
                        job.output_files.append(f.name)
        else:
            job.status = 'failed'
            job.error = 'Process returned non-zero exit code'
        
    except Exception as e:
        job.status = 'failed'
        job.error = str(e)
    
    finally:
        job.completed_at = datetime.now().isoformat()
        save_jobs()
        socketio.emit('job_update', asdict(job), room=job_id)
        if job_id in job_queues:
            del job_queues[job_id]

# Routes
@app.route('/')
def index():
    """Serve the main page"""
    return send_from_directory('templates', 'index.html')

@app.route('/static/<path:filename>')
def static_files(filename):
    """Serve static files"""
    return send_from_directory('static', filename)

@app.route('/api/models', methods=['GET'])
def api_models():
    """Get list of available models"""
    models = get_model_list()
    return jsonify(models)

@app.route('/api/tts-voices', methods=['GET'])
def api_tts_voices():
    """Get list of TTS voices"""
    return jsonify(get_tts_voices())

@app.route('/api/languages', methods=['GET'])
def api_languages():
    """Get list of supported languages"""
    return jsonify(get_languages())

@app.route('/api/upload', methods=['POST'])
def upload_file():
    """Upload a file"""
    if 'file' not in request.files:
        return jsonify({'error': 'No file provided'}), 400
    
    file = request.files['file']
    if file.filename == '':
        return jsonify({'error': 'No file selected'}), 400
    
    file_type = request.form.get('type', 'image')
    
    if not allowed_file(file.filename, file_type):
        return jsonify({'error': f'File type not allowed for {file_type}'}), 400
    
    # Generate unique filename
    ext = file.filename.rsplit('.', 1)[1].lower()
    filename = f"{uuid.uuid4().hex}.{ext}"
    filepath = UPLOAD_FOLDER / filename
    
    file.save(filepath)
    
    return jsonify({
        'filename': filename,
        'path': str(filepath),
        'size': filepath.stat().st_size
    })

@app.route('/api/jobs', methods=['GET'])
def api_jobs():
    """Get list of jobs"""
    job_list = [asdict(job) for job in jobs.values()]
    job_list.sort(key=lambda x: x['created_at'], reverse=True)
    return jsonify(job_list)

@app.route('/api/jobs/<job_id>', methods=['GET'])
def api_job(job_id):
    """Get job details"""
    job = jobs.get(job_id)
    if not job:
        return jsonify({'error': 'Job not found'}), 404
    return jsonify(asdict(job))

@app.route('/api/jobs', methods=['POST'])
def create_job():
    """Create a new job"""
    params = request.json
    
    # Generate job ID
    job_id = uuid.uuid4().hex[:8]
    
    # Build command
    cmd = build_command(params)
    
    # Create job
    job = Job(
        id=job_id,
        status='pending',
        command=' '.join(cmd),
        created_at=datetime.now().isoformat()
    )
    
    jobs[job_id] = job
    save_jobs()
    
    # Start job in background
    thread = threading.Thread(target=run_job, args=(job_id, cmd))
    thread.daemon = True
    thread.start()
    
    return jsonify(asdict(job))

@app.route('/api/jobs/<job_id>/cancel', methods=['POST'])
def cancel_job(job_id):
    """Cancel a running job"""
    job = jobs.get(job_id)
    if not job:
        return jsonify({'error': 'Job not found'}), 404
    
    if job.status == 'running':
        job.status = 'cancelled'
        job.error = 'Cancelled by user'
        save_jobs()
        socketio.emit('job_update', asdict(job), room=job_id)
    
    return jsonify(asdict(job))

@app.route('/api/jobs/<job_id>/retry', methods=['POST'])
def retry_job(job_id):
    """Retry a failed job"""
    job = jobs.get(job_id)
    if not job:
        return jsonify({'error': 'Job not found'}), 404
    
    if job.status not in ['failed', 'cancelled']:
        return jsonify({'error': 'Can only retry failed or cancelled jobs'}), 400
    
    # Reset job
    job.status = 'pending'
    job.error = None
    job.progress = 0.0
    job.progress_text = ''
    job.logs = []
    job.output_files = []
    job.started_at = None
    job.completed_at = None
    save_jobs()
    
    # Rebuild command and start
    cmd = job.command.split()
    thread = threading.Thread(target=run_job, args=(job_id, cmd))
    thread.daemon = True
    thread.start()
    
    return jsonify(asdict(job))

@app.route('/api/jobs/<job_id>', methods=['DELETE'])
def delete_job(job_id):
    """Delete a job"""
    if job_id in jobs:
        del jobs[job_id]
        save_jobs()
        return jsonify({'success': True})
    return jsonify({'error': 'Job not found'}), 404

@app.route('/api/download/<filename>')
def download_file(filename):
    """Download an output file"""
    filepath = OUTPUT_FOLDER / filename
    if not filepath.exists():
        # Check in uploads folder too
        filepath = UPLOAD_FOLDER / filename
        if not filepath.exists():
            return jsonify({'error': 'File not found'}), 404
    
    return send_file(filepath, as_attachment=True)

@app.route('/api/outputs')
def list_outputs():
    """List all output files"""
    files = []
    for f in OUTPUT_FOLDER.iterdir():
        if f.is_file():
            files.append({
                'name': f.name,
                'size': f.stat().st_size,
                'modified': datetime.fromtimestamp(f.stat().st_mtime).isoformat()
            })
    files.sort(key=lambda x: x['modified'], reverse=True)
    return jsonify(files)

@app.route('/api/outputs/<filename>', methods=['DELETE'])
def delete_output(filename):
    """Delete an output file"""
    filepath = OUTPUT_FOLDER / filename
    if filepath.exists():
        filepath.unlink()
        return jsonify({'success': True})
    return jsonify({'error': 'File not found'}), 404

# Character Profile API endpoints
CHARACTERS_DIR = Path.home() / ".config" / "videogen" / "characters"

@app.route('/api/characters', methods=['GET'])
def api_list_characters():
    """List all character profiles"""
    characters = []
    if CHARACTERS_DIR.exists():
        for profile_file in CHARACTERS_DIR.glob("*.json"):
            try:
                with open(profile_file, 'r') as f:
                    profile = json.load(f)
                    characters.append({
                        'name': profile.get('name', profile_file.stem),
                        'description': profile.get('description', ''),
                        'image_count': len(profile.get('reference_images', [])),
                        'created': profile.get('created', ''),
                        'tags': profile.get('tags', [])
                    })
            except Exception as e:
                print(f"Error loading character profile {profile_file}: {e}")
    
    return jsonify(characters)

@app.route('/api/characters/<name>', methods=['GET'])
def api_get_character(name):
    """Get a specific character profile"""
    profile_path = CHARACTERS_DIR / f"{name}.json"
    if profile_path.exists():
        try:
            with open(profile_path, 'r') as f:
                return jsonify(json.load(f))
        except Exception as e:
            return jsonify({'error': str(e)}), 500
    return jsonify({'error': 'Character not found'}), 404

@app.route('/api/characters', methods=['POST'])
def api_create_character():
    """Create a new character profile"""
    name = request.form.get('name')
    description = request.form.get('description', '')
    
    if not name:
        return jsonify({'error': 'Name is required'}), 400
    
    # Sanitize name
    name = re.sub(r'[^a-zA-Z0-9_-]', '_', name)
    
    # Handle uploaded images
    images = request.files.getlist('images')
    if not images or len(images) == 0:
        return jsonify({'error': 'At least one reference image is required'}), 400
    
    # Create character directory
    CHARACTERS_DIR.mkdir(parents=True, exist_ok=True)
    char_image_dir = CHARACTERS_DIR / name
    char_image_dir.mkdir(parents=True, exist_ok=True)
    
    # Save images
    saved_images = []
    for i, img in enumerate(images[:5]):  # Max 5 images
        if img and img.filename:
            ext = img.filename.rsplit('.', 1)[-1].lower()
            if ext in ALLOWED_EXTENSIONS['image']:
                filename = f"reference_{i+1}.{ext}"
                filepath = char_image_dir / filename
                img.save(filepath)
                saved_images.append(str(filepath))
    
    if not saved_images:
        return jsonify({'error': 'No valid images uploaded'}), 400
    
    # Create profile
    profile = {
        'name': name,
        'description': description,
        'reference_images': saved_images,
        'created': datetime.now().isoformat(),
        'tags': []
    }
    
    # Save profile
    profile_path = CHARACTERS_DIR / f"{name}.json"
    with open(profile_path, 'w') as f:
        json.dump(profile, f, indent=2)
    
    return jsonify(profile)

@app.route('/api/characters/<name>', methods=['DELETE'])
def api_delete_character(name):
    """Delete a character profile"""
    profile_path = CHARACTERS_DIR / f"{name}.json"
    char_image_dir = CHARACTERS_DIR / name
    
    if not profile_path.exists():
        return jsonify({'error': 'Character not found'}), 404
    
    try:
        # Delete profile file
        profile_path.unlink()
        
        # Delete images directory
        if char_image_dir.exists():
            shutil.rmtree(char_image_dir)
        
        return jsonify({'success': True})
    except Exception as e:
        return jsonify({'error': str(e)}), 500

@app.route('/api/upload-multiple', methods=['POST'])
def upload_multiple_files():
    """Upload multiple files (for reference images)"""
    files = request.files.getlist('files')
    upload_type = request.form.get('type', 'general')
    
    if not files:
        return jsonify({'error': 'No files provided'}), 400
    
    saved_paths = []
    for f in files:
        if f and f.filename:
            ext = f.filename.rsplit('.', 1)[-1].lower()
            if ext in ALLOWED_EXTENSIONS['image']:
                filename = f"{uuid.uuid4().hex[:8]}_{secure_filename(f.filename)}"
                filepath = UPLOAD_FOLDER / filename
                f.save(filepath)
                saved_paths.append(str(filepath))
    
    return jsonify({'paths': saved_paths})

# WebSocket events
@socketio.on('connect')
def handle_connect():
    print(f"Client connected: {request.sid}")

@socketio.on('disconnect')
def handle_disconnect():
    print(f"Client disconnected: {request.sid}")

@socketio.on('subscribe_job')
def handle_subscribe_job(job_id):
    """Subscribe to job updates"""
    from flask_socketio import join_room
    join_room(job_id)
    if job_id in jobs:
        emit('job_update', asdict(jobs[job_id]))

# Main entry point
if __name__ == '__main__':
    parser = argparse.ArgumentParser(description='VideoGen Web Interface')
    parser.add_argument('--host', default='0.0.0.0', help='Host to bind to')
    parser.add_argument('--port', type=int, default=5000, help='Port to bind to')
    parser.add_argument('--debug', action='store_true', help='Enable debug mode')
    args = parser.parse_args()
    
    # Load existing jobs
    load_jobs()
    
    print(f"""
╔══════════════════════════════════════════════════════════════╗
║                    VideoGen Web Interface                    ║
╠══════════════════════════════════════════════════════════════╣
║  Starting server on http://{args.host}:{args.port}                       ║
║  Upload folder: {UPLOAD_FOLDER}              ║
║  Output folder: {OUTPUT_FOLDER}              ║
╚══════════════════════════════════════════════════════════════╝
    """)
    
    socketio.run(app, host=args.host, port=args.port, debug=args.debug)