#!/usr/bin/env python3
"""Check available pipeline variants in diffusers library"""
import sys

def check_pipelines():
    """Check for pipeline variants in diffusers"""
    print("=" * 70)
    print("CHECKING AVAILABLE PIPELINE VARIANTS IN DIFFUSERS")
    print("=" * 70)
    
    try:
        import diffusers
    except ImportError:
        print("ERROR: diffusers not installed")
        print("Install with: pip install diffusers")
        sys.exit(1)
    
    # Get all pipeline-related names
    all_names = dir(diffusers)
    pipeline_names = [n for n in all_names if 'Pipeline' in n]
    
    print(f"\nFound {len(pipeline_names)} Pipeline classes\n")
    
    # Print all Video-related pipelines
    print("\n" + "=" * 70)
    print("VIDEO PIPELINES (T2V, I2V, V2V)")
    print("=" * 70)
    
    video_pipelines = {
        "T2V": [],
        "I2V": [],
        "V2V": []
    }
    
    for name in sorted(pipeline_names):
        name_lower = name.lower()
        # T2V pipelines
        if any(x in name_lower for x in ["texttovideo", "text-to-video", "mochi", "ltx", "cogvideo", "animatediff"]) and "image" not in name_lower and "video" not in name_lower:
            if "TextToVideo" in name or "MoChi" in name or name in ["LTXPipeline", "CogVideoXPipeline", "AnimateDiffPipeline"]:
                video_pipelines["T2V"].append(name)
        # I2V pipelines
        elif "imagetovideo" in name_lower or "image-to-video" in name_lower or "i2v" in name_lower:
            video_pipelines["I2V"].append(name)
        # V2V pipelines
        elif "videotovideo" in name_lower or "video-to-video" in name_lower or "v2v" in name_lower:
            video_pipelines["V2V"].append(name)
    
    print("\nT2V (Text-to-Video):")
    for p in sorted(video_pipelines["T2V"]):
        print(f"  - {p}")
    
    print("\nI2V (Image-to-Video):")
    for p in sorted(video_pipelines["I2V"]):
        print(f"  - {p}")
    
    print("\nV2V (Video-to-Video):")
    for p in sorted(video_pipelines["V2V"]):
        print(f"  - {p}")
    
    # Print all Image-related pipelines
    print("\n" + "=" * 70)
    print("IMAGE PIPELINES (T2I, I2I)")
    print("=" * 70)
    
    image_pipelines = {
        "T2I": [],
        "I2I": []
    }
    
    for name in sorted(pipeline_names):
        name_lower = name.lower()
        # T2I pipelines
        if "texttoimage" in name_lower or "text-to-image" in name_lower:
            if "img2img" not in name_lower and "inpaint" not in name_lower:
                image_pipelines["T2I"].append(name)
        # I2I pipelines
        elif "img2img" in name_lower or "image-to-image" in name_lower:
            image_pipelines["I2I"].append(name)
    
    print("\nT2I (Text-to-Image):")
    for p in sorted(set(image_pipelines["T2I"])):
        print(f"  - {p}")
    
    print("\nI2I (Image-to-Image):")
    for p in sorted(set(image_pipelines["I2I"])):
        print(f"  - {p}")
    
    print("\n" + "=" * 70)
    print("SUMMARY - PIPELINE SELECTION BY MODEL + TASK")
    print("=" * 70)
    print("""
T2V (Text-to-Video):
  - WanPipeline
  - LTXPipeline  
  - CogVideoXPipeline
  - MochiPipeline
  - AnimateDiffPipeline

I2V (Image-to-Video):
  - WanImageToVideoPipeline
  - LTXImageToVideoPipeline
  - StableVideoDiffusionPipeline
  - CogVideoXImageToVideoPipeline
  - I2VGenXLPipeline

V2V (Video-to-Video):
  - WanVideoToVideoPipeline
  - CogVideoXVideoToVideoPipeline
  - AnimateDiffVideoToVideoPipeline

T2I (Text-to-Image):
  - FluxPipeline
  - StableDiffusionXLPipeline
  - StableDiffusion3Pipeline

I2I (Image-to-Image):
  - FluxImg2ImgPipeline
  - StableDiffusionXLImg2ImgPipeline
  - StableDiffusion3Img2ImgPipeline
""")
    print(f"\ndiffusers version: {diffusers.__version__}")

if __name__ == "__main__":
    check_pipelines()
