# VideoGen Examples

This document contains comprehensive examples for using the VideoGen toolkit, covering all modes, model sizes, and features.

---

## Table of Contents

1. [Basic Usage](#basic-usage)
2. [Auto Mode](#auto-mode)
3. [Text-to-Video (T2V)](#text-to-video-t2v)
4. [Image-to-Video (I2V)](#image-to-video-i2v)
5. [Text-to-Image (T2I)](#text-to-image-t2i)
6. [Image-to-Image (I2I)](#image-to-image-i2i)
7. [Audio Generation](#audio-generation)
8. [Lip Sync](#lip-sync)
9. [Distributed Multi-GPU](#distributed-multi-gpu)
10. [Model Management](#model-management)
11. [VRAM Management](#vram-management)
12. [Upscaling](#upscaling)
13. [NSFW Content](#nsfw-content)
14. [Advanced Combinations](#advanced-combinations)

---

## Basic Usage

### Simple Text-to-Video
```bash
# Basic video generation with default settings
python3 videogen --model wan_1.3b_t2v --prompt "a cat playing piano" --output cat_piano

# Specify duration and resolution
python3 videogen --model wan_1.3b_t2v --prompt "sunset over mountains" \
  --length 10 --width 1280 --height 720 --output sunset

# Set specific seed for reproducibility
python3 videogen --model wan_1.3b_t2v --prompt "ocean waves" \
  --seed 42 --output ocean
```

### Quick Start with Auto Mode
```bash
# Let the script decide everything
python3 videogen --auto --prompt "a beautiful woman walking on the beach"

# Auto mode with custom output
python3 videogen --auto --prompt "epic space battle" --output space_battle

# Auto mode preferring speed over quality
python3 videogen --auto --prefer-speed --prompt "quick animation test"
```

---

## Auto Mode

Auto mode analyzes your prompt and automatically:
- Detects the generation type (T2V, I2V, T2I, I2I)
- Detects NSFW content and enables appropriate settings
- Selects the best model for your VRAM
- Splits prompts for I2V mode
- Configures audio if needed

### Basic Auto Mode
```bash
# Simple auto generation
python3 videogen --auto --prompt "a dog running in the park"

# Auto with custom duration
python3 videogen --auto --prompt "cinematic drone shot of city" --length 15

# Auto with custom resolution
python3 videogen --auto --prompt "portrait of a woman" \
  --width 1024 --height 1024
```

### Auto Mode with Audio
```bash
# Auto mode detects speech and enables TTS
python3 videogen --auto --prompt "a woman speaking to camera"

# Auto mode with music for action scene
python3 videogen --auto --prompt "epic battle scene with warriors fighting"

# Auto mode for nature documentary style
python3 videogen --auto --prompt "wildlife footage of lions hunting"
```

### Auto Mode for Different Content Types
```bash
# Static scene (will use T2I)
python3 videogen --auto --prompt "beautiful landscape photograph of mountains"

# Animated scene (will use T2V)
python3 videogen --auto --prompt "flowing river with fish jumping"

# Portrait with motion (will use I2V)
python3 videogen --auto --prompt "a woman with flowing hair in the wind"

# Action sequence (will use T2V with fast motion)
python3 videogen --auto --prompt "car chase through city streets"
```

### Auto Mode Output
When you run auto mode, it prints a command line you can use to reproduce the result:
```bash
# Example output from auto mode:
# 📝 COMMAND LINE (to reproduce without --auto):
# python3 videogen \
#   --model wan_14b_t2v \
#   --prompt "epic battle scene" \
#   --length 10 \
#   --fps 18 \
#   --generate_audio \
#   --audio_type music \
#   --sync_audio
```

---

## Text-to-Video (T2V)

### Small Models (<16GB VRAM)
```bash
# Wan 1.3B - Fast, good for testing
python3 videogen --model wan_1.3b_t2v --prompt "a bird flying" --length 5

# Zeroscope - Lightweight T2V
python3 videogen --model zeroscope_v2_576w --prompt "robot dancing" \
  --width 576 --height 320

# ModelScope - Basic T2V
python3 videogen --model modelscope_t2v --prompt "fireworks in night sky"
```

### Medium Models (16-30GB VRAM)
```bash
# Wan 14B - High quality T2V
python3 videogen --model wan_14b_t2v --prompt "cinematic shot of ancient ruins" \
  --length 10 --width 1280 --height 720

# CogVideoX 2B - Good balance
python3 videogen --model cogvideox_2b --prompt "underwater coral reef scene" \
  --length 8

# CogVideoX 5B - Higher quality
python3 videogen --model cogvideox_5b --prompt "northern lights over snowy landscape" \
  --length 12 --width 1280 --height 720

# Mochi - High quality
python3 videogen --model mochi_1_preview --prompt "slow motion water droplet" \
  --length 6 --fps 24
```

### Large Models (30-50GB VRAM)
```bash
# Allegro - Very high quality
python3 videogen --model allegro --prompt "epic fantasy battle" \
  --length 15 --width 1920 --height 1080 \
  --vram_limit 45 --offload_strategy sequential

# HunyuanVideo - Tencent's model
python3 videogen --model hunyuanvideo --prompt "photorealistic city street" \
  --length 10 --vram_limit 50
```

### Huge Models (50GB+ VRAM)
```bash
# Open-Sora - Open source Sora alternative
python3 videogen --model open_sora_1_2 --prompt "detailed nature documentary" \
  --length 20 --width 1920 --height 1080 \
  --vram_limit 60 --offload_strategy sequential

# Step-Video - Extreme quality (90-140GB VRAM)
python3 videogen --model step_video_t2v --prompt "cinematic masterpiece" \
  --length 30 --vram_limit 120 --offload_strategy sequential
```

### T2V with Custom Settings
```bash
# High FPS for smooth motion
python3 videogen --model wan_14b_t2v --prompt "sports action replay" \
  --fps 30 --length 5

# Low FPS for choppy/artistic effect
python3 videogen --model wan_14b_t2v --prompt "vintage film look" \
  --fps 12 --length 8

# Custom resolution (landscape)
python3 videogen --model wan_14b_t2v --prompt "panoramic mountain view" \
  --width 1920 --height 480

# Custom resolution (portrait)
python3 videogen --model wan_14b_t2v --prompt "vertical phone video" \
  --width 720 --height 1280

# Square format
python3 videogen --model wan_14b_t2v --prompt "instagram reel" \
  --width 1080 --height 1080
```

---

## Image-to-Video (I2V)

### I2V with Auto-Generated Image
```bash
# Basic I2V - generates image first, then animates
python3 videogen --image_to_video --model svd_xt_1.1 \
  --image_model flux_dev \
  --prompt "a beautiful woman in a red dress" \
  --prompt_animation "gentle wind blowing her hair" \
  --output woman_dress

# I2V with different image model
python3 videogen --image_to_video --model svd_xt_1.1 \
  --image_model pony_v6 \
  --prompt "anime girl with blue hair" \
  --prompt_animation "blinking and smiling" \
  --output anime_girl

# I2V with SDXL image model
python3 videogen --image_to_video --model svd_xt_1.1 \
  --image_model sdxl_base \
  --prompt "cyberpunk city street at night" \
  --prompt_animation "neon lights flickering, cars passing" \
  --output cyberpunk
```

### I2V with Existing Image
```bash
# Animate a photo
python3 videogen --model svd_xt_1.1 \
  --image my_photo.jpg \
  --prompt "add subtle motion to this scene" \
  --output animated_photo

# I2V with specific animation prompt
python3 videogen --model svd_xt_1.1 \
  --image portrait.png \
  --prompt "person turns head slightly and smiles" \
  --output portrait_animated

# I2V with custom resolution (must match image aspect ratio)
python3 videogen --model svd_xt_1.1 \
  --image landscape.jpg \
  --prompt "clouds moving across sky" \
  --width 1920 --height 1080 \
  --output landscape_motion
```

### I2V with Different Models
```bash
# SVD XT 1.1 - Standard I2V
python3 videogen --image_to_video --model svd_xt_1.1 \
  --image_model flux_dev --prompt "forest scene" \
  --prompt_animation "leaves rustling in wind"

# SVD XT - Original version
python3 videogen --image_to_video --model svd_xt \
  --image_model sdxl_base --prompt "ocean sunset" \
  --prompt_animation "waves gently rolling"

# Wan I2V - Alternative I2V model
python3 videogen --image_to_video --model wan_14b_i2v \
  --image_model flux_dev --prompt "mountain landscape" \
  --prompt_animation "clouds drifting, birds flying"

# LTX Video - Fast I2V
python3 videogen --image_to_video --model ltx_video \
  --image_model sdxl_base --prompt "city street" \
  --prompt_animation "people walking, cars moving"
```

### I2V with LoRA Adapters
```bash
# I2V with NSFW LoRA
python3 videogen --image_to_video \
  --model wan_i2v_nsfw_lora \
  --image_model flux_nsfw \
  --prompt "intimate scene" \
  --prompt_animation "passionate motion" \
  --no_filter --output intimate

# I2V with style LoRA
python3 videogen --image_to_video \
  --model wan_i2v_anime_lora \
  --image_model pony_v6 \
  --prompt "anime character" \
  --prompt_animation "dramatic pose change"
```

---

## Text-to-Image (T2I)

### Basic T2I
```bash
# Generate single image (auto-detected by .png extension)
python3 videogen --model flux_dev --prompt "beautiful landscape" --output landscape.png

# Generate with SDXL
python3 videogen --model sdxl_base --prompt "cyberpunk city" \
  --width 1024 --height 1024 --output cyberpunk.png

# Generate with Pony (anime style)
python3 videogen --model pony_v6 --prompt "anime girl with pink hair" \
  --output anime_girl.jpg

# Generate with custom steps
python3 videogen --model flux_dev --prompt "portrait of a woman" \
  --image-steps 50 --guidance-scale 3.5 --output portrait.png
```

### T2I with Different Sizes
```bash
# Square
python3 videogen --model sdxl_base --prompt "abstract art" \
  --width 1024 --height 1024 --output square.png

# Portrait
python3 videogen --model flux_dev --prompt "full body portrait" \
  --width 768 --height 1024 --output portrait.png

# Landscape
python3 videogen --model sdxl_base --prompt "panoramic view" \
  --width 1280 --height 720 --output landscape.png

# Ultra-wide
python3 videogen --model flux_dev --prompt "cinematic wide shot" \
  --width 1920 --height 540 --output ultrawide.png
```

### T2I for I2V Preparation
```bash
# Generate image for later I2V use
python3 videogen --model flux_dev \
  --prompt "cinematic shot of a woman in evening gown, dramatic lighting" \
  --width 832 --height 480 --output i2v_source.png

# Then animate it
python3 videogen --model svd_xt_1.1 --image i2v_source.png \
  --prompt "subtle elegant movement" --output animated_scene
```

---

## Image-to-Image (I2I)

### Basic I2I
```bash
# Modify existing image
python3 videogen --model flux_dev --image-to-image \
  --image photo.jpg \
  --prompt "make it look like an oil painting" \
  --output painted.png

# I2I with SDXL
python3 videogen --model sdxl_base --image-to-image \
  --image input.png \
  --prompt "add dramatic sunset lighting" \
  --output sunset.png

# I2I with Pony
python3 videogen --model pony_v6 --image-to-image \
  --image photo.jpg \
  --prompt "convert to anime style" \
  --output anime_version.png
```

### I2I with Strength Control
```bash
# Low strength - subtle changes
python3 videogen --model flux_dev --image-to-image \
  --image photo.jpg \
  --prompt "add slight blur effect" \
  --strength 0.3 --output subtle.png

# Medium strength - balanced
python3 videogen --model flux_dev --image-to-image \
  --image photo.jpg \
  --prompt "change to winter scene" \
  --strength 0.5 --output winter.png

# High strength - major changes
python3 videogen --model flux_dev --image-to-image \
  --image photo.jpg \
  --prompt "transform into fantasy art" \
  --strength 0.8 --output fantasy.png

# Maximum strength - almost complete reimagining
python3 videogen --model sdxl_base --image-to-image \
  --image sketch.png \
  --prompt "highly detailed realistic rendering" \
  --strength 0.95 --output detailed.png
```

### I2I Style Transfers
```bash
# Photo to painting
python3 videogen --model flux_dev --image-to-image \
  --image photo.jpg --prompt "impressionist oil painting style" \
  --strength 0.7 --output impressionist.png

# Realistic to anime
python3 videogen --model pony_v6 --image-to-image \
  --image portrait.jpg --prompt "anime style illustration" \
  --strength 0.75 --output anime.png

# Day to night
python3 videogen --model sdxl_base --image-to-image \
  --image daytime.jpg --prompt "night time scene with moon" \
  --strength 0.6 --output nighttime.png

# Add weather effects
python3 videogen --model flux_dev --image-to-image \
  --image sunny.jpg --prompt "rainy weather, wet surfaces" \
  --strength 0.5 --output rainy.png
```

---

## Audio Generation

### Text-to-Speech (TTS)
```bash
# Basic TTS with default voice
python3 videogen --model wan_14b_t2v --prompt "woman speaking" \
  --generate_audio --audio_type tts \
  --audio_text "Hello, welcome to my channel" \
  --output speaker

# TTS with male voice
python3 videogen --model wan_14b_t2v --prompt "man giving presentation" \
  --generate_audio --audio_type tts \
  --audio_text "Today we will discuss the future of technology" \
  --tts_voice edge_male_us --output presentation

# TTS with British accent
python3 videogen --model wan_14b_t2v --prompt "british narrator" \
  --generate_audio --audio_type tts \
  --audio_text "In this episode, we explore ancient mysteries" \
  --tts_voice edge_male_uk --output documentary

# TTS with Australian accent
python3 videogen --model wan_14b_t2v --prompt "australian guide" \
  --generate_audio --audio_type tts \
  --audio_text "Welcome to the outback adventure tour" \
  --tts_voice edge_male_au --output australia

# TTS with female voice
python3 videogen --model wan_14b_t2v --prompt "woman telling story" \
  --generate_audio --audio_type tts \
  --audio_text "Once upon a time in a land far away" \
  --tts_voice edge_female_us --output story
```

### TTS with Bark (Suno AI)
```bash
# Bark male voice
python3 videogen --model wan_14b_t2v --prompt "dramatic reading" \
  --generate_audio --audio_type tts \
  --audio_text "To be or not to be, that is the question" \
  --tts_voice bark_male --output hamlet

# Bark female voice
python3 videogen --model wan_14b_t2v --prompt "singing performance" \
  --generate_audio --audio_type tts \
  --audio_text "La la la, singing a beautiful melody" \
  --tts_voice bark_female --output singing

# Bark narrator
python3 videogen --model wan_14b_t2v --prompt "audiobook narrator" \
  --generate_audio --audio_type tts \
  --audio_text "Chapter one: The beginning of our journey" \
  --tts_voice bark_narrator --output audiobook
```

### Music Generation
```bash
# Epic orchestral music
python3 videogen --model wan_14b_t2v --prompt "epic battle scene" \
  --generate_audio --audio_type music \
  --audio_text "epic orchestral battle music with drums and brass, cinematic" \
  --music_model medium --output battle

# Ambient music
python3 videogen --model wan_14b_t2v --prompt "peaceful nature" \
  --generate_audio --audio_type music \
  --audio_text "calm ambient music with soft piano and nature sounds" \
  --music_model small --output nature

# Electronic music
python3 videogen --model wan_14b_t2v --prompt "cyberpunk city" \
  --generate_audio --audio_type music \
  --audio_text "dark electronic synthwave, futuristic beats" \
  --music_model medium --output cyberpunk

# High quality music (large model)
python3 videogen --model wan_14b_t2v --prompt "music video" \
  --generate_audio --audio_type music \
  --audio_text "upbeat pop music with catchy melody" \
  --music_model large --output pop_music
```

### Audio Sync Options
```bash
# Stretch audio to match video
python3 videogen --model wan_14b_t2v --prompt "sunset timelapse" \
  --generate_audio --audio_type music \
  --audio_text "peaceful ambient music" \
  --sync_audio --sync_mode stretch --output timelapse

# Trim audio to video length
python3 videogen --model wan_14b_t2v --prompt "short clip" \
  --length 3 --generate_audio --audio_type music \
  --audio_text "energetic rock music" \
  --sync_audio --sync_mode trim --output short_clip

# Loop audio for longer video
python3 videogen --model wan_14b_t2v --prompt "long sequence" \
  --length 30 --generate_audio --audio_type music \
  --audio_text "short drum beat" \
  --sync_audio --sync_mode loop --output long_video

# Pad with silence
python3 videogen --model wan_14b_t2v --prompt "documentary" \
  --length 15 --generate_audio --audio_type tts \
  --audio_text "Short narration" \
  --sync_audio --sync_mode pad --output doc
```

### Using External Audio File
```bash
# Use existing audio file
python3 videogen --model wan_14b_t2v --prompt "music video" \
  --audio_file my_song.mp3 \
  --sync_audio --output music_video

# External audio with lip sync
python3 videogen --image_to_video --model svd_xt_1.1 \
  --image_model flux_dev --prompt "person singing" \
  --audio_file vocals.wav \
  --lip_sync --output singing
```

---

## Lip Sync

### Basic Lip Sync
```bash
# TTS with lip sync
python3 videogen --image_to_video --model svd_xt_1.1 \
  --image_model flux_dev --prompt "news anchor speaking" \
  --generate_audio --audio_type tts \
  --audio_text "Breaking news tonight at eleven" \
  --tts_voice edge_male_us \
  --lip_sync --output news

# Lip sync with female voice
python3 videogen --image_to_video --model svd_xt_1.1 \
  --image_model flux_dev --prompt "woman giving speech" \
  --generate_audio --audio_type tts \
  --audio_text "Thank you all for coming today" \
  --tts_voice edge_female_us \
  --lip_sync --output speech
```

### Lip Sync with Specific Method
```bash
# Force Wav2Lip
python3 videogen --image_to_video --model svd_xt_1.1 \
  --image_model flux_dev --prompt "interview subject" \
  --generate_audio --audio_type tts \
  --audio_text "I've been working on this project for years" \
  --lip_sync --lip_sync_method wav2lip --output interview

# Force SadTalker
python3 videogen --image_to_video --model svd_xt_1.1 \
  --image_model flux_dev --prompt "talking head" \
  --generate_audio --audio_type tts \
  --audio_text "Welcome to my tutorial" \
  --lip_sync --lip_sync_method sadtalker --output tutorial
```

### Complete Lip Sync Pipeline
```bash
# Full pipeline: I2V + TTS + Sync + Lip Sync
python3 videogen --image_to_video --model svd_xt_1.1 \
  --image_model flux_dev \
  --prompt "professional presenter" \
  --prompt_animation "natural speaking gestures" \
  --generate_audio --audio_type tts \
  --audio_text "Hello and welcome to our comprehensive guide on video generation" \
  --tts_voice edge_male_us \
  --sync_audio \
  --lip_sync \
  --output presenter

# NSFW content with lip sync
python3 videogen --image_to_video --model svd_xt_1.1 \
  --image_model flux_nsfw \
  --prompt "intimate scene" \
  --prompt_animation "passionate movement" \
  --generate_audio --audio_type tts \
  --audio_text "I've been waiting for you" \
  --tts_voice edge_female_us \
  --lip_sync \
  --no_filter \
  --output intimate
```

---

## Distributed Multi-GPU

### Basic Distributed Setup
```bash
# Distribute across multiple GPUs
python3 videogen --model wan_14b_t2v \
  --prompt "epic space battle with thousands of ships" \
  --length 20 --distribute --output space_battle

# Distributed with specific network interface
python3 videogen --model mochi_1_preview \
  --prompt "complex nature scene" \
  --length 15 --distribute --interface eth0 --output nature

# Distributed with VRAM limit per GPU
python3 videogen --model hunyuanvideo \
  --prompt "cinematic masterpiece" \
  --length 30 --distribute --vram_limit 20 --output masterpiece
```

### Distributed with Large Models
```bash
# Open-Sora distributed
python3 videogen --model open_sora_1_2 \
  --prompt "detailed fantasy world" \
  --length 60 --width 1920 --height 1080 \
  --distribute --vram_limit 30 --output fantasy_world

# Step-Video distributed (extreme VRAM)
python3 videogen --model step_video_t2v \
  --prompt "feature film quality scene" \
  --length 120 \
  --distribute --vram_limit 40 --offload_strategy sequential \
  --output feature_film
```

### Distributed I2V
```bash
# Distributed I2V generation
python3 videogen --image_to_video --model svd_xt_1.1 \
  --image_model flux_dev \
  --prompt "complex animated scene" \
  --prompt_animation "multiple characters moving" \
  --length 20 --distribute --output complex_i2v
```

---

## Model Management

### Listing Models
```bash
# List all models
python3 videogen --model-list

# List only I2V models
python3 videogen --model-list --i2v-only

# List only T2V models
python3 videogen --model-list --t2v-only

# List NSFW-friendly models
python3 videogen --model-list --nsfw-friendly

# List low VRAM models (≤16GB)
python3 videogen --model-list --low-vram

# List high VRAM models (>30GB)
python3 videogen --model-list --high-vram

# List extreme VRAM models (>55GB)
python3 videogen --model-list --huge-vram

# Combine filters
python3 videogen --model-list --i2v-only --low-vram
python3 videogen --model-list --nsfw-friendly --high-vram
```

### Model Details
```bash
# Show details by numeric ID
python3 videogen --show-model 1

# Show details by name
python3 videogen --show-model wan_14b_t2v
python3 videogen --show-model svd_xt_1.1
```

### Searching for Models
```bash
# Search HuggingFace for models
python3 videogen --search-models "video generation"
python3 videogen --search-models "stable video diffusion"
python3 videogen --search-models "nsfw video"
python3 videogen --search-models "anime"

# Search with limit
python3 videogen --search-models "flux" --search-limit 50
```

### Adding Models
```bash
# Add model by ID
python3 videogen --add-model stabilityai/stable-video-diffusion-img2vid-xt-1.1 \
  --name svd_xt_1.1

# Add model by URL
python3 videogen --add-model "https://huggingface.co/stabilityai/stable-video-diffusion-img2vid-xt-1.1" \
  --name svd_xt_1.1

# Add model (name auto-generated)
python3 videogen --add-model Wan-AI/Wan2.1-T2V-14B-Diffusers

# Validate model exists
python3 videogen --validate-model stabilityai/stable-video-diffusion-img2vid-xt-1.1
```

### Updating Model Database
```bash
# Update from HuggingFace
python3 videogen --update-models

# Update with HuggingFace token (for gated models)
export HF_TOKEN=your_token_here
python3 videogen --update-models
```

### Listing TTS Voices
```bash
# List available TTS voices
python3 videogen --tts-list
```

---

## VRAM Management

### Offloading Strategies
```bash
# No offloading (fastest, needs most VRAM)
python3 videogen --model wan_14b_t2v --prompt "test" \
  --offload_strategy none

# Model offloading (balanced)
python3 videogen --model wan_14b_t2v --prompt "test" \
  --offload_strategy model

# Sequential offloading (slower, less VRAM)
python3 videogen --model wan_14b_t2v --prompt "test" \
  --offload_strategy sequential

# Group offloading
python3 videogen --model wan_14b_t2v --prompt "test" \
  --offload_strategy group --offload_group_size 8

# Auto map (let accelerate decide)
python3 videogen --model wan_14b_t2v --prompt "test" \
  --offload_strategy auto_map
```

### VRAM Limits
```bash
# Limit VRAM usage
python3 videogen --model wan_14b_t2v --prompt "test" \
  --vram_limit 16

# Very limited VRAM
python3 videogen --model wan_1.3b_t2v --prompt "test" \
  --vram_limit 8 --offload_strategy sequential

# High VRAM system
python3 videogen --model hunyuanvideo --prompt "test" \
  --vram_limit 48
```

### Low RAM Mode
```bash
# Enable low CPU RAM mode
python3 videogen --model wan_14b_t2v --prompt "test" \
  --low_ram_mode

# Low RAM with offloading
python3 videogen --model wan_14b_t2v --prompt "test" \
  --low_ram_mode --offload_strategy sequential

# Offload to disk
python3 videogen --model wan_14b_t2v --prompt "test" \
  --low_ram_mode --offload_dir ./offload_cache
```

### System RAM Limit
```bash
# Limit system RAM usage
python3 videogen --model wan_14b_t2v --prompt "test" \
  --system_ram_limit 32
```

---

## Upscaling

### Basic Upscaling
```bash
# 2x upscale
python3 videogen --model wan_14b_t2v --prompt "beautiful scene" \
  --upscale --output upscaled

# 4x upscale
python3 videogen --model wan_14b_t2v --prompt "detailed landscape" \
  --upscale --upscale_factor 4.0 --output high_res

# 1.5x upscale
python3 videogen --model wan_14b_t2v --prompt "portrait" \
  --upscale --upscale_factor 1.5 --output medium_res
```

### Upscaling with I2V
```bash
# I2V with upscale
python3 videogen --image_to_video --model svd_xt_1.1 \
  --image_model flux_dev --prompt "cinematic scene" \
  --prompt_animation "slow camera movement" \
  --upscale --upscale_factor 2.0 --output cinematic_hd
```

### Upscaling Considerations
```bash
# Upscaling takes significant time, use for final output
# Original: 832x480 -> Upscaled 2x: 1664x960
python3 videogen --model wan_14b_t2v \
  --prompt "high quality nature footage" \
  --width 832 --height 480 \
  --upscale --upscale_factor 2.0 \
  --output nature_hd

# For 4K output (from 1080p base)
python3 videogen --model wan_14b_t2v \
  --prompt "4K ready content" \
  --width 1920 --height 1080 \
  --upscale --upscale_factor 2.0 \
  --output 4k_output
```

---

## NSFW Content

### Basic NSFW Generation
```bash
# NSFW T2V
python3 videogen --model wan_14b_t2v \
  --prompt "intimate romantic scene" \
  --no_filter --output romantic

# NSFW I2V
python3 videogen --image_to_video --model svd_xt_1.1 \
  --image_model flux_nsfw \
  --prompt "adult content scene" \
  --prompt_animation "passionate movement" \
  --no_filter --output adult
```

### NSFW with Specific Models
```bash
# Pony NSFW
python3 videogen --model pony_v6 \
  --prompt "explicit anime content" \
  --no_filter --output anime_adult.png

# Flux NSFW
python3 videogen --model flux_nsfw \
  --prompt "adult photography style" \
  --no_filter --output adult.png

# SDXL NSFW
python3 videogen --model sdxl_nsfw \
  --prompt "explicit content" \
  --no_filter --output explicit.png
```

### NSFW with Audio
```bash
# NSFW with TTS
python3 videogen --image_to_video --model svd_xt_1.1 \
  --image_model flux_nsfw \
  --prompt "intimate scene" \
  --generate_audio --audio_type tts \
  --audio_text "I want you so bad" \
  --tts_voice edge_female_us \
  --no_filter --output intimate_audio

# NSFW with lip sync
python3 videogen --image_to_video --model svd_xt_1.1 \
  --image_model flux_nsfw \
  --prompt "adult scene with dialogue" \
  --generate_audio --audio_type tts \
  --audio_text "Explicit dialogue here" \
  --lip_sync \
  --no_filter --output adult_lipsync
```

### Auto Mode NSFW Detection
```bash
# Auto mode will detect NSFW and enable --no_filter automatically
python3 videogen --auto --prompt "explicit adult scene with nudity"

# Auto mode with NSFW I2V
python3 videogen --auto --prompt "passionate intimate scene between lovers"
```

---

## Advanced Combinations

### Complete Production Pipeline
```bash
# Full pipeline: I2V + Audio + Sync + Lip Sync + Upscale
python3 videogen --image_to_video --model svd_xt_1.1 \
  --image_model flux_dev \
  --prompt "professional news anchor" \
  --prompt_animation "natural speaking gestures and expressions" \
  --width 1280 --height 720 \
  --length 15 --fps 24 \
  --generate_audio --audio_type tts \
  --audio_text "Good evening, I'm your host for tonight's broadcast. We have some breaking news to share with you." \
  --tts_voice edge_male_us \
  --sync_audio \
  --lip_sync \
  --upscale --upscale_factor 2.0 \
  --output news_broadcast
```

### Multi-Stage Workflow
```bash
# Stage 1: Generate base image
python3 videogen --model flux_dev \
  --prompt "cinematic portrait of a woman, dramatic lighting" \
  --width 832 --height 480 \
  --output stage1_image.png

# Stage 2: Modify with I2I
python3 videogen --model flux_dev --image-to-image \
  --image stage1_image.png \
  --prompt "add rain effects, wet skin" \
  --strength 0.4 \
  --output stage2_modified.png

# Stage 3: Animate with I2V
python3 videogen --model svd_xt_1.1 \
  --image stage2_modified.png \
  --prompt "rain falling, subtle head movement" \
  --output stage3_animated

# Stage 4: Add audio
python3 videogen --model svd_xt_1.1 \
  --image stage2_modified.png \
  --prompt "rain falling, subtle head movement" \
  --generate_audio --audio_type music \
  --audio_text "melancholic piano with rain sounds" \
  --sync_audio \
  --output stage4_with_audio
```

### Batch Processing Script
```bash
#!/bin/bash
# batch_generate.sh

PROMPTS=(
  "sunset over ocean"
  "mountain landscape"
  "city street at night"
  "forest in autumn"
  "desert dunes"
)

MODELS=(
  "wan_1.3b_t2v"
  "wan_14b_t2v"
)

for prompt in "${PROMPTS[@]}"; do
  for model in "${MODELS[@]}"; do
    safe_name=$(echo "$prompt" | tr ' ' '_')
    python3 videogen --model "$model" \
      --prompt "$prompt" \
      --length 5 \
      --output "${model}_${safe_name}"
  done
done
```

### Comparison Generation
```bash
# Generate same prompt with different models
python3 videogen --model wan_1.3b_t2v --prompt "epic battle" --output battle_small
python3 videogen --model wan_14b_t2v --prompt "epic battle" --output battle_medium
python3 videogen --model mochi_1_preview --prompt "epic battle" --output battle_large
python3 videogen --model hunyuanvideo --prompt "epic battle" --output battle_huge
```

### Resolution Testing
```bash
# Test different resolutions
for res in "640x360" "832x480" "1280x720" "1920x1080"; do
  w=$(echo $res | cut -dx -f1)
  h=$(echo $res | cut -dx -f2)
  python3 videogen --model wan_14b_t2v \
    --prompt "test scene" \
    --width $w --height $h \
    --output "test_${w}x${h}"
done
```

### Seed Exploration
```bash
# Generate multiple variations with different seeds
for seed in 1 42 123 456 789; do
  python3 videogen --model wan_14b_t2v \
    --prompt "beautiful woman portrait" \
    --seed $seed \
    --output "portrait_seed_$seed"
done
```

### Style Variations
```bash
# Same subject, different styles
STYLES=(
  "photorealistic"
  "oil painting"
  "watercolor"
  "anime"
  "cyberpunk"
  "fantasy art"
)

for style in "${STYLES[@]}"; do
  python3 videogen --model wan_14b_t2v \
    --prompt "woman in red dress, $style style" \
    --output "woman_$style"
done
```

---

## Quick Reference

### Common Flags
| Flag | Description | Example |
|------|-------------|---------|
| `--model` | Model to use | `--model wan_14b_t2v` |
| `--prompt` | Main prompt | `--prompt "a cat"` |
| `--length` | Duration in seconds | `--length 10` |
| `--width` | Video width | `--width 1280` |
| `--height` | Video height | `--height 720` |
| `--fps` | Frames per second | `--fps 24` |
| `--output` | Output filename | `--output my_video` |
| `--seed` | Random seed | `--seed 42` |
| `--no_filter` | Disable NSFW filter | `--no_filter` |
| `--upscale` | Enable upscaling | `--upscale` |

### Audio Flags
| Flag | Description | Example |
|------|-------------|---------|
| `--generate_audio` | Enable audio | `--generate_audio` |
| `--audio_type` | tts or music | `--audio_type tts` |
| `--audio_text` | Audio prompt | `--audio_text "Hello"` |
| `--tts_voice` | TTS voice | `--tts_voice edge_male_us` |
| `--sync_audio` | Sync to video | `--sync_audio` |
| `--lip_sync` | Enable lip sync | `--lip_sync` |

### Model Management
| Flag | Description | Example |
|------|-------------|---------|
| `--model-list` | List models | `--model-list` |
| `--show-model` | Show details | `--show-model 1` |
| `--search-models` | Search HF | `--search-models "video"` |
| `--add-model` | Add model | `--add-model org/model` |
| `--update-models` | Update database | `--update-models` |

### VRAM Management
| Flag | Description | Example |
|------|-------------|---------|
| `--vram_limit` | GPU VRAM limit | `--vram_limit 20` |
| `--offload_strategy` | Offload mode | `--offload_strategy sequential` |
| `--low_ram_mode` | Low CPU RAM | `--low_ram_mode` |

---

## Troubleshooting

### Out of Memory
```bash
# Reduce resolution
python3 videogen --model wan_14b_t2v --prompt "test" \
  --width 640 --height 360

# Use smaller model
python3 videogen --model wan_1.3b_t2v --prompt "test"

# Enable aggressive offloading
python3 videogen --model wan_14b_t2v --prompt "test" \
  --offload_strategy sequential --low_ram_mode

# Limit VRAM
python3 videogen --model wan_14b_t2v --prompt "test" \
  --vram_limit 16
```

### Slow Generation
```bash
# Use smaller model
python3 videogen --model wan_1.3b_t2v --prompt "test"

# Reduce length
python3 videogen --model wan_14b_t2v --prompt "test" --length 3

# Lower FPS
python3 videogen --model wan_14b_t2v --prompt "test" --fps 12

# Skip upscaling
python3 videogen --model wan_14b_t2v --prompt "test"
# (don't add --upscale)
```

### Model Not Found
```bash
# Update model database
python3 videogen --update-models

# Search for model
python3 videogen --search-models "model name"

# Add manually
python3 videogen --add-model "org/model-name" --name my_model
```

### Audio Issues
```bash
# Check audio dependencies
pip install scipy soundfile librosa edge-tts
pip install git+https://github.com/suno-ai/bark.git
pip install audiocraft

# Use simpler TTS
python3 videogen --model wan_14b_t2v --prompt "test" \
  --generate_audio --audio_type tts \
  --tts_voice edge_female_us  # Edge-TTS is most reliable
```

---

## Environment Variables

```bash
# Set HuggingFace token for gated models
export HF_TOKEN=your_token_here

# Set custom cache directory
export HF_HOME=/path/to/cache

# Set CUDA device
export CUDA_VISIBLE_DEVICES=0,1
```

---

## Configuration Files

Models are stored in `~/.config/videogen/models.json`

You can edit this file directly to:
- Add custom models
- Modify VRAM estimates
- Change default settings
- Add model aliases