// VideoGen Web Interface - JavaScript Application

// Global state
let socket = null;
let currentJobId = null;
let models = [];
let ttsVoices = [];
let languages = [];
let currentMode = 't2v';

// Initialize application
document.addEventListener('DOMContentLoaded', () => {
    initializeSocket();
    loadModels();
    loadTTSVoices();
    loadLanguages();
    setupEventListeners();
    loadJobs();
    refreshGallery();
    
    // Set API endpoint display
    document.getElementById('api-endpoint').value = window.location.origin;
});

// Socket.IO initialization
function initializeSocket() {
    socket = io();
    
    socket.on('connect', () => {
        console.log('Connected to server');
        showToast('Connected to server', 'success');
    });
    
    socket.on('disconnect', () => {
        console.log('Disconnected from server');
        showToast('Disconnected from server', 'error');
    });
    
    socket.on('job_update', (job) => {
        updateJobInList(job);
        updateProgressModal(job);
        updateJobsBadge();
    });
    
    socket.on('job_log', (data) => {
        appendLog(data.line);
    });
}

// Load models from API
async function loadModels() {
    try {
        const response = await fetch('/api/models');
        models = await response.json();
        
        const modelSelect = document.getElementById('model');
        const imageModelSelect = document.getElementById('image_model');
        
        modelSelect.innerHTML = '<option value="">Select a model...</option>';
        imageModelSelect.innerHTML = '<option value="">Select a model...</option>';
        
        models.forEach(model => {
            const option = document.createElement('option');
            option.value = model.name;
            option.textContent = `${model.name} (${model.type || 'video'})`;
            option.dataset.type = model.type;
            modelSelect.appendChild(option);
            
            // Add image models to image model select
            if (model.type === 'image' || model.type === 't2i') {
                const imgOption = option.cloneNode(true);
                imageModelSelect.appendChild(imgOption);
            }
        });
    } catch (error) {
        console.error('Error loading models:', error);
        showToast('Failed to load models', 'error');
    }
}

// Load TTS voices
async function loadTTSVoices() {
    try {
        const response = await fetch('/api/tts-voices');
        ttsVoices = await response.json();
        
        const select = document.getElementById('tts_voice');
        select.innerHTML = '<option value="">Select a voice...</option>';
        
        ttsVoices.forEach(voice => {
            const option = document.createElement('option');
            option.value = voice.id;
            option.textContent = voice.name;
            select.appendChild(option);
        });
    } catch (error) {
        console.error('Error loading TTS voices:', error);
    }
}

// Load languages
async function loadLanguages() {
    try {
        const response = await fetch('/api/languages');
        languages = await response.json();
        
        const sourceSelect = document.getElementById('source_lang');
        const targetSelect = document.getElementById('target_lang');
        
        languages.forEach(lang => {
            const sourceOption = document.createElement('option');
            sourceOption.value = lang.code;
            sourceOption.textContent = lang.name;
            sourceSelect.appendChild(sourceOption);
            
            const targetOption = sourceOption.cloneNode(true);
            targetSelect.appendChild(targetOption);
        });
    } catch (error) {
        console.error('Error loading languages:', error);
    }
}

// Setup event listeners
function setupEventListeners() {
    // Tab navigation
    document.querySelectorAll('.nav-btn').forEach(btn => {
        btn.addEventListener('click', () => switchTab(btn.dataset.tab));
    });
    
    // Mode selection
    document.querySelectorAll('.mode-btn').forEach(btn => {
        btn.addEventListener('click', () => switchMode(btn.dataset.mode));
    });
    
    // Form submission
    document.getElementById('generate-form').addEventListener('submit', handleGenerate);
    
    // Strength slider
    document.getElementById('strength').addEventListener('input', (e) => {
        document.getElementById('strength-value').textContent = e.target.value;
    });
}

// Switch tab
function switchTab(tabName) {
    document.querySelectorAll('.nav-btn').forEach(btn => {
        btn.classList.toggle('active', btn.dataset.tab === tabName);
    });
    
    document.querySelectorAll('.tab').forEach(tab => {
        tab.classList.toggle('active', tab.id === `tab-${tabName}`);
    });
    
    if (tabName === 'jobs') {
        loadJobs();
    } else if (tabName === 'gallery') {
        refreshGallery();
    }
}

// Switch generation mode
function switchMode(mode) {
    currentMode = mode;
    
    document.querySelectorAll('.mode-btn').forEach(btn => {
        btn.classList.toggle('active', btn.dataset.mode === mode);
    });
    
    // Show/hide relevant sections
    const i2vPrompts = document.getElementById('i2v-prompts');
    const imageModelGroup = document.getElementById('image-model-group');
    const imageUploadBox = document.getElementById('image-upload-box');
    const videoUploadBox = document.getElementById('video-upload-box');
    const audioUploadBox = document.getElementById('audio-upload-box');
    const audioSection = document.getElementById('audio-section');
    const dubbingSection = document.getElementById('dubbing-section');
    const subtitleSection = document.getElementById('subtitle-section');
    const strengthGroup = document.getElementById('strength-group');
    
    // Reset all sections
    i2vPrompts.classList.add('hidden');
    imageModelGroup.classList.add('hidden');
    imageUploadBox.classList.remove('hidden');
    videoUploadBox.classList.add('hidden');
    audioUploadBox.classList.add('hidden');
    audioSection.classList.remove('hidden');
    dubbingSection.classList.add('hidden');
    subtitleSection.classList.add('hidden');
    strengthGroup.classList.add('hidden');
    
    // Configure for each mode
    switch (mode) {
        case 't2v':
            imageUploadBox.classList.add('hidden');
            break;
            
        case 'i2v':
            i2vPrompts.classList.remove('hidden');
            imageModelGroup.classList.remove('hidden');
            imageUploadBox.classList.add('hidden'); // Will generate image first
            break;
            
        case 't2i':
            imageUploadBox.classList.add('hidden');
            audioSection.classList.add('hidden');
            break;
            
        case 'i2i':
            strengthGroup.classList.remove('hidden');
            break;
            
        case 'v2v':
            imageUploadBox.classList.add('hidden');
            videoUploadBox.classList.remove('hidden');
            strengthGroup.classList.remove('hidden');
            break;
            
        case 'dub':
            imageUploadBox.classList.add('hidden');
            videoUploadBox.classList.remove('hidden');
            audioSection.classList.add('hidden');
            dubbingSection.classList.remove('hidden');
            break;
            
        case 'subtitles':
            imageUploadBox.classList.add('hidden');
            videoUploadBox.classList.remove('hidden');
            audioSection.classList.add('hidden');
            subtitleSection.classList.remove('hidden');
            break;
            
        case 'upscale':
            imageUploadBox.classList.add('hidden');
            videoUploadBox.classList.remove('hidden');
            audioSection.classList.add('hidden');
            break;
    }
}

// Handle file upload
async function handleFileUpload(input, type) {
    const file = input.files[0];
    if (!file) return;
    
    const formData = new FormData();
    formData.append('file', file);
    formData.append('type', type);
    
    try {
        const response = await fetch('/api/upload', {
            method: 'POST',
            body: formData
        });
        
        const data = await response.json();
        
        if (response.ok) {
            // Update the hidden input with the file path
            const hiddenInput = document.getElementById(`input_${type}`);
            if (hiddenInput) {
                hiddenInput.value = data.path;
            }
            
            // Update the file name display
            const fileNameDisplay = document.getElementById(`${type}-file-name`);
            if (fileNameDisplay) {
                fileNameDisplay.textContent = file.name;
            }
            
            showToast(`File uploaded: ${file.name}`, 'success');
        } else {
            showToast(`Upload failed: ${data.error}`, 'error');
        }
    } catch (error) {
        console.error('Upload error:', error);
        showToast('Upload failed', 'error');
    }
}

// Handle form submission
async function handleGenerate(e) {
    e.preventDefault();
    
    const form = e.target;
    const formData = new FormData(form);
    const params = {};
    
    // Convert FormData to object
    for (let [key, value] of formData.entries()) {
        if (value !== '' && value !== null) {
            params[key] = value;
        }
    }
    
    // Add mode
    params.mode = currentMode;
    
    // Handle checkboxes
    params.auto = form.querySelector('#auto')?.checked || false;
    params.generate_audio = form.querySelector('#generate_audio')?.checked || false;
    params.sync_audio = form.querySelector('#sync_audio')?.checked || false;
    params.lip_sync = form.querySelector('#lip_sync')?.checked || false;
    params.no_filter = form.querySelector('#no_filter')?.checked || false;
    params.debug = form.querySelector('#debug')?.checked || false;
    params.voice_clone = form.querySelector('#voice_clone')?.checked || false;
    params.create_subtitles = form.querySelector('#create_subtitles')?.checked || false;
    params.translate_subtitles = form.querySelector('#translate_subtitles')?.checked || false;
    params.burn_subtitles = form.querySelector('#burn_subtitles')?.checked || false;
    
    // Convert numeric values
    params.width = parseInt(params.width) || 832;
    params.height = parseInt(params.height) || 480;
    params.fps = parseInt(params.fps) || 15;
    params.length = parseFloat(params.length) || 5;
    params.seed = parseInt(params.seed) || -1;
    params.vram_limit = parseInt(params.vram_limit) || 22;
    params.strength = parseFloat(params.strength) || 0.7;
    
    // Mode-specific settings
    if (currentMode === 'i2i' || currentMode === 'v2v') {
        params.image_to_image = currentMode === 'i2i';
        params.video_to_video = currentMode === 'v2v';
    }
    
    if (currentMode === 't2i') {
        params.generate_image = true;
    }
    
    if (currentMode === 'dub') {
        params.dub_video = true;
    }
    
    if (currentMode === 'upscale') {
        params.upscale = true;
    }
    
    try {
        const response = await fetch('/api/jobs', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json'
            },
            body: JSON.stringify(params)
        });
        
        const job = await response.json();
        
        if (response.ok) {
            currentJobId = job.id;
            socket.emit('subscribe_job', job.id);
            showProgressModal(job);
            showToast('Generation started', 'info');
            loadJobs();
        } else {
            showToast(`Failed to start job: ${job.error}`, 'error');
        }
    } catch (error) {
        console.error('Error starting job:', error);
        showToast('Failed to start generation', 'error');
    }
}

// Load jobs
async function loadJobs() {
    try {
        const response = await fetch('/api/jobs');
        const jobs = await response.json();
        
        const jobsList = document.getElementById('jobs-list');
        
        if (jobs.length === 0) {
            jobsList.innerHTML = `
                <div class="empty-state">
                    <i class="fas fa-inbox"></i>
                    <p>No jobs yet. Start generating!</p>
                </div>
            `;
            return;
        }
        
        jobsList.innerHTML = jobs.map(job => createJobCard(job)).join('');
        updateJobsBadge();
    } catch (error) {
        console.error('Error loading jobs:', error);
    }
}

// Create job card HTML
function createJobCard(job) {
    const statusIcons = {
        pending: 'fa-clock',
        running: 'fa-spinner fa-spin',
        completed: 'fa-check',
        failed: 'fa-times',
        cancelled: 'fa-ban'
    };
    
    const createdDate = new Date(job.created_at).toLocaleString();
    
    let actionsHtml = '';
    if (job.status === 'running') {
        actionsHtml = `<button class="btn btn-small btn-danger" onclick="cancelJob('${job.id}')">
            <i class="fas fa-stop"></i> Cancel
        </button>`;
    } else if (job.status === 'failed' || job.status === 'cancelled') {
        actionsHtml = `<button class="btn btn-small btn-primary" onclick="retryJob('${job.id}')">
            <i class="fas fa-redo"></i> Retry
        </button>`;
    }
    
    if (job.output_files && job.output_files.length > 0) {
        actionsHtml += job.output_files.map(file => `
            <a href="/api/download/${file}" class="btn btn-small btn-success" download>
                <i class="fas fa-download"></i> ${file}
            </a>
        `).join('');
    }
    
    return `
        <div class="job-card" id="job-${job.id}">
            <div class="job-header">
                <div>
                    <span class="job-id">Job #${job.id}</span>
                    <p style="color: var(--text-muted); font-size: 0.8rem;">${createdDate}</p>
                </div>
                <span class="job-status ${job.status}">
                    <i class="fas ${statusIcons[job.status]}"></i>
                    ${job.status.toUpperCase()}
                </span>
            </div>
            <div class="job-command">${escapeHtml(job.command)}</div>
            ${job.status === 'running' ? `
                <div class="job-progress">
                    <div class="job-progress-bar">
                        <div class="job-progress-fill" style="width: ${job.progress}%"></div>
                    </div>
                    <div class="job-progress-text">${escapeHtml(job.progress_text || 'Processing...')}</div>
                </div>
            ` : ''}
            ${job.error ? `<p style="color: var(--danger); font-size: 0.85rem;"><i class="fas fa-exclamation-triangle"></i> ${escapeHtml(job.error)}</p>` : ''}
            <div class="job-actions">
                ${actionsHtml}
                <button class="btn btn-small btn-secondary" onclick="deleteJob('${job.id}')">
                    <i class="fas fa-trash"></i>
                </button>
            </div>
        </div>
    `;
}

// Update job in list
function updateJobInList(job) {
    const existingCard = document.getElementById(`job-${job.id}`);
    
    if (existingCard) {
        const newCard = document.createElement('div');
        newCard.innerHTML = createJobCard(job);
        existingCard.replaceWith(newCard.firstElementChild);
    } else {
        loadJobs();
    }
}

// Update jobs badge
function updateJobsBadge() {
    const runningJobs = Object.values(jobs || {}).filter(j => j.status === 'running' || j.status === 'pending').length;
    document.getElementById('jobs-badge').textContent = runningJobs;
}

// Cancel job
async function cancelJob(jobId) {
    if (!jobId) jobId = currentJobId;
    if (!jobId) return;
    
    try {
        const response = await fetch(`/api/jobs/${jobId}/cancel`, {
            method: 'POST'
        });
        
        if (response.ok) {
            showToast('Job cancelled', 'warning');
            closeProgressModal();
            loadJobs();
        }
    } catch (error) {
        console.error('Error cancelling job:', error);
    }
}

// Retry job
async function retryJob(jobId) {
    try {
        const response = await fetch(`/api/jobs/${jobId}/retry`, {
            method: 'POST'
        });
        
        const job = await response.json();
        
        if (response.ok) {
            currentJobId = job.id;
            socket.emit('subscribe_job', job.id);
            showProgressModal(job);
            showToast('Job restarted', 'info');
            loadJobs();
        } else {
            showToast(`Failed to retry: ${job.error}`, 'error');
        }
    } catch (error) {
        console.error('Error retrying job:', error);
    }
}

// Delete job
async function deleteJob(jobId) {
    if (!confirm('Delete this job?')) return;
    
    try {
        await fetch(`/api/jobs/${jobId}`, {
            method: 'DELETE'
        });
        
        loadJobs();
        showToast('Job deleted', 'info');
    } catch (error) {
        console.error('Error deleting job:', error);
    }
}

// Clear completed jobs
async function clearCompletedJobs() {
    const jobs = await (await fetch('/api/jobs')).json();
    
    for (const job of jobs) {
        if (['completed', 'failed', 'cancelled'].includes(job.status)) {
            await fetch(`/api/jobs/${job.id}`, { method: 'DELETE' });
        }
    }
    
    loadJobs();
    showToast('Completed jobs cleared', 'info');
}

// Progress Modal
function showProgressModal(job) {
    const modal = document.getElementById('progress-modal');
    modal.classList.add('active');
    
    updateProgressModal(job);
}

function closeProgressModal() {
    const modal = document.getElementById('progress-modal');
    modal.classList.remove('active');
    currentJobId = null;
}

function minimizeModal() {
    closeProgressModal();
    switchTab('jobs');
}

function updateProgressModal(job) {
    if (!job) return;
    
    const progressFill = document.getElementById('progress-fill');
    const progressText = document.getElementById('progress-text');
    const progressPercent = document.getElementById('progress-percent');
    
    progressFill.style.width = `${job.progress}%`;
    progressText.textContent = job.progress_text || 'Processing...';
    progressPercent.textContent = `${Math.round(job.progress)}%`;
    
    // Update modal header based on status
    const modalHeader = document.querySelector('#progress-modal h3');
    if (job.status === 'completed') {
        modalHeader.innerHTML = '<i class="fas fa-check"></i> Completed!';
        progressFill.style.background = 'var(--success)';
    } else if (job.status === 'failed') {
        modalHeader.innerHTML = '<i class="fas fa-times"></i> Failed';
        progressFill.style.background = 'var(--danger)';
    } else if (job.status === 'cancelled') {
        modalHeader.innerHTML = '<i class="fas fa-ban"></i> Cancelled';
        progressFill.style.background = 'var(--secondary)';
    }
}

// Append log line
function appendLog(line) {
    const logContent = document.getElementById('log-content');
    const logLine = document.createElement('div');
    logLine.className = 'log-line';
    
    // Detect error/success lines
    if (line.includes('❌') || line.includes('Error') || line.includes('Failed')) {
        logLine.classList.add('error');
    } else if (line.includes('✅') || line.includes('Success') || line.includes('Done')) {
        logLine.classList.add('success');
    }
    
    logLine.textContent = line;
    logContent.appendChild(logLine);
    logContent.scrollTop = logContent.scrollHeight;
}

function toggleLogs() {
    const logContent = document.getElementById('log-content');
    logContent.classList.toggle('collapsed');
}

// Gallery
async function refreshGallery() {
    try {
        const response = await fetch('/api/outputs');
        const files = await response.json();
        
        const gallery = document.getElementById('gallery-grid');
        
        if (files.length === 0) {
            gallery.innerHTML = `
                <div class="empty-state">
                    <i class="fas fa-photo-video"></i>
                    <p>No generated files yet.</p>
                </div>
            `;
            return;
        }
        
        gallery.innerHTML = files.map(file => createGalleryItem(file)).join('');
    } catch (error) {
        console.error('Error loading gallery:', error);
    }
}

function createGalleryItem(file) {
    const ext = file.name.split('.').pop().toLowerCase();
    const isVideo = ['mp4', 'mov', 'webm', 'avi', 'mkv'].includes(ext);
    const isImage = ['png', 'jpg', 'jpeg', 'gif', 'webp'].includes(ext);
    const isAudio = ['mp3', 'wav', 'ogg', 'flac'].includes(ext);
    
    let preview = '';
    if (isVideo) {
        preview = `<video src="/api/download/${file.name}" muted></video>`;
    } else if (isImage) {
        preview = `<img src="/api/download/${file.name}" alt="${file.name}">`;
    } else if (isAudio) {
        preview = `<i class="fas fa-music"></i>`;
    } else {
        preview = `<i class="fas fa-file"></i>`;
    }
    
    const size = formatFileSize(file.size);
    const date = new Date(file.modified).toLocaleDateString();
    
    return `
        <div class="gallery-item">
            <div class="gallery-preview">
                ${preview}
            </div>
            <div class="gallery-info">
                <h4>${file.name}</h4>
                <p>${size} • ${date}</p>
                <div class="gallery-actions">
                    <a href="/api/download/${file.name}" class="btn btn-small btn-primary" download>
                        <i class="fas fa-download"></i> Download
                    </a>
                    <button class="btn btn-small btn-danger" onclick="deleteOutput('${file.name}')">
                        <i class="fas fa-trash"></i>
                    </button>
                </div>
            </div>
        </div>
    `;
}

async function deleteOutput(filename) {
    if (!confirm(`Delete ${filename}?`)) return;
    
    try {
        await fetch(`/api/outputs/${filename}`, {
            method: 'DELETE'
        });
        
        refreshGallery();
        showToast('File deleted', 'info');
    } catch (error) {
        console.error('Error deleting file:', error);
    }
}

// Toast notifications
function showToast(message, type = 'info') {
    const container = document.getElementById('toast-container');
    const toast = document.createElement('div');
    toast.className = `toast ${type}`;
    
    const icons = {
        success: 'fa-check-circle',
        error: 'fa-exclamation-circle',
        warning: 'fa-exclamation-triangle',
        info: 'fa-info-circle'
    };
    
    toast.innerHTML = `
        <i class="fas ${icons[type]}"></i>
        <span>${message}</span>
    `;
    
    container.appendChild(toast);
    
    setTimeout(() => {
        toast.remove();
    }, 5000);
}

// Utility functions
function formatFileSize(bytes) {
    if (bytes < 1024) return bytes + ' B';
    if (bytes < 1024 * 1024) return (bytes / 1024).toFixed(1) + ' KB';
    if (bytes < 1024 * 1024 * 1024) return (bytes / (1024 * 1024)).toFixed(1) + ' MB';
    return (bytes / (1024 * 1024 * 1024)).toFixed(1) + ' GB';
}

function escapeHtml(text) {
    const div = document.createElement('div');
    div.textContent = text;
    return div.innerHTML;
}

function addPromptHint(hint) {
    const prompt = document.getElementById('prompt');
    if (prompt.value && !prompt.value.endsWith(', ')) {
        prompt.value += ', ';
    }
    prompt.value += hint;
    prompt.focus();
}

function toggleSection(header) {
    const section = header.parentElement;
    section.classList.toggle('collapsed');
}

function toggleAudioOptions() {
    const audioOptions = document.getElementById('audio-options');
    const checkbox = document.getElementById('generate_audio');
    audioOptions.classList.toggle('hidden', !checkbox.checked);
}

function toggleAudioType() {
    const audioType = document.getElementById('audio_type').value;
    const ttsVoiceGroup = document.getElementById('tts-voice-group');
    const audioTextGroup = document.getElementById('audio-text-group');
    const musicPromptGroup = document.getElementById('music-prompt-group');
    
    if (audioType === 'tts') {
        ttsVoiceGroup.classList.remove('hidden');
        audioTextGroup.classList.remove('hidden');
        musicPromptGroup.classList.add('hidden');
    } else {
        ttsVoiceGroup.classList.add('hidden');
        audioTextGroup.classList.add('hidden');
        musicPromptGroup.classList.remove('hidden');
    }
}

function resetForm() {
    document.getElementById('generate-form').reset();
    document.getElementById('strength-value').textContent = '0.7';
    
    // Clear file uploads
    document.getElementById('input_image').value = '';
    document.getElementById('input_video').value = '';
    document.getElementById('input_audio').value = '';
    document.getElementById('image-file-name').textContent = 'PNG, JPG, WEBP';
    document.getElementById('video-file-name').textContent = 'MP4, MOV, WEBM';
    document.getElementById('audio-file-name').textContent = 'MP3, WAV, OGG';
    
    // Reset mode
    switchMode('t2v');
}

// Keyboard shortcuts
document.addEventListener('keydown', (e) => {
    // Escape to close modal
    if (e.key === 'Escape') {
        closeProgressModal();
    }
    
    // Ctrl+Enter to submit
    if (e.ctrlKey && e.key === 'Enter') {
        document.getElementById('generate-form').dispatchEvent(new Event('submit'));
    }
});