#!/usr/bin/env python3
"""Check available pipeline variants in diffusers library"""
import sys

def check_pipelines():
    """Check for pipeline variants in diffusers"""
    print("=" * 70)
    print("CHECKING AVAILABLE PIPELINE VARIANTS IN DIFFUSERS")
    print("=" * 70)
    
    try:
        import diffusers
    except ImportError:
        print("ERROR: diffusers not installed")
        print("Install with: pip install diffusers")
        sys.exit(1)
    
    # Get all pipeline-related names
    all_names = dir(diffusers)
    pipeline_names = [n for n in all_names if 'Pipeline' in n]
    
    print(f"\nFound {len(pipeline_names)} Pipeline classes\n")
    
    # Group by base model
    pipelines_by_base = {}
    
    for name in sorted(pipeline_names):
        # Categorize
        base = name.replace("Pipeline", "").replace("Img2Img", "_I2I").replace("ImageToVideo", "_I2V").replace("VideoToVideo", "_V2V").replace("TextToImage", "_T2I").replace("TextToVideo", "_T2V").replace("ImageToImage", "_I2I").replace("VideoImageToVideo", "_V2V")
        
        # Group by base
        for prefix in ["Wan", "Flux", "StableDiffusion", "SD3", "LTX", "Mochi", "CogVideo", "I2VGen", "AnimateDiff"]:
            if name.startswith(prefix):
                base = prefix
                break
        
        if base not in pipelines_by_base:
            pipelines_by_base[base] = []
        pipelines_by_base[base].append(name)
    
    # Print grouped results
    print("\n" + "=" * 70)
    print("PIPELINES GROUPED BY BASE MODEL")
    print("=" * 70)
    
    for base in sorted(pipelines_by_base.keys()):
        names = pipelines_by_base[base]
        if len(names) > 1:
            print(f"\n{base}:")
            for n in sorted(names):
                print(f"  - {n}")
    
    # Check specific models
    print("\n" + "=" * 70)
    print("CHECKING SPECIFIC PIPELINE VARIANTS")
    print("=" * 70)
    
    # Wan pipelines
    print("\n--- Wan ---")
    try:
        from diffusers import WanPipeline
        print(f"WanPipeline (T2V): {WanPipeline}")
    except Exception as e:
        print(f"WanPipeline (T2V): ERROR - {e}")
    
    try:
        from diffusers import WanImageToVideoPipeline
        print(f"WanImageToVideoPipeline (I2V): {WanImageToVideoPipeline}")
    except Exception as e:
        print(f"WanImageToVideoPipeline (I2V): ERROR - {e}")
    
    try:
        from diffusers import WanVideoToVideoPipeline
        print(f"WanVideoToVideoPipeline (V2V): {WanVideoToVideoPipeline}")
    except Exception as e:
        print(f"WanVideoToVideoPipeline (V2V): ERROR - {e}")
    
    # LTX pipelines
    print("\n--- LTX ---")
    try:
        from diffusers import LTXPipeline
        print(f"LTXPipeline (T2V): {LTXPipeline}")
    except Exception as e:
        print(f"LTXPipeline (T2V): ERROR - {e}")
    
    try:
        from diffusers import LTXImageToVideoPipeline
        print(f"LTXImageToVideoPipeline (I2V): {LTXImageToVideoPipeline}")
    except Exception as e:
        print(f"LTXImageToVideoPipeline (I2V): ERROR - {e}")
    
    # Flux pipelines
    print("\n--- Flux ---")
    try:
        from diffusers import FluxPipeline
        print(f"FluxPipeline (T2I): {FluxPipeline}")
    except Exception as e:
        print(f"FluxPipeline (T2I): ERROR - {e}")
    
    try:
        from diffusers import FluxImg2ImgPipeline
        print(f"FluxImg2ImgPipeline (I2I): {FluxImg2ImgPipeline}")
    except Exception as e:
        print(f"FluxImg2ImgPipeline (I2I): ERROR - {e}")
    
    # Stable Diffusion XL pipelines
    print("\n--- Stable Diffusion XL ---")
    try:
        from diffusers import StableDiffusionXLPipeline
        print(f"StableDiffusionXLPipeline (T2I): {StableDiffusionXLPipeline}")
    except Exception as e:
        print(f"StableDiffusionXLPipeline (T2I): ERROR - {e}")
    
    try:
        from diffusers import StableDiffusionXLImg2ImgPipeline
        print(f"StableDiffusionXLImg2ImgPipeline (I2I): {StableDiffusionXLImg2ImgPipeline}")
    except Exception as e:
        print(f"StableDiffusionXLImg2ImgPipeline (I2I): ERROR - {e}")
    
    # Stable Diffusion 3 pipelines
    print("\n--- Stable Diffusion 3 ---")
    try:
        from diffusers import StableDiffusion3Pipeline
        print(f"StableDiffusion3Pipeline (T2I): {StableDiffusion3Pipeline}")
    except Exception as e:
        print(f"StableDiffusion3Pipeline (T2I): ERROR - {e}")
    
    try:
        from diffusers import StableDiffusion3Img2ImgPipeline
        print(f"StableDiffusion3Img2ImgPipeline (I2I): {StableDiffusion3Img2ImgPipeline}")
    except Exception as e:
        print(f"StableDiffusion3Img2ImgPipeline (I2I): ERROR - {e}")
    
    # Stable Video Diffusion
    print("\n--- Stable Video Diffusion ---")
    try:
        from diffusers import StableVideoDiffusionPipeline
        print(f"StableVideoDiffusionPipeline (I2V): {StableVideoDiffusionPipeline}")
    except Exception as e:
        print(f"StableVideoDiffusionPipeline (I2V): ERROR - {e}")
    
    # CogVideoX
    print("\n--- CogVideoX ---")
    try:
        from diffusers import CogVideoXPipeline
        print(f"CogVideoXPipeline (T2V): {CogVideoXPipeline}")
    except Exception as e:
        print(f"CogVideoXPipeline (T2V): ERROR - {e}")
    
    try:
        from diffusers import CogVideoXImageToVideoPipeline
        print(f"CogVideoXImageToVideoPipeline (I2V): {CogVideoXImageToVideoPipeline}")
    except Exception as e:
        print(f"CogVideoXImageToVideoPipeline (I2V): ERROR - {e}")
    
    # I2VGenXL
    print("\n--- I2VGenXL ---")
    try:
        from diffusers import I2VGenXLPipeline
        print(f"I2VGenXLPipeline (I2V): {I2VGenXLPipeline}")
    except Exception as e:
        print(f"I2VGenXLPipeline (I2V): ERROR - {e}")
    
    # Mochi
    print("\n--- Mochi ---")
    try:
        from diffusers import MochiPipeline
        print(f"MochiPipeline (T2V): {MochiPipeline}")
    except Exception as e:
        print(f"MochiPipeline (T2V): ERROR - {e}")
    
    # AnimateDiff
    print("\n--- AnimateDiff ---")
    try:
        from diffusers import AnimateDiffPipeline
        print(f"AnimateDiffPipeline (T2V): {AnimateDiffPipeline}")
    except Exception as e:
        print(f"AnimateDiffPipeline (T2V): ERROR - {e}")
    
    print("\n" + "=" * 70)
    print("SUMMARY OF PIPELINE VARIANTS")
    print("=" * 70)
    print("""
Based on the check, here's the mapping of pipelines:

T2V (Text-to-Video):
  - WanPipeline
  - LTXPipeline  
  - CogVideoXPipeline
  - MochiPipeline
  - AnimateDiffPipeline

I2V (Image-to-Video):
  - WanImageToVideoPipeline
  - LTXImageToVideoPipeline
  - StableVideoDiffusionPipeline
  - CogVideoXImageToVideoPipeline
  - I2VGenXLPipeline

T2I (Text-to-Image):
  - FluxPipeline
  - StableDiffusionXLPipeline
  - StableDiffusion3Pipeline

I2I (Image-to-Image):
  - FluxImg2ImgPipeline
  - StableDiffusionXLImg2ImgPipeline
  - StableDiffusion3Img2ImgPipeline
""")
    print(f"\ndiffusers version: {diffusers.__version__}")

if __name__ == "__main__":
    check_pipelines()
