
#!/usr/bin/env python3
import sys
import os
import json
from pathlib import Path

def main():
    # Load models from JSON
    config_dir = Path.home() / ".config" / "videogen"
    models_config_file = config_dir / "models.json"
    
    if not models_config_file.exists():
        print(f"❌ Models config not found: {models_config_file}")
        return
    
    with open(models_config_file, 'r') as f:
        config = json.load(f)
    
    models = config.get("models", {})
    
    print(f"Loaded {len(models)} models from {models_config_file}")
    
    # Look for the specific model in question
    model_name = "goddess_of_realism_gor_pony_v2art_sdxl"
    model_info = models.get(model_name)
    
    if model_info:
        print(f"\n=== Model: {model_name} ===")
        print(f"  ID: {model_info.get('id')}")
        print(f"  Class: {model_info.get('class')}")
        print(f"  VRAM: {model_info.get('vram')}")
        print(f"  Supports I2V: {model_info.get('supports_i2v', False)}")
        print(f"  Tags: {model_info.get('tags', [])}")
        print(f"  Description: {model_info.get('desc')}")
    else:
        print(f"❌ Model not found: {model_name}")
    
    # Find all models that support I2V
    print("\n=== I2V-capable models ===")
    i2v_models = []
    for name, info in models.items():
        supports_i2v = info.get('supports_i2v', False)
        pipeline_class = info.get('class', '')
        
        # Check pipeline class for I2V support
        i2v_pipeline = any(keyword in pipeline_class.lower() for keyword in ['i2v', 'image_to_video', 'stablevideodiffusion', 'i2vgen', 'ltximage'])
        
        if supports_i2v or i2v_pipeline:
            i2v_models.append((name, info))
    
    print(f"Found {len(i2v_models)} I2V-capable models:")
    for i, (name, info) in enumerate(i2v_models[:10]):
        print(f"{i+1}. {name} ({info.get('class')})")
        print(f"    {info.get('id')}")
    
    # Look for the model in the results
    model_in_i2v = any(name == model_name for name, info in i2v_models)
    print(f"\nModel '{model_name}' in I2V list: {model_in_i2v}")

if __name__ == "__main__":
    main()
