/*
 * Network Monitoring Suite - Packet Capture
 * Copyright (C) 2024 Stefy Lanza <stefy@sexhack.me>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */

#include <winsock2.h>
#include <windows.h>
#include <iphlpapi.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>

// #pragma comment(lib, "ws2_32.lib")
// #pragma comment(lib, "iphlpapi.lib")

// PCAP file header structure
typedef struct {
    DWORD magic_number;   // 0xa1b2c3d4
    WORD version_major;   // 2
    WORD version_minor;   // 4
    DWORD thiszone;       // 0
    DWORD sigfigs;        // 0
    DWORD snaplen;        // 65535
    DWORD network;        // 1 (Ethernet)
} pcap_hdr_t;

// PCAP packet header structure
typedef struct {
    DWORD ts_sec;         // timestamp seconds
    DWORD ts_usec;        // timestamp microseconds
    DWORD incl_len;       // number of octets of packet saved in file
    DWORD orig_len;       // actual length of packet
} pcaprec_hdr_t;

// Ethernet header
typedef struct {
    BYTE dest_mac[6];
    BYTE src_mac[6];
    WORD ethertype;
} ethernet_hdr_t;

// IP header
typedef struct {
    BYTE version_ihl;
    BYTE tos;
    WORD total_length;
    WORD identification;
    WORD flags_offset;
    BYTE ttl;
    BYTE protocol;
    WORD checksum;
    DWORD src_ip;
    DWORD dest_ip;
} ip_hdr_t;

// TCP header
typedef struct {
    WORD src_port;
    WORD dest_port;
    DWORD seq_num;
    DWORD ack_num;
    BYTE data_offset;
    BYTE flags;
    WORD window;
    WORD checksum;
    WORD urgent_ptr;
} tcp_hdr_t;

// Function to create fake Ethernet/IP/TCP headers for Wireshark compatibility
void create_pcap_packet(BYTE* buffer, int* size, DWORD src_ip, DWORD dest_ip, WORD src_port, WORD dest_port, const BYTE* payload, int payload_len) {
    ethernet_hdr_t* eth = (ethernet_hdr_t*)buffer;
    ip_hdr_t* ip = (ip_hdr_t*)(buffer + sizeof(ethernet_hdr_t));
    tcp_hdr_t* tcp = (tcp_hdr_t*)(buffer + sizeof(ethernet_hdr_t) + sizeof(ip_hdr_t));

    // Fake MAC addresses
    memset(eth->dest_mac, 0xFF, 6); // broadcast
    memset(eth->src_mac, 0x00, 6);
    eth->ethertype = htons(0x0800); // IPv4

    // IP header
    ip->version_ihl = 0x45; // IPv4, 20 byte header
    ip->tos = 0;
    ip->total_length = htons(sizeof(ip_hdr_t) + sizeof(tcp_hdr_t) + payload_len);
    ip->identification = 0;
    ip->flags_offset = 0;
    ip->ttl = 64;
    ip->protocol = 6; // TCP
    ip->checksum = 0; // Skip checksum calculation for simplicity
    ip->src_ip = src_ip;
    ip->dest_ip = dest_ip;

    // TCP header
    tcp->src_port = src_port;
    tcp->dest_port = dest_port;
    tcp->seq_num = 0;
    tcp->ack_num = 0;
    tcp->data_offset = 0x50; // 20 byte header
    tcp->flags = 0x18; // PSH+ACK
    tcp->window = htons(8192);
    tcp->checksum = 0; // Skip checksum
    tcp->urgent_ptr = 0;

    // Copy payload
    memcpy(buffer + sizeof(ethernet_hdr_t) + sizeof(ip_hdr_t) + sizeof(tcp_hdr_t), payload, payload_len);

    *size = sizeof(ethernet_hdr_t) + sizeof(ip_hdr_t) + sizeof(tcp_hdr_t) + payload_len;
}

// Function to write packet to PCAP file
void write_pcap_packet(FILE* pcap_file, DWORD src_ip, DWORD dest_ip, WORD src_port, WORD dest_port, const BYTE* payload, int payload_len) {
    BYTE packet_buffer[65535];
    int packet_size;

    create_pcap_packet(packet_buffer, &packet_size, src_ip, dest_ip, src_port, dest_port, payload, payload_len);

    pcaprec_hdr_t rec_hdr;
    rec_hdr.ts_sec = (DWORD)time(NULL);
    rec_hdr.ts_usec = 0;
    rec_hdr.incl_len = packet_size;
    rec_hdr.orig_len = packet_size;

    fwrite(&rec_hdr, sizeof(pcaprec_hdr_t), 1, pcap_file);
    fwrite(packet_buffer, packet_size, 1, pcap_file);
}

// Function to initialize PCAP file
FILE* init_pcap_file(const char* filename) {
    FILE* file = fopen(filename, "wb");
    if (!file) return NULL;

    pcap_hdr_t hdr;
    hdr.magic_number = 0xa1b2c3d4;
    hdr.version_major = 2;
    hdr.version_minor = 4;
    hdr.thiszone = 0;
    hdr.sigfigs = 0;
    hdr.snaplen = 65535;
    hdr.network = 1; // Ethernet

    fwrite(&hdr, sizeof(pcap_hdr_t), 1, file);
    return file;
}

// Function to write raw binary dump per connection
void log_raw_dump(const BYTE* data, int len, int is_internal, DWORD src_ip, DWORD dest_ip, WORD src_port, WORD dest_port) {
    char filename[256];
    sprintf(filename, "tcp_dump_%s_%lu.%lu.%lu.%lu:%u-%lu.%lu.%lu.%lu:%u.bin",
            is_internal ? "internal" : "external",
            (src_ip >> 24) & 0xFF, (src_ip >> 16) & 0xFF, (src_ip >> 8) & 0xFF, src_ip & 0xFF, src_port,
            (dest_ip >> 24) & 0xFF, (dest_ip >> 16) & 0xFF, (dest_ip >> 8) & 0xFF, dest_ip & 0xFF, dest_port);
    FILE* file = fopen(filename, "ab");
    if (file) {
        fwrite(data, 1, len, file);
        fclose(file);
    }
}

// Function to write hex dump per connection
void log_hex_dump(const BYTE* data, int len, int is_internal, DWORD src_ip, DWORD dest_ip, WORD src_port, WORD dest_port) {
    char filename[256];
    sprintf(filename, "tcp_hexdump_%s_%lu.%lu.%lu.%lu:%u-%lu.%lu.%lu.%lu:%u.log",
            is_internal ? "internal" : "external",
            (src_ip >> 24) & 0xFF, (src_ip >> 16) & 0xFF, (src_ip >> 8) & 0xFF, src_ip & 0xFF, src_port,
            (dest_ip >> 24) & 0xFF, (dest_ip >> 16) & 0xFF, (dest_ip >> 8) & 0xFF, dest_ip & 0xFF, dest_port);
    FILE* file = fopen(filename, "a");
    if (file) {
        SYSTEMTIME st;
        GetSystemTime(&st);
        fprintf(file, "[%04d-%02d-%02d %02d:%02d:%02d] TCP Payload (%d bytes):\n",
                st.wYear, st.wMonth, st.wDay, st.wHour, st.wMinute, st.wSecond, len);
        for (int i = 0; i < len; i++) {
            fprintf(file, "%02x ", data[i]);
            if ((i + 1) % 16 == 0) fprintf(file, "\n");
        }
        fprintf(file, "\n\n");
        fclose(file);
    }
}

// Export function for use by ssl_hook.dll
__declspec(dllexport) void log_unencrypted_traffic(DWORD src_ip, DWORD dest_ip, WORD src_port, WORD dest_port, const BYTE* data, int len, int is_internal) {
    // Create per-connection PCAP filename
    char pcap_filename[256];
    sprintf(pcap_filename, "tcp_wireshark_%s_%lu.%lu.%lu.%lu:%u-%lu.%lu.%lu.%lu:%u.pcap",
            is_internal ? "internal" : "external",
            (src_ip >> 24) & 0xFF, (src_ip >> 16) & 0xFF, (src_ip >> 8) & 0xFF, src_ip & 0xFF, src_port,
            (dest_ip >> 24) & 0xFF, (dest_ip >> 16) & 0xFF, (dest_ip >> 8) & 0xFF, dest_ip & 0xFF, dest_port);

    // Open or create per-connection PCAP file
    FILE* pcap_file = fopen(pcap_filename, "r+b");
    if (!pcap_file) {
        // File doesn't exist, create it
        pcap_file = init_pcap_file(pcap_filename);
    } else {
        // File exists, seek to end for appending
        fseek(pcap_file, 0, SEEK_END);
    }

    if (pcap_file) {
        write_pcap_packet(pcap_file, src_ip, dest_ip, src_port, dest_port, data, len);
        fflush(pcap_file);
        fclose(pcap_file);
    }

    // Write raw binary dump per connection
    log_raw_dump(data, len, is_internal, src_ip, dest_ip, src_port, dest_port);

    // Write hex dump per connection
    log_hex_dump(data, len, is_internal, src_ip, dest_ip, src_port, dest_port);
}