/*
 * Network Monitoring Suite - SSL Hook Implementation
 * Copyright (C) 2024 Stefy Lanza <stefy@sexhack.me>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */

#include "ssl_hook.h"
#ifndef NO_DETOURS
#include <detours.h>
#endif
#define SECURITY_WIN32
#include <sspi.h>

// Original function pointers
SSL_write_t original_SSL_write = NULL;
SSL_read_t original_SSL_read = NULL;
EncryptMessage_t original_EncryptMessage = NULL;
DecryptMessage_t original_DecryptMessage = NULL;

// Additional SSL library function pointers
LibreSSL_write_t original_LibreSSL_write = NULL;
LibreSSL_read_t original_LibreSSL_read = NULL;
NSS_SSL_Write_t original_NSS_SSL_Write = NULL;
NSS_SSL_Read_t original_NSS_SSL_Read = NULL;
NSS_SSL_ForceHandshake_t original_NSS_SSL_ForceHandshake = NULL;
GnuTLS_record_send_t original_GnuTLS_record_send = NULL;
GnuTLS_record_recv_t original_GnuTLS_record_recv = NULL;
mbedTLS_ssl_write_t original_mbedTLS_ssl_write = NULL;
mbedTLS_ssl_read_t original_mbedTLS_ssl_read = NULL;
wolfSSL_write_t original_wolfSSL_write = NULL;
wolfSSL_read_t original_wolfSSL_read = NULL;
Botan_TLS_write_t original_Botan_TLS_write = NULL;
Botan_TLS_read_t original_Botan_TLS_read = NULL;

// Syscall monitoring function pointers
LoadLibraryA_t original_LoadLibraryA = NULL;
LoadLibraryW_t original_LoadLibraryW = NULL;
GetProcAddress_t original_GetProcAddress = NULL;

// Hooked functions
int hooked_SSL_write(void* ssl, const void* buf, int num) {
    log_data("SSL_WRITE", buf, num);
#ifndef NO_DETOURS
    // For PCAP logging, we'd need to track connection details
    // This is simplified - in practice you'd need to maintain connection state
    log_unencrypted_traffic(0, 0, 0, 443, (const BYTE*)buf, num, 0); // Assume external for now
#endif
    return original_SSL_write ? original_SSL_write(ssl, buf, num) : -1;
}

int hooked_SSL_read(void* ssl, void* buf, int num) {
    int result = original_SSL_read ? original_SSL_read(ssl, buf, num) : -1;
    if (result > 0) {
        log_data("SSL_READ", buf, result);
#ifndef NO_DETOURS
        log_unencrypted_traffic(0, 0, 443, 0, (const BYTE*)buf, result, 0); // Assume external
#endif
    }
    return result;
}

SECURITY_STATUS hooked_EncryptMessage(PCtxtHandle phContext, ULONG fQOP, PSecBufferDesc pMessage, ULONG MessageSeqNo) {
    // Log plaintext data before encryption
    for (ULONG i = 0; i < pMessage->cBuffers; i++) {
        if (pMessage->pBuffers[i].BufferType == SECBUFFER_DATA) {
            log_data("SCHANNEL_ENCRYPT", pMessage->pBuffers[i].pvBuffer, pMessage->pBuffers[i].cbBuffer);
#ifndef NO_DETOURS
            log_unencrypted_traffic(0, 0, 0, 443, (const BYTE*)pMessage->pBuffers[i].pvBuffer, pMessage->pBuffers[i].cbBuffer, 0);
#endif
            break;
        }
    }
    return original_EncryptMessage ? original_EncryptMessage(phContext, fQOP, pMessage, MessageSeqNo) : SEC_E_UNSUPPORTED_FUNCTION;
}

SECURITY_STATUS hooked_DecryptMessage(PCtxtHandle phContext, PSecBufferDesc pMessage, ULONG MessageSeqNo, PULONG pfQOP) {
    SECURITY_STATUS status = original_DecryptMessage ? original_DecryptMessage(phContext, pMessage, MessageSeqNo, pfQOP) : SEC_E_UNSUPPORTED_FUNCTION;
    if (status == SEC_E_OK) {
        // Log decrypted data after decryption
        for (ULONG i = 0; i < pMessage->cBuffers; i++) {
            if (pMessage->pBuffers[i].BufferType == SECBUFFER_DATA) {
                log_data("SCHANNEL_DECRYPT", pMessage->pBuffers[i].pvBuffer, pMessage->pBuffers[i].cbBuffer);
#ifndef NO_DETOURS
                log_unencrypted_traffic(0, 0, 443, 0, (const BYTE*)pMessage->pBuffers[i].pvBuffer, pMessage->pBuffers[i].cbBuffer, 0);
#endif
                break;
            }
        }
    }
    return status;
}

// Additional SSL library hooked functions
int hooked_LibreSSL_write(void* ssl, const void* buf, int num) {
    log_data("LIBRESSL_WRITE", buf, num);
#ifndef NO_DETOURS
    log_unencrypted_traffic(0, 0, 0, 443, (const BYTE*)buf, num, 0);
#endif
    return original_LibreSSL_write ? original_LibreSSL_write(ssl, buf, num) : -1;
}

int hooked_LibreSSL_read(void* ssl, void* buf, int num) {
    int result = original_LibreSSL_read ? original_LibreSSL_read(ssl, buf, num) : -1;
    if (result > 0) {
        log_data("LIBRESSL_READ", buf, result);
#ifndef NO_DETOURS
        log_unencrypted_traffic(0, 0, 443, 0, (const BYTE*)buf, result, 0);
#endif
    }
    return result;
}

int hooked_NSS_SSL_Write(void* ssl, const void* buf, int num) {
    log_data("NSS_SSL_WRITE", buf, num);
#ifndef NO_DETOURS
    log_unencrypted_traffic(0, 0, 0, 443, (const BYTE*)buf, num, 0);
#endif
    return original_NSS_SSL_Write ? original_NSS_SSL_Write(ssl, buf, num) : -1;
}

int hooked_NSS_SSL_Read(void* ssl, void* buf, int num) {
    int result = original_NSS_SSL_Read ? original_NSS_SSL_Read(ssl, buf, num) : -1;
    if (result > 0) {
        log_data("NSS_SSL_READ", buf, result);
#ifndef NO_DETOURS
        log_unencrypted_traffic(0, 0, 443, 0, (const BYTE*)buf, result, 0);
#endif
    }
    return result;
}

int hooked_NSS_SSL_ForceHandshake(void* ssl) {
    log_syscall("NSS_SSL_ForceHandshake", "SSL handshake initiated");
    return original_NSS_SSL_ForceHandshake ? original_NSS_SSL_ForceHandshake(ssl) : -1;
}

int hooked_GnuTLS_record_send(void* session, const void* data, size_t data_size) {
    log_data("GNUTLS_RECORD_SEND", data, (int)data_size);
#ifndef NO_DETOURS
    log_unencrypted_traffic(0, 0, 0, 443, (const BYTE*)data, (int)data_size, 0);
#endif
    return original_GnuTLS_record_send ? original_GnuTLS_record_send(session, data, data_size) : -1;
}

int hooked_GnuTLS_record_recv(void* session, void* data, size_t data_size) {
    int result = original_GnuTLS_record_recv ? original_GnuTLS_record_recv(session, data, data_size) : -1;
    if (result > 0) {
        log_data("GNUTLS_RECORD_RECV", data, result);
#ifndef NO_DETOURS
        log_unencrypted_traffic(0, 0, 443, 0, (const BYTE*)data, result, 0);
#endif
    }
    return result;
}

int hooked_mbedTLS_ssl_write(void* ssl, const unsigned char* buf, size_t len) {
    log_data("MBEDTLS_SSL_WRITE", buf, (int)len);
#ifndef NO_DETOURS
    log_unencrypted_traffic(0, 0, 0, 443, buf, (int)len, 0);
#endif
    return original_mbedTLS_ssl_write ? original_mbedTLS_ssl_write(ssl, buf, len) : -1;
}

int hooked_mbedTLS_ssl_read(void* ssl, unsigned char* buf, size_t len) {
    int result = original_mbedTLS_ssl_read ? original_mbedTLS_ssl_read(ssl, buf, len) : -1;
    if (result > 0) {
        log_data("MBEDTLS_SSL_READ", buf, result);
#ifndef NO_DETOURS
        log_unencrypted_traffic(0, 0, 443, 0, buf, result, 0);
#endif
    }
    return result;
}

int hooked_wolfSSL_write(void* ssl, const void* data, int sz) {
    log_data("WOLFSSL_WRITE", data, sz);
#ifndef NO_DETOURS
    log_unencrypted_traffic(0, 0, 0, 443, (const BYTE*)data, sz, 0);
#endif
    return original_wolfSSL_write ? original_wolfSSL_write(ssl, data, sz) : -1;
}

int hooked_wolfSSL_read(void* ssl, void* data, int sz) {
    int result = original_wolfSSL_read ? original_wolfSSL_read(ssl, data, sz) : -1;
    if (result > 0) {
        log_data("WOLFSSL_READ", data, result);
#ifndef NO_DETOURS
        log_unencrypted_traffic(0, 0, 443, 0, (const BYTE*)data, result, 0);
#endif
    }
    return result;
}

int hooked_Botan_TLS_write(void* channel, const void* buf, size_t length) {
    log_data("BOTAN_TLS_WRITE", buf, (int)length);
#ifndef NO_DETOURS
    log_unencrypted_traffic(0, 0, 0, 443, (const BYTE*)buf, (int)length, 0);
#endif
    return original_Botan_TLS_write ? original_Botan_TLS_write(channel, buf, length) : -1;
}

int hooked_Botan_TLS_read(void* channel, void* buf, size_t length) {
    int result = original_Botan_TLS_read ? original_Botan_TLS_read(channel, buf, length) : -1;
    if (result > 0) {
        log_data("BOTAN_TLS_READ", buf, result);
#ifndef NO_DETOURS
        log_unencrypted_traffic(0, 0, 443, 0, (const BYTE*)buf, result, 0);
#endif
    }
    return result;
}

HMODULE hooked_LoadLibraryA(LPCSTR lpLibFileName) {
    log_syscall("LoadLibraryA", lpLibFileName);
    return original_LoadLibraryA(lpLibFileName);
}

HMODULE hooked_LoadLibraryW(LPCWSTR lpLibFileName) {
    char libName[MAX_PATH];
    WideCharToMultiByte(CP_ACP, 0, lpLibFileName, -1, libName, MAX_PATH, NULL, NULL);
    log_syscall("LoadLibraryW", libName);
    return original_LoadLibraryW(lpLibFileName);
}

FARPROC hooked_GetProcAddress(HMODULE hModule, LPCSTR lpProcName) {
    char details[256];
    sprintf(details, "Module: 0x%p, Function: %s", hModule, lpProcName);
    log_syscall("GetProcAddress", details);
    return original_GetProcAddress(hModule, lpProcName);
}

// Helper functions
void log_data(const char* direction, const void* buf, int num) {
    FILE* logFile = fopen("ssl_log.txt", "a");
    if (logFile) {
        fprintf(logFile, "[%s] %d bytes: ", direction, num);
        for (int i = 0; i < num && i < 100; i++) { // Log first 100 bytes
            fprintf(logFile, "%02x ", ((unsigned char*)buf)[i]);
        }
        fprintf(logFile, "\n");
        fclose(logFile);
    }
}

void log_syscall(const char* syscall_name, const char* details) {
    FILE* logFile = fopen("syscall_log.txt", "a");
    if (logFile) {
        SYSTEMTIME st;
        GetSystemTime(&st);
        fprintf(logFile, "[%04d-%02d-%02d %02d:%02d:%02d] %s: %s\n",
                st.wYear, st.wMonth, st.wDay, st.wHour, st.wMinute, st.wSecond,
                syscall_name, details);
        fclose(logFile);
    }
}

// DLL entry point
BOOL APIENTRY DllMain(HMODULE hModule __attribute__((unused)), DWORD ul_reason_for_call, LPVOID lpReserved __attribute__((unused))) {
    if (ul_reason_for_call == DLL_PROCESS_ATTACH) {
        // Hook multiple SSL implementations

        // OpenSSL/LibreSSL
        HMODULE hLibSSL = GetModuleHandle("libssl-1_1.dll"); // OpenSSL 1.1.x
        if (!hLibSSL) hLibSSL = GetModuleHandle("libssl-3.dll"); // OpenSSL 3.x
        if (!hLibSSL) hLibSSL = GetModuleHandle("ssleay32.dll"); // Older OpenSSL
        if (!hLibSSL) hLibSSL = GetModuleHandle("libssl.dll"); // Generic

        if (hLibSSL) {
            original_SSL_write = (SSL_write_t)(void*)GetProcAddress(hLibSSL, "SSL_write");
            original_SSL_read = (SSL_read_t)(void*)GetProcAddress(hLibSSL, "SSL_read");

            // LibreSSL uses same function names
            original_LibreSSL_write = original_SSL_write;
            original_LibreSSL_read = original_SSL_read;

            if (original_SSL_write && original_SSL_read) {
#ifndef NO_DETOURS
                DetourTransactionBegin();
                DetourUpdateThread(GetCurrentThread());
                DetourAttach(&(PVOID&)original_SSL_write, hooked_SSL_write);
                DetourAttach(&(PVOID&)original_SSL_read, hooked_SSL_read);
                DetourAttach(&(PVOID&)original_LibreSSL_write, hooked_LibreSSL_write);
                DetourAttach(&(PVOID&)original_LibreSSL_read, hooked_LibreSSL_read);
                DetourTransactionCommit();
#endif
            }
        }

        // NSS (Mozilla)
        HMODULE hNSS = GetModuleHandle("nss3.dll");
        if (hNSS) {
            original_NSS_SSL_Write = (NSS_SSL_Write_t)(void*)GetProcAddress(hNSS, "SSL_Write");
            original_NSS_SSL_Read = (NSS_SSL_Read_t)(void*)GetProcAddress(hNSS, "SSL_Read");
            original_NSS_SSL_ForceHandshake = (NSS_SSL_ForceHandshake_t)(void*)GetProcAddress(hNSS, "SSL_ForceHandshake");

            if (original_NSS_SSL_Write && original_NSS_SSL_Read) {
#ifndef NO_DETOURS
                DetourTransactionBegin();
                DetourUpdateThread(GetCurrentThread());
                DetourAttach(&(PVOID&)original_NSS_SSL_Write, hooked_NSS_SSL_Write);
                DetourAttach(&(PVOID&)original_NSS_SSL_Read, hooked_NSS_SSL_Read);
                if (original_NSS_SSL_ForceHandshake) {
                    DetourAttach(&(PVOID&)original_NSS_SSL_ForceHandshake, hooked_NSS_SSL_ForceHandshake);
                }
                DetourTransactionCommit();
#endif
            }
        }

        // GnuTLS
        HMODULE hGnuTLS = GetModuleHandle("libgnutls-30.dll");
        if (!hGnuTLS) hGnuTLS = GetModuleHandle("libgnutls.dll");
        if (hGnuTLS) {
            original_GnuTLS_record_send = (GnuTLS_record_send_t)(void*)GetProcAddress(hGnuTLS, "gnutls_record_send");
            original_GnuTLS_record_recv = (GnuTLS_record_recv_t)(void*)GetProcAddress(hGnuTLS, "gnutls_record_recv");

            if (original_GnuTLS_record_send && original_GnuTLS_record_recv) {
#ifndef NO_DETOURS
                DetourTransactionBegin();
                DetourUpdateThread(GetCurrentThread());
                DetourAttach(&(PVOID&)original_GnuTLS_record_send, hooked_GnuTLS_record_send);
                DetourAttach(&(PVOID&)original_GnuTLS_record_recv, hooked_GnuTLS_record_recv);
                DetourTransactionCommit();
#endif
            }
        }

        // mbed TLS
        HMODULE hmbedTLS = GetModuleHandle("libmbedtls.dll");
        if (!hmbedTLS) hmbedTLS = GetModuleHandle("mbedtls.dll");
        if (hmbedTLS) {
            original_mbedTLS_ssl_write = (mbedTLS_ssl_write_t)(void*)GetProcAddress(hmbedTLS, "mbedtls_ssl_write");
            original_mbedTLS_ssl_read = (mbedTLS_ssl_read_t)(void*)GetProcAddress(hmbedTLS, "mbedtls_ssl_read");

            if (original_mbedTLS_ssl_write && original_mbedTLS_ssl_read) {
#ifndef NO_DETOURS
                DetourTransactionBegin();
                DetourUpdateThread(GetCurrentThread());
                DetourAttach(&(PVOID&)original_mbedTLS_ssl_write, hooked_mbedTLS_ssl_write);
                DetourAttach(&(PVOID&)original_mbedTLS_ssl_read, hooked_mbedTLS_ssl_read);
                DetourTransactionCommit();
#endif
            }
        }

        // wolfSSL
        HMODULE hwolfSSL = GetModuleHandle("wolfssl.dll");
        if (hwolfSSL) {
            original_wolfSSL_write = (wolfSSL_write_t)(void*)GetProcAddress(hwolfSSL, "wolfSSL_write");
            original_wolfSSL_read = (wolfSSL_read_t)(void*)GetProcAddress(hwolfSSL, "wolfSSL_read");

            if (original_wolfSSL_write && original_wolfSSL_read) {
#ifndef NO_DETOURS
                DetourTransactionBegin();
                DetourUpdateThread(GetCurrentThread());
                DetourAttach(&(PVOID&)original_wolfSSL_write, hooked_wolfSSL_write);
                DetourAttach(&(PVOID&)original_wolfSSL_read, hooked_wolfSSL_read);
                DetourTransactionCommit();
#endif
            }
        }

        // Botan (simplified - actual Botan TLS API is more complex)
        HMODULE hBotan = GetModuleHandle("botan.dll");
        if (!hBotan) hBotan = GetModuleHandle("libbotan-2.dll");
        if (hBotan) {
            // Note: Botan TLS API is C++ and more complex than this
            // This is a simplified example - real implementation would need Botan headers
            original_Botan_TLS_write = (Botan_TLS_write_t)(void*)GetProcAddress(hBotan, "botan_tls_write");
            original_Botan_TLS_read = (Botan_TLS_read_t)(void*)GetProcAddress(hBotan, "botan_tls_read");

            if (original_Botan_TLS_write && original_Botan_TLS_read) {
#ifndef NO_DETOURS
                DetourTransactionBegin();
                DetourUpdateThread(GetCurrentThread());
                DetourAttach(&(PVOID&)original_Botan_TLS_write, hooked_Botan_TLS_write);
                DetourAttach(&(PVOID&)original_Botan_TLS_read, hooked_Botan_TLS_read);
                DetourTransactionCommit();
#endif
            }
        }

        // Also try to hook Windows Schannel if available
        HMODULE hSecur32 = GetModuleHandle("secur32.dll");
        if (hSecur32) {
            original_EncryptMessage = (EncryptMessage_t)(void*)GetProcAddress(hSecur32, "EncryptMessage");
            original_DecryptMessage = (DecryptMessage_t)(void*)GetProcAddress(hSecur32, "DecryptMessage");

            if (original_EncryptMessage && original_DecryptMessage) {
#ifndef NO_DETOURS
                DetourTransactionBegin();
                DetourUpdateThread(GetCurrentThread());
                DetourAttach(&(PVOID&)original_EncryptMessage, hooked_EncryptMessage);
                DetourAttach(&(PVOID&)original_DecryptMessage, hooked_DecryptMessage);
                DetourTransactionCommit();
#endif
            }
        }

        // Hook system DLL loading functions
        HMODULE hKernel32 = GetModuleHandle("kernel32.dll");
        if (hKernel32) {
            original_LoadLibraryA = (LoadLibraryA_t)(void*)GetProcAddress(hKernel32, "LoadLibraryA");
            original_LoadLibraryW = (LoadLibraryW_t)(void*)GetProcAddress(hKernel32, "LoadLibraryW");
            original_GetProcAddress = (GetProcAddress_t)(void*)GetProcAddress(hKernel32, "GetProcAddress");

            if (original_LoadLibraryA && original_LoadLibraryW && original_GetProcAddress) {
#ifndef NO_DETOURS
                DetourTransactionBegin();
                DetourUpdateThread(GetCurrentThread());
                DetourAttach(&(PVOID&)original_LoadLibraryA, hooked_LoadLibraryA);
                DetourAttach(&(PVOID&)original_LoadLibraryW, hooked_LoadLibraryW);
                DetourAttach(&(PVOID&)original_GetProcAddress, hooked_GetProcAddress);
                DetourTransactionCommit();
#endif
            }
        }
    } else if (ul_reason_for_call == DLL_PROCESS_DETACH) {
        if (original_SSL_write && original_SSL_read) {
#ifndef NO_DETOURS
            DetourTransactionBegin();
            DetourUpdateThread(GetCurrentThread());
            DetourDetach(&(PVOID&)original_SSL_write, hooked_SSL_write);
            DetourDetach(&(PVOID&)original_SSL_read, hooked_SSL_read);
            DetourDetach(&(PVOID&)original_LibreSSL_write, hooked_LibreSSL_write);
            DetourDetach(&(PVOID&)original_LibreSSL_read, hooked_LibreSSL_read);
            DetourTransactionCommit();
#endif
        }

        if (original_NSS_SSL_Write && original_NSS_SSL_Read) {
#ifndef NO_DETOURS
            DetourTransactionBegin();
            DetourUpdateThread(GetCurrentThread());
            DetourDetach(&(PVOID&)original_NSS_SSL_Write, hooked_NSS_SSL_Write);
            DetourDetach(&(PVOID&)original_NSS_SSL_Read, hooked_NSS_SSL_Read);
            if (original_NSS_SSL_ForceHandshake) {
                DetourDetach(&(PVOID&)original_NSS_SSL_ForceHandshake, hooked_NSS_SSL_ForceHandshake);
            }
            DetourTransactionCommit();
#endif
        }

        if (original_GnuTLS_record_send && original_GnuTLS_record_recv) {
#ifndef NO_DETOURS
            DetourTransactionBegin();
            DetourUpdateThread(GetCurrentThread());
            DetourDetach(&(PVOID&)original_GnuTLS_record_send, hooked_GnuTLS_record_send);
            DetourDetach(&(PVOID&)original_GnuTLS_record_recv, hooked_GnuTLS_record_recv);
            DetourTransactionCommit();
#endif
        }

        if (original_mbedTLS_ssl_write && original_mbedTLS_ssl_read) {
#ifndef NO_DETOURS
            DetourTransactionBegin();
            DetourUpdateThread(GetCurrentThread());
            DetourDetach(&(PVOID&)original_mbedTLS_ssl_write, hooked_mbedTLS_ssl_write);
            DetourDetach(&(PVOID&)original_mbedTLS_ssl_read, hooked_mbedTLS_ssl_read);
            DetourTransactionCommit();
#endif
        }

        if (original_wolfSSL_write && original_wolfSSL_read) {
#ifndef NO_DETOURS
            DetourTransactionBegin();
            DetourUpdateThread(GetCurrentThread());
            DetourDetach(&(PVOID&)original_wolfSSL_write, hooked_wolfSSL_write);
            DetourDetach(&(PVOID&)original_wolfSSL_read, hooked_wolfSSL_read);
            DetourTransactionCommit();
#endif
        }

        if (original_Botan_TLS_write && original_Botan_TLS_read) {
#ifndef NO_DETOURS
            DetourTransactionBegin();
            DetourUpdateThread(GetCurrentThread());
            DetourDetach(&(PVOID&)original_Botan_TLS_write, hooked_Botan_TLS_write);
            DetourDetach(&(PVOID&)original_Botan_TLS_read, hooked_Botan_TLS_read);
            DetourTransactionCommit();
#endif
        }
        if (original_EncryptMessage && original_DecryptMessage) {
#ifndef NO_DETOURS
            DetourTransactionBegin();
            DetourUpdateThread(GetCurrentThread());
            DetourDetach(&(PVOID&)original_EncryptMessage, hooked_EncryptMessage);
            DetourDetach(&(PVOID&)original_DecryptMessage, hooked_DecryptMessage);
            DetourTransactionCommit();
#endif
        }
        if (original_LoadLibraryA && original_LoadLibraryW && original_GetProcAddress) {
#ifndef NO_DETOURS
            DetourTransactionBegin();
            DetourUpdateThread(GetCurrentThread());
            DetourDetach(&(PVOID&)original_LoadLibraryA, hooked_LoadLibraryA);
            DetourDetach(&(PVOID&)original_LoadLibraryW, hooked_LoadLibraryW);
            DetourDetach(&(PVOID&)original_GetProcAddress, hooked_GetProcAddress);
            DetourTransactionCommit();
#endif
        }
    }
    return TRUE;
}