#!/bin/bash

# Network Monitoring Suite Configuration Script
# This script sets up the build environment for cross-compiling Windows binaries on Linux

set -e

echo "Network Monitoring Suite - Configuration Script"
echo "==============================================="

# Check if running on Linux
if [[ "$OSTYPE" != "linux-gnu"* ]]; then
    echo "Error: This script is designed for Linux systems only."
    exit 1
fi

# Function to check if a command exists
command_exists() {
    command -v "$1" >/dev/null 2>&1
}

# Check for required tools
echo "Checking for required tools..."

if ! command_exists x86_64-w64-mingw32-gcc; then
    echo "MinGW-w64 not found. Installing..."
    sudo apt-get update
    sudo apt-get install -y gcc-mingw-w64
    echo "MinGW-w64 installed successfully."
else
    echo "✓ MinGW-w64 found"
fi

# Check for Microsoft Detours
DETOURS_PATH="/usr/local/detours"
if [ ! -f "$DETOURS_PATH/lib64/detours.lib" ]; then
    echo "Microsoft Detours not found."
    echo "Please download Detours from: https://github.com/microsoft/Detours"
    echo "Extract it to $DETOURS_PATH or set DETOURS_PATH in the Makefile"
    echo ""
    echo "Example installation:"
    echo "  mkdir -p $DETOURS_PATH"
    echo "  cd $DETOURS_PATH"
    echo "  # Extract Detours zip here"
    echo "  cd src"
    echo "  nmake"  # This requires Visual Studio Build Tools on Windows
    echo ""
    echo "Note: Detours must be built on Windows first, then copied to Linux."
    DETOURS_MISSING=1
else
    echo "✓ Microsoft Detours found"
fi

# Create build directory
echo "Creating build directory..."
mkdir -p build

# Generate config.mk for Makefile
echo "Generating configuration..."
cat > config.mk << EOF
# Auto-generated configuration file
DETOURS_PATH = $DETOURS_PATH
CC = x86_64-w64-mingw32-gcc
EOF

if [ -n "$DETOURS_MISSING" ]; then
    echo ""
    echo "WARNING: Detours not found. ssl_hook.dll cannot be built."
    echo "Set DETOURS_PATH in config.mk after installing Detours."
fi

echo ""
echo "Configuration complete!"
echo ""
echo "To build the project:"
echo "  make -f Makefile"
echo ""
echo "Or if Detours is missing:"
echo "  make -f Makefile network_monitor.exe packet_capture.dll bgvnc.exe"
echo ""
echo "Output files will be placed in the current directory."