/*
 * Network Monitoring Suite - VNC Server
 * Copyright (C) 2024 Stefy Lanza <stefy@sexhack.me>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */

#include <windows.h>
#include <winsock2.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <process.h>

#pragma comment(lib, "ws2_32.lib")
#pragma comment(lib, "user32.lib")
#pragma comment(lib, "gdi32.lib")

#define VNC_PORT 5900
#define SCREEN_WIDTH 1024
#define SCREEN_HEIGHT 768

typedef struct {
    SOCKET client_sock;
    HDC screen_dc;
    HBITMAP bitmap;
    BITMAPINFO bmi;
    BYTE* framebuffer;
    int fb_width, fb_height;
} VNC_CLIENT;

unsigned __stdcall vnc_client_thread(void* arg) {
    VNC_CLIENT* client = (VNC_CLIENT*)arg;

    // Simple VNC handshake (simplified)
    char version[] = "RFB 003.008\n";
    send(client->client_sock, version, strlen(version), 0);

    char client_version[12];
    recv(client->client_sock, client_version, sizeof(client_version), 0);

    // Send security types
    char sec_types[] = {1, 1}; // 1 = None
    send(client->client_sock, sec_types, 2, 0);

    char sec_result[4] = {0,0,0,0}; // OK
    send(client->client_sock, sec_result, 4, 0);

    // Client init
    char client_init[1];
    recv(client->client_sock, client_init, 1, 0);

    // Server init
    char server_init[24];
    memset(server_init, 0, sizeof(server_init));
    *(DWORD*)(server_init) = htonl(client->fb_width);
    *(DWORD*)(server_init + 4) = htonl(client->fb_height);
    *(DWORD*)(server_init + 8) = htonl(32); // bits per pixel
    *(DWORD*)(server_init + 12) = htonl(24); // depth
    *(DWORD*)(server_init + 16) = htonl(1); // big endian
    *(DWORD*)(server_init + 20) = htonl(1); // true color
    send(client->client_sock, server_init, 24, 0);

    // Name
    char name[] = {0,0,0,4,'B','G','V','N','C'};
    send(client->client_sock, name, sizeof(name), 0);

    while (1) {
        // Capture screen and send updates
        HDC mem_dc = CreateCompatibleDC(client->screen_dc);
        SelectObject(mem_dc, client->bitmap);

        BitBlt(mem_dc, 0, 0, client->fb_width, client->fb_height, client->screen_dc, 0, 0, SRCCOPY);

        GetDIBits(mem_dc, client->bitmap, 0, client->fb_height, client->framebuffer, &client->bmi, DIB_RGB_COLORS);

        // Send framebuffer update
        char update_header[4] = {0, 0, 0, 1}; // FramebufferUpdate message
        send(client->client_sock, update_header, 4, 0);

        // Rectangle header
        char rect_header[12] = {0,0, 0,0, 0,0, 0,0, 0,0,0,0}; // x,y,w,h,encoding
        *(WORD*)(rect_header + 2) = htons(client->fb_width);
        *(WORD*)(rect_header + 4) = htons(client->fb_height);
        *(DWORD*)(rect_header + 8) = htonl(0); // Raw encoding
        send(client->client_sock, rect_header, 12, 0);

        // Send pixel data
        send(client->client_sock, (char*)client->framebuffer, client->fb_width * client->fb_height * 4, 0);

        DeleteDC(mem_dc);
        Sleep(100); // Update every 100ms
    }

    closesocket(client->client_sock);
    free(client);
    return 0;
}

int main(int argc, char* argv[]) {
    if (argc < 2) {
        printf("Usage: bgvnc.exe <rdp_client_path> [args...]\n");
        return 1;
    }

    WSADATA wsa;
    WSAStartup(MAKEWORD(2,2), &wsa);

    SOCKET server_sock = socket(AF_INET, SOCK_STREAM, 0);
    struct sockaddr_in server_addr = {0};
    server_addr.sin_family = AF_INET;
    server_addr.sin_port = htons(VNC_PORT);
    server_addr.sin_addr.s_addr = INADDR_ANY;

    bind(server_sock, (struct sockaddr*)&server_addr, sizeof(server_addr));
    listen(server_sock, 5);

    printf("VNC server listening on port %d\n", VNC_PORT);

    // Launch RDP client as subprocess
    STARTUPINFO si = { sizeof(si) };
    PROCESS_INFORMATION pi;
    char cmdline[1024] = {0};
    for (int i = 1; i < argc; i++) {
        strcat(cmdline, argv[i]);
        if (i < argc - 1) strcat(cmdline, " ");
    }

    if (!CreateProcess(NULL, cmdline, NULL, NULL, FALSE, 0, NULL, NULL, &si, &pi)) {
        printf("Failed to start RDP client\n");
        return 1;
    }

    // Wait a bit for RDP client to start
    Sleep(2000);

    // Get screen dimensions
    int width = GetSystemMetrics(SM_CXSCREEN);
    int height = GetSystemMetrics(SM_CYSCREEN);

    HDC screen_dc = GetDC(NULL);
    HBITMAP bitmap = CreateCompatibleBitmap(screen_dc, width, height);

    BITMAPINFO bmi = {0};
    bmi.bmiHeader.biSize = sizeof(BITMAPINFOHEADER);
    bmi.bmiHeader.biWidth = width;
    bmi.bmiHeader.biHeight = -height; // Negative for top-down
    bmi.bmiHeader.biPlanes = 1;
    bmi.bmiHeader.biBitCount = 32;
    bmi.bmiHeader.biCompression = BI_RGB;

    BYTE* framebuffer = (BYTE*)malloc(width * height * 4);

    while (1) {
        SOCKET client_sock = accept(server_sock, NULL, NULL);
        if (client_sock == INVALID_SOCKET) continue;

        VNC_CLIENT* client = (VNC_CLIENT*)malloc(sizeof(VNC_CLIENT));
        client->client_sock = client_sock;
        client->screen_dc = screen_dc;
        client->bitmap = bitmap;
        client->bmi = bmi;
        client->framebuffer = framebuffer;
        client->fb_width = width;
        client->fb_height = height;

        _beginthreadex(NULL, 0, vnc_client_thread, client, 0, NULL);
    }

    // Cleanup
    free(framebuffer);
    DeleteObject(bitmap);
    ReleaseDC(NULL, screen_dc);
    CloseHandle(pi.hProcess);
    CloseHandle(pi.hThread);
    closesocket(server_sock);
    WSACleanup();

    return 0;
}