#!/bin/bash

# MBetter ISO Customization Script
# Allows changing root password and adding OpenVPN config to existing ISO

set -e

# Colors for output
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
BLUE='\033[0;34m'
NC='\033[0m' # No Color

# Function to print colored output
print_status() {
    echo -e "${GREEN}[INFO]${NC} $1"
}

print_warning() {
    echo -e "${YELLOW}[WARNING]${NC} $1"
}

print_error() {
    echo -e "${RED}[ERROR]${NC} $1"
}

print_header() {
    echo -e "${BLUE}================================${NC}"
    echo -e "${BLUE}$1${NC}"
    echo -e "${BLUE}================================${NC}"
}

# Function to check if running as root
check_root() {
    if [ "$EUID" -ne 0 ]; then
        print_error "This script must be run with sudo privileges."
        echo "Usage: sudo ./customize_iso.sh"
        exit 1
    fi
}

# Function to validate ISO file
validate_iso() {
    local iso_file="$1"
    if [ ! -f "$iso_file" ]; then
        print_error "ISO file '$iso_file' not found!"
        exit 1
    fi

    if ! file "$iso_file" | grep -q "ISO 9660"; then
        print_error "'$iso_file' is not a valid ISO file!"
        exit 1
    fi

    print_status "ISO file validated: $iso_file"
}

# Function to create temporary directories
create_temp_dirs() {
    TEMP_DIR=$(mktemp -d)
    ISO_MOUNT="$TEMP_DIR/iso_mount"
    ISO_EXTRACT="$TEMP_DIR/iso_extract"
    ISO_NEW="$TEMP_DIR/iso_new"

    mkdir -p "$ISO_MOUNT" "$ISO_EXTRACT" "$ISO_NEW"

    print_status "Created temporary directories in: $TEMP_DIR"
}

# Function to mount and extract ISO
mount_and_extract_iso() {
    local iso_file="$1"

    print_status "Mounting ISO..."
    mount -o loop,ro "$iso_file" "$ISO_MOUNT"

    print_status "Extracting ISO contents..."
    cp -r "$ISO_MOUNT"/* "$ISO_EXTRACT"/

    # Copy filesystem.squashfs for modification
    if [ -f "$ISO_MOUNT/live/filesystem.squashfs" ]; then
        cp "$ISO_MOUNT/live/filesystem.squashfs" "$ISO_EXTRACT/live/"
    fi
}

# Function to modify root password
modify_root_password() {
    local new_password="$1"

    print_status "Modifying root password..."

    # Generate encrypted password
    local encrypted_password
    encrypted_password=$(openssl passwd -1 "$new_password")

    # Find and modify preseed file
    local preseed_file="$ISO_EXTRACT/preseed.cfg"
    if [ -f "$preseed_file" ]; then
        sed -i "s|d-i passwd/root-password-crypted password .*|d-i passwd/root-password-crypted password $encrypted_password|" "$preseed_file"
        print_status "Updated preseed file with new root password"
    else
        print_warning "Preseed file not found in ISO. Password modification may not work."
    fi
}

# Function to add OpenVPN configuration
add_openvpn_config() {
    local config_file="$1"
    local key_file="$2"

    print_status "Adding OpenVPN configuration..."

    # Create OpenVPN directory in extracted ISO
    local vpn_dir="$ISO_EXTRACT/config/includes.chroot/etc/openvpn"
    mkdir -p "$vpn_dir"

    # Copy configuration file
    if [ -f "$config_file" ]; then
        cp "$config_file" "$vpn_dir/client.conf"
        print_status "Added OpenVPN config: $config_file"
    else
        print_error "OpenVPN config file not found: $config_file"
        return 1
    fi

    # Copy key file if provided
    if [ -n "$key_file" ] && [ -f "$key_file" ]; then
        cp "$key_file" "$vpn_dir/"
        local key_filename=$(basename "$key_file")
        print_status "Added OpenVPN key: $key_filename"
    fi

    # Create OpenVPN start hook
    local hook_dir="$ISO_EXTRACT/config/hooks/live"
    mkdir -p "$hook_dir"

    cat > "$hook_dir/start-openvpn.hook.chroot" << 'EOF'
#!/bin/bash
# Start OpenVPN on boot
if [ -f /etc/openvpn/client.conf ]; then
    systemctl enable openvpn@client
    systemctl start openvpn@client
fi
EOF

    chmod +x "$hook_dir/start-openvpn.hook.chroot"
    print_status "Created OpenVPN startup hook"
}

# Function to rebuild ISO
rebuild_iso() {
    local output_iso="$1"

    print_status "Rebuilding ISO..."

    # Use genisoimage or mkisofs to create new ISO
    if command -v genisoimage &> /dev/null; then
        genisoimage -o "$output_iso" \
            -b isolinux/isolinux.bin \
            -c isolinux/boot.cat \
            -no-emul-boot \
            -boot-load-size 4 \
            -boot-info-table \
            -J -R -V "MBetter Live" \
            "$ISO_EXTRACT"
    elif command -v mkisofs &> /dev/null; then
        mkisofs -o "$output_iso" \
            -b isolinux/isolinux.bin \
            -c isolinux/boot.cat \
            -no-emul-boot \
            -boot-load-size 4 \
            -boot-info-table \
            -J -R -V "MBetter Live" \
            "$ISO_EXTRACT"
    else
        print_error "Neither genisoimage nor mkisofs found. Please install one of them."
        return 1
    fi

    print_status "New ISO created: $output_iso"
}

# Function to cleanup
cleanup() {
    print_status "Cleaning up..."

    # Unmount ISO
    if mountpoint -q "$ISO_MOUNT"; then
        umount "$ISO_MOUNT"
    fi

    # Remove temporary directories
    if [ -d "$TEMP_DIR" ]; then
        rm -rf "$TEMP_DIR"
    fi

    print_status "Cleanup completed"
}

# Function to show usage
show_usage() {
    echo "MBetter ISO Customization Script"
    echo ""
    echo "Usage: sudo ./customize_iso.sh [OPTIONS]"
    echo ""
    echo "Required:"
    echo "  -i, --iso FILE        Input ISO file to customize"
    echo "  -p, --password PASS   New root password"
    echo ""
    echo "Optional:"
    echo "  -o, --openvpn FILE    OpenVPN configuration file (.ovpn)"
    echo "  -k, --key FILE        OpenVPN key/certificate file"
    echo "  -O, --output FILE     Output ISO file (default: customized.iso)"
    echo "  -h, --help           Show this help"
    echo ""
    echo "Examples:"
    echo "  sudo ./customize_iso.sh -i mbeter.iso -p mynewpassword"
    echo "  sudo ./customize_iso.sh -i mbeter.iso -p mypass -o client.ovpn -k client.key -O custom.iso"
}

# Main function
main() {
    local iso_file=""
    local new_password=""
    local openvpn_config=""
    local openvpn_key=""
    local output_iso="customized.iso"

    # Parse command line arguments
    while [[ $# -gt 0 ]]; do
        case $1 in
            -i|--iso)
                iso_file="$2"
                shift 2
                ;;
            -p|--password)
                new_password="$2"
                shift 2
                ;;
            -o|--openvpn)
                openvpn_config="$2"
                shift 2
                ;;
            -k|--key)
                openvpn_key="$2"
                shift 2
                ;;
            -O|--output)
                output_iso="$2"
                shift 2
                ;;
            -h|--help)
                show_usage
                exit 0
                ;;
            *)
                print_error "Unknown option: $1"
                show_usage
                exit 1
                ;;
        esac
    done

    # Validate required arguments
    if [ -z "$iso_file" ]; then
        print_error "Input ISO file is required!"
        show_usage
        exit 1
    fi

    if [ -z "$new_password" ]; then
        print_error "New root password is required!"
        show_usage
        exit 1
    fi

    print_header "MBetter ISO Customization Tool"

    # Check if running as root
    check_root

    # Validate ISO file
    validate_iso "$iso_file"

    # Create temporary directories
    create_temp_dirs

    # Set up cleanup trap
    trap cleanup EXIT

    # Mount and extract ISO
    mount_and_extract_iso "$iso_file"

    # Modify root password
    modify_root_password "$new_password"

    # Add OpenVPN config if provided
    if [ -n "$openvpn_config" ]; then
        add_openvpn_config "$openvpn_config" "$openvpn_key"
    fi

    # Rebuild ISO
    rebuild_iso "$output_iso"

    print_header "Customization Complete!"
    print_status "Original ISO: $iso_file"
    print_status "Customized ISO: $output_iso"
    print_status "New root password: $new_password"

    if [ -n "$openvpn_config" ]; then
        print_status "OpenVPN config added: $openvpn_config"
        if [ -n "$openvpn_key" ]; then
            print_status "OpenVPN key added: $openvpn_key"
        fi
    fi

    print_status "You can now use the customized ISO: $output_iso"
}

# Run main function with all arguments
main "$@"