#!/bin/bash

echo "MBetter ISO to USB Creator (Linux)"
echo "==================================="
echo

# Check if ISO file exists
if [ ! -f "live-image-amd64.iso" ]; then
    echo "ERROR: live-image-amd64.iso not found in current directory!"
    echo "Please ensure the ISO file is in the same directory as this script."
    exit 1
fi

# Check if running as root
if [ "$EUID" -ne 0 ]; then
    echo "ERROR: This script must be run with sudo privileges."
    echo "Usage: sudo ./create_usb.sh"
    exit 1
fi

echo "Available block devices:"
echo
lsblk -d -o NAME,SIZE,MODEL | grep -v loop
echo
echo "USB drives typically appear as /dev/sdX (where X is a letter like b, c, etc.)"
echo "WARNING: This will completely erase the selected device!"
echo "Make sure you select the correct device for your USB drive."
echo

read -p "Enter the device path for your USB drive (e.g., /dev/sdb): " USB_DEVICE

if [ -z "$USB_DEVICE" ]; then
    echo "ERROR: No device entered."
    exit 1
fi

# Validate device exists
if [ ! -b "$USB_DEVICE" ]; then
    echo "ERROR: Device $USB_DEVICE does not exist or is not a block device."
    exit 1
fi

# Get device info
DEVICE_INFO=$(lsblk -d -o NAME,SIZE,MODEL "$USB_DEVICE" 2>/dev/null)
if [ $? -ne 0 ]; then
    echo "ERROR: Cannot access device $USB_DEVICE"
    exit 1
fi

echo
echo "You selected: $DEVICE_INFO"
read -p "Are you sure you want to erase $USB_DEVICE? This will destroy all data! (yes/no): " CONFIRM

if [ "$CONFIRM" != "yes" ]; then
    echo "Operation cancelled."
    exit 0
fi

echo
echo "Creating bootable USB drive..."
echo "This may take several minutes. Please wait..."
echo

# Unmount any mounted partitions on the device
umount "${USB_DEVICE}"* 2>/dev/null

# Write ISO to USB using dd with progress
dd if=live-image-amd64.iso of="$USB_DEVICE" bs=4M status=progress oflag=sync

if [ $? -eq 0 ]; then
    echo
    echo "SUCCESS: USB drive created successfully!"
    echo
    echo "Your MBetter USB install media is ready at $USB_DEVICE"
    echo "You can now boot from this USB drive to install or run MBetter live system."
    echo
    echo "IMPORTANT: Make sure your BIOS/UEFI is set to boot from USB first."
    echo
    # Sync to ensure write is complete
    sync
    echo "Write operation completed. It's safe to remove the USB drive."
else
    echo
    echo "ERROR: Failed to create USB drive."
    exit 1
fi