#!/bin/bash

echo "Cleaning up MBetter ISO build configuration..."

# Check if running as root
if [ "$EUID" -ne 0 ]; then
    echo "ERROR: This script must be run with sudo privileges."
    echo "Usage: sudo ./cleanup.sh"
    exit 1
fi

# Revert root password to default in preseed file
sed -i 's|d-i passwd/root-password-crypted password .*|d-i passwd/root-password password changeme|' config/preseed/debian-installer.cfg

# Remove OpenVPN configuration
if [ -d "config/includes.chroot/etc/openvpn" ]; then
    rm -rf config/includes.chroot/etc/openvpn
    echo "Removed OpenVPN configuration"
fi

# Remove OpenVPN start hook
if [ -f "config/hooks/live/start-openvpn.hook.chroot" ]; then
    rm config/hooks/live/start-openvpn.hook.chroot
    echo "Removed OpenVPN start hook"
fi

# Clean build artifacts
if [ -f "live-image-amd64.iso" ]; then
    rm live-image-amd64.iso
    echo "Removed previous ISO file"
fi

# Remove any other ISO files
for iso_file in *.iso; do
    if [ -f "$iso_file" ]; then
        rm "$iso_file"
        echo "Removed ISO file: $iso_file"
    fi
done

# Remove build logs
if [ -f "build.log" ]; then
    rm build.log
    echo "Removed build.log"
fi

if [ -f "build_retry.log" ]; then
    rm build_retry.log
    echo "Removed build_retry.log"
fi

# Remove .build directory
if [ -d ".build" ]; then
    rm -rf .build
    echo "Removed .build directory"
fi

# Remove cache directory
if [ -d "cache" ]; then
    rm -rf cache
    echo "Removed cache directory"
fi

# Unmount any mounted filesystems in chroot
if [ -d "chroot" ]; then
    echo "Checking for mounted filesystems in chroot..."

    # Try to unmount common mount points
    mount_points=(
        "chroot/dev/pts"
        "chroot/dev/shm"
        "chroot/dev"
        "chroot/proc"
        "chroot/sys"
        "chroot/run"
    )

    for mount_point in "${mount_points[@]}"; do
        if mountpoint -q "$mount_point" 2>/dev/null; then
            echo "Unmounting $mount_point..."
            umount "$mount_point" 2>/dev/null || true
        fi
    done

    # Force unmount any remaining mounts
    if mount | grep -q "chroot"; then
        echo "Force unmounting remaining chroot mounts..."
        mount | grep "chroot" | awk '{print $3}' | xargs -r umount -f 2>/dev/null || true
    fi
fi

# Clean live-build cache and temporary files
lb clean

echo "Cleanup completed. Configuration reset to defaults."