# GLIBC Compatibility Issue

## Problem
The PyInstaller executable built with Python 3.13 requires GLIBC 2.38 or higher, but the target deployment system has an older GLIBC version.

**Error Message:**
```
[PYI-16812:ERROR] Failed to load Python shared library '/tmp/_MEIXZCEYU/libpython3.13.so': /lib/x86_64-linux-gnu/libm.so.6: version `GLIBC_2.38' not found (required by /tmp/_MEIXZCEYU/libpython3.13.so)
```

## Solutions

### Solution 1: Build on Target System (Recommended)
Build the executable directly on the target system or a system with the same GLIBC version:

```bash
# On the target system:
git clone <repository>
cd MBetter
python3 -m venv venv
source venv/bin/activate
pip install -r requirements-build.txt
python build.py
```

### Solution 2: Use Docker for Consistent Build Environment
Create a Docker container with an older base system:

```dockerfile
# Dockerfile
FROM ubuntu:20.04
RUN apt-get update && apt-get install -y python3 python3-pip python3-venv git
WORKDIR /app
COPY . .
RUN python3 -m venv venv
RUN . venv/bin/activate && pip install -r requirements-build.txt
RUN . venv/bin/activate && python build.py
```

Build with Docker:
```bash
docker build -t fixture-manager-builder .
docker run --rm -v $(pwd)/dist:/app/dist fixture-manager-builder
```

### Solution 3: Use Older Python Version
If available, use Python 3.11 or 3.10 which have better GLIBC compatibility:

```bash
# Check available Python versions
ls /usr/bin/python3*

# Use older Python version
python3.11 -m venv venv
source venv/bin/activate
# Update requirements to compatible versions for older Python
pip install -r requirements-build.txt
python build.py
```

### Solution 4: Static Linking (Advanced)
Modify the PyInstaller spec file to use static linking:

```python
# In fixture-manager.spec, add to the EXE section:
exe = EXE(
    # ... existing parameters ...
    strip=False,
    upx=True,
    upx_exclude=[],
    runtime_tmpdir=None,
    console=True,
    # Add static linking options
    bootloader_ignore_signals=False,
    disable_windowed_traceback=False,
    argv_emulation=False,
    target_arch=None,
    codesign_identity=None,
    entitlements_file=None,
    icon=None,
    # Force static linking
    exclude_binaries=False,
)
```

### Solution 5: Virtual Environment Deployment
Instead of a single executable, deploy with a virtual environment:

```bash
# Create deployment package
python3 -m venv deployment_venv
source deployment_venv/bin/activate
pip install -r requirements.txt

# Package the entire virtual environment
tar -czf fixture-manager-deployment.tar.gz deployment_venv/ fixture_daemon.py app/ database/ config.py .env.example

# On target system:
tar -xzf fixture-manager-deployment.tar.gz
source deployment_venv/bin/activate
python fixture_daemon.py start
```

## Checking Target System Compatibility

To check the GLIBC version on your target system:
```bash
ldd --version
# or
/lib/x86_64-linux-gnu/libc.so.6
```

## Recommended Approach

For maximum compatibility, **Solution 1** (building on the target system) is recommended as it ensures the executable is built against the exact libraries available on the deployment system.

If that's not possible, **Solution 5** (virtual environment deployment) provides the most reliable cross-system compatibility while maintaining all functionality.