# Reports Sync Implementation Guide

## Overview
This document describes the implementation of the reports sync API endpoint and web interface according to the Reports Synchronization Protocol Documentation.

## Changes Made

### 1. Database Model Updates

#### ReportSync Model (`app/models.py`)
- **Added**: `cap_compensation_balance` field to track the CAP redistribution balance at the time of sync
- **Type**: `NUMERIC(15, 2)` with default value 0.00
- **Purpose**: Stores the accumulated shortfall from cap compensation system for historical tracking

#### Updated `to_dict()` method
- Now includes `cap_compensation_balance` in JSON serialization

### 2. API Endpoint Updates (`app/api/routes.py`)

#### `/api/reports/sync` Endpoint

**New Features:**
1. **Cap Compensation Balance Handling**
   - Accepts `cap_compensation_balance` field in request payload
   - Stores the value in the ReportSync record
   - Tracks historical CAP balance over time

2. **Differentiation Between New and Updated Records**
   - **Bets**: 
     - Checks for existing bet UUIDs
     - Updates existing bets if fields have changed (total_amount, paid, paid_out)
     - Tracks `bets_new` and `bets_duplicate` counts
     - Skips unchanged duplicate bets
   
   - **Extraction Stats**:
     - Checks for existing stats by match_id + client_id
     - Updates existing stats with new data
     - Tracks `stats_new` and `stats_updated` counts

3. **Enhanced Logging**
   - `ReportSyncLog` now tracks:
     - `bets_new`: Number of new bets added
     - `bets_duplicate`: Number of duplicate bets (updated or skipped)
     - `stats_new`: Number of new extraction stats
     - `stats_updated`: Number of updated extraction stats

**Request Payload Example:**
```json
{
  "sync_id": "sync_20260201_214327_abc12345",
  "client_id": "client_unique_identifier",
  "sync_timestamp": "2026-02-01T21:43:27.249Z",
  "date_range": "all",
  "start_date": "2026-01-01T00:00:00",
  "end_date": "2026-02-01T21:43:27.249Z",
  "bets": [...],
  "extraction_stats": [...],
  "cap_compensation_balance": 5000.0,
  "summary": {...},
  "is_incremental": true,
  "sync_type": "incremental"
}
```

### 3. Web Interface Updates

#### Reports Page (`app/templates/main/reports.html`)

**New Features:**

1. **Enhanced Date Range Filters**
   - Today (default)
   - Yesterday
   - This Week
   - Last Week
   - This Month
   - All Time
   - Custom (with date and time selection)

2. **Custom Date/Time Selection**
   - Start Date and Start Time (optional)
   - End Date and End Time (optional)
   - JavaScript toggles visibility when "Custom" is selected

3. **Client Selection Dropdown**
   - Shows all clients with their token names
   - Format: "Token Name (client_id)"
   - Filters reports by selected client

4. **Enhanced Report Table**
   - Added "Balance" column (payin - payout)
   - Added "CAP Balance" column showing cap_compensation_balance
   - Color-coded balance (green for positive, red for negative)
   - CAP balance shown in blue for visibility

**Filter Logic:**
- **Today**: From 00:00:00 today to now
- **Yesterday**: From 00:00:00 to 23:59:59 yesterday
- **This Week**: From Monday 00:00:00 to now
- **Last Week**: From Monday 00:00:00 to Sunday 23:59:59
- **This Month**: From 1st 00:00:00 to now
- **All**: No date filtering
- **Custom**: User-specified date and time range

#### Report Detail Page (`app/templates/main/report_detail.html`)

**New Features:**

1. **Enhanced Summary Section**
   - Added "CAP Redistribution Balance" card
   - Shows balance at the time of the sync
   - Displayed in yellow for visibility
   - Helps track CAP balance over time

2. **Match-Level Data Display**
   - **Bets Tab**: Shows all bets with expandable details
     - Bet UUID, Fixture ID, DateTime, Amount, Count
     - Paid/Paid Out status indicators
     - Expandable bet details with match-level information
     - Match ID, Match Number, Outcome, Amount, Win Amount, Result
   
   - **Extraction Stats Tab**: Shows all extraction stats with expandable details
     - Match ID, Fixture ID, DateTime, Bets, Collected, Redistributed
     - Actual Result, Extraction Result, CAP Applied status
     - Expandable details showing:
       - Under/Over bets and amounts
       - Result breakdown by outcome

### 4. Backend Route Updates (`app/main/routes.py`)

#### `/reports` Route

**New Features:**

1. **Date Range Calculation**
   - Implements logic for all date range filters
   - Handles week calculations (Monday start)
   - Supports custom date/time ranges

2. **Client Data Aggregation**
   - Joins ReportSync with ClientActivity and APIToken
   - Provides client_id and token_name for dropdown
   - Filters by user's tokens for non-admin users

3. **Enhanced Filtering**
   - Applies date range filters to query
   - Supports client_id filtering
   - Maintains pagination and sorting

## Database Migration

### Automatic Migration System

The application uses an automatic migration system that runs all pending migrations at application launch. The migration to add `cap_compensation_balance` is already integrated into this system.

**Migration Class**: `Migration_011_AddCapCompensationBalance` in [`app/database/migrations.py`](app/database/migrations.py:777)

**What it does:**
1. Checks if `report_syncs` table exists
2. Checks if `cap_compensation_balance` column already exists
3. Adds `cap_compensation_balance DECIMAL(15,2) NOT NULL DEFAULT 0.00` if not present
4. Provides rollback capability (downgrade function)

**Automatic Execution:**
- Migrations run automatically when the application starts (see [`app/__init__.py`](app/__init__.py:68-77))
- The migration system tracks applied migrations in the `database_versions` table
- Only pending migrations are executed on each startup
- Migration status is logged to the application log

**Manual Execution (if needed):**
```bash
# The migration will run automatically on app startup
# To manually trigger migrations:
python -c "from app.database import run_migrations; print(run_migrations())"
```

**Upgrade SQL:**
```sql
ALTER TABLE report_syncs
ADD COLUMN cap_compensation_balance DECIMAL(15,2) NOT NULL DEFAULT 0.00
```

**Downgrade SQL:**
```sql
ALTER TABLE report_syncs
DROP COLUMN IF EXISTS cap_compensation_balance
```

## Usage Examples

### 1. Automatic Migration Execution

The migration will run automatically when you start the application. No manual intervention is required.

**Expected log output on first run:**
```
INFO: Applying migration 011: Add cap_compensation_balance column to report_syncs table
INFO: Added cap_compensation_balance column successfully
INFO: Migration 011 applied successfully
INFO: Database migrations completed: Successfully applied 1 migrations
```

**Expected log output on subsequent runs:**
```
INFO: Migration 011 already applied, skipping
INFO: No pending migrations
```

**To verify migration was applied:**
```bash
# Check application logs
tail -f logs/fixture-daemon.log | grep migration

# Or check database directly
mysql -u your_user -p your_database -e "DESCRIBE report_syncs;" | grep cap_compensation_balance
```

### 2. Syncing Reports from Client

```bash
curl -X POST http://your-server/api/reports/sync \
  -H "Content-Type: application/json" \
  -H "Authorization: Bearer YOUR_API_TOKEN" \
  -d '{
    "sync_id": "sync_20260201_214327_abc12345",
    "client_id": "client_machine_001",
    "sync_timestamp": "2026-02-01T21:43:27.249Z",
    "date_range": "all",
    "start_date": "2026-01-01T00:00:00",
    "end_date": "2026-02-01T21:43:27.249Z",
    "bets": [
      {
        "uuid": "bet-uuid-12345",
        "fixture_id": "fixture-20260201",
        "bet_datetime": "2026-02-01T10:30:00",
        "paid": true,
        "paid_out": false,
        "total_amount": 5000.0,
        "bet_count": 2,
        "details": [...]
      }
    ],
    "extraction_stats": [...],
    "cap_compensation_balance": 5000.0,
    "summary": {
      "total_payin": 100000.0,
      "total_payout": 95000.0,
      "net_profit": 5000.0,
      "total_bets": 50,
      "total_matches": 1
    },
    "is_incremental": true,
    "sync_type": "incremental"
  }'
```

### 3. Viewing Reports in Web Interface

1. Navigate to `/reports`
2. Use filters to narrow down data:
   - Select a client from dropdown
   - Choose date range (Today, Yesterday, This Week, etc.)
   - For custom range, select dates and optional times
3. Click "Apply Filters"
4. View aggregated data:
   - Total Payin, Payout, Balance
   - CAP Redistribution Balance
   - Total Bets and Matches
5. Click "View" on any report to see details

### 4. Viewing Report Details

1. Click "View" on any report in the list
2. See summary with CAP balance at sync time
3. Switch between tabs:
   - **Bets**: View all bets with match-level details
   - **Extraction Stats**: View extraction statistics with breakdowns
4. Expand rows to see detailed information

## Key Features

### 1. Historical CAP Balance Tracking
- CAP balance is stored with each sync
- Shows balance at the time of the report
- Enables tracking CAP balance changes over time
- Displayed prominently in reports and details

### 2. Record Differentiation
- **New Records**: First-time sync of a bet or extraction stat
- **Updated Records**: Existing records with changed data
- **Duplicate Records**: Existing records without changes (skipped)
- All tracked in sync logs for audit trail

### 3. Flexible Date Filtering
- Predefined ranges for common use cases
- Custom range with optional time precision
- JavaScript toggles custom fields visibility
- Default to "Today" for immediate relevance

### 4. Client-Centric View
- Filter by specific client
- Shows token names for easy identification
- Aggregates data by client
- Supports multi-client environments

### 5. Match-Level Detail
- Expandable bet details showing individual match bets
- Extraction stats with result breakdowns
- Under/Over bet distribution
- Complete audit trail for each match

## Data Flow

### Sync Process
1. Client sends sync request with all data
2. Server validates request and authenticates
3. Server checks for duplicate sync_id (idempotency)
4. For each bet:
   - Check if UUID exists
   - If exists and changed: Update
   - If exists and unchanged: Skip
   - If new: Create
5. For each extraction stat:
   - Check if match_id + client_id exists
   - If exists: Update
   - If new: Create
6. Store cap_compensation_balance with sync
7. Log operation details
8. Return success response

### Report Viewing Process
1. User navigates to reports page
2. Applies filters (client, date range, etc.)
3. Server queries database with filters
4. Aggregates data by client
5. Displays in table with CAP balance
6. User clicks "View" for details
7. Server retrieves all bets and stats for sync
8. Displays with expandable match-level details

## Testing

### 1. Test Automatic Migration
```bash
# Start the application - migration will run automatically
python main.py
```

Check logs for migration status:
```bash
tail -f logs/fixture-daemon.log | grep -i migration
```

Verify column exists in database:
```sql
DESCRIBE report_syncs;
```

### 2. Test API Endpoint
```bash
# Test with cap_compensation_balance
curl -X POST http://localhost:5000/api/reports/sync \
  -H "Content-Type: application/json" \
  -H "Authorization: Bearer YOUR_TOKEN" \
  -d @test_sync_payload.json
```

### 3. Test Web Interface
1. Start application
2. Navigate to `/reports`
3. Test each date range filter
4. Test custom date/time selection
5. Test client filtering
6. Verify CAP balance display
7. Click through to detail pages
8. Verify match-level data display

### 4. Test Record Updates
1. Send initial sync with bet data
2. Send updated sync with same bet UUID but different amount
3. Verify bet was updated (not duplicated)
4. Check sync logs for update tracking

## Troubleshooting

### Migration Issues
- **Error**: Column already exists
  - **Solution**: Migration checks for this, safe to ignore
- **Error**: Permission denied
  - **Solution**: Run with database user that has ALTER TABLE privileges

### API Issues
- **Error**: Invalid UUID format
  - **Solution**: Ensure UUIDs are valid UUID v4 format
- **Error**: Duplicate sync_id
  - **Solution**: This is expected behavior (idempotency)
- **Error**: Missing cap_compensation_balance
  - **Solution**: Field is optional, defaults to 0.00

### Web Interface Issues
- **Issue**: Custom date fields not showing
  - **Solution**: Check JavaScript console for errors, ensure Bootstrap loaded
- **Issue**: Client dropdown empty
  - **Solution**: Ensure clients have synced data and tokens are active
- **Issue**: CAP balance not displaying
  - **Solution**: Run migration to add column, check database schema

## Security Considerations

1. **Authentication**: All API endpoints require valid API token
2. **Authorization**: Non-admin users only see their own clients' data
3. **Input Validation**: All datetime fields validated before processing
4. **SQL Injection**: Uses SQLAlchemy ORM, parameterized queries
5. **XSS Protection**: Templates use Jinja2 auto-escaping

## Performance Considerations

1. **Indexing**: 
   - `sync_id` indexed for duplicate detection
   - `client_id` indexed for filtering
   - `sync_timestamp` indexed for date range queries
   - `uuid` indexed in bets table for lookups

2. **Pagination**: 
   - Reports list paginated (default 20 per page)
   - Prevents loading large datasets at once

3. **Query Optimization**:
   - Uses JOINs efficiently
   - Filters applied at database level
   - Only retrieves necessary columns

## Future Enhancements

1. **Real-time Updates**: WebSocket for live sync status
2. **Charts**: Visual representation of CAP balance over time
3. **Export Enhancements**: More export formats and customization
4. **Advanced Filtering**: Filter by match status, outcome, etc.
5. **Bulk Actions**: Bulk export, delete, or archive operations

## References

- [Reports Synchronization Protocol Documentation](REPORTS_SYNC_PROTOCOL_DOCUMENTATION.md)
- [API Routes](app/api/routes.py)
- [Main Routes](app/main/routes.py)
- [Database Models](app/models.py)

## Support

For issues or questions:
1. Check this documentation
2. Review protocol documentation
3. Check application logs
4. Verify database schema
5. Test with sample data