import logging
import json
from datetime import datetime
from flask import request, jsonify, current_app
from flask_jwt_extended import jwt_required, get_jwt_identity
from sqlalchemy import func, desc
from app.api import bp
from app import db, csrf
from app.utils.security import require_admin, require_active_user
from app.utils.logging import log_api_request
from app.upload.file_handler import get_file_upload_handler
from app.upload.fixture_parser import get_fixture_parser

logger = logging.getLogger(__name__)

@bp.route('/matches', methods=['GET'])
@csrf.exempt
@jwt_required()
def api_get_matches():
    """Get matches with pagination and filtering"""
    try:
        from app.models import User, Match, FileUpload
        user_id = get_jwt_identity()
        user = User.query.get(user_id)
        
        if not user or not user.is_active:
            return jsonify({'error': 'User not found or inactive'}), 404
        
        # Pagination parameters
        page = request.args.get('page', 1, type=int)
        per_page = min(request.args.get('per_page', 20, type=int), 100)
        
        # Filtering parameters
        status_filter = request.args.get('status')
        search_query = request.args.get('search', '').strip()
        active_only = request.args.get('active_only', 'false').lower() == 'true'
        
        # Base query
        if user.is_admin:
            query = Match.query
        else:
            query = Match.query.filter_by(created_by=user_id)
        
        # Apply filters
        if active_only:
            query = query.filter_by(active_status=True)
        
        if status_filter == 'active':
            query = query.filter_by(active_status=True)
        elif status_filter == 'pending':
            query = query.filter_by(active_status=False)
        elif status_filter == 'zip_pending':
            query = query.filter_by(zip_upload_status='pending')
        elif status_filter == 'zip_completed':
            query = query.filter_by(zip_upload_status='completed')
        elif status_filter == 'zip_failed':
            query = query.filter_by(zip_upload_status='failed')
        
        # Search functionality
        if search_query:
            search_pattern = f"%{search_query}%"
            query = query.filter(
                db.or_(
                    Match.fighter1_township.ilike(search_pattern),
                    Match.fighter2_township.ilike(search_pattern),
                    Match.venue_kampala_township.ilike(search_pattern),
                    Match.match_number.like(search_pattern)
                )
            )
        
        # Sorting
        sort_by = request.args.get('sort_by', 'created_at')
        sort_order = request.args.get('sort_order', 'desc')
        
        if hasattr(Match, sort_by):
            sort_column = getattr(Match, sort_by)
            if sort_order == 'asc':
                query = query.order_by(sort_column.asc())
            else:
                query = query.order_by(sort_column.desc())
        else:
            query = query.order_by(Match.created_at.desc())
        
        # Execute pagination
        matches_pagination = query.paginate(
            page=page, per_page=per_page, error_out=False
        )
        
        # Include outcomes if requested
        include_outcomes = request.args.get('include_outcomes', 'false').lower() == 'true'
        
        matches_data = []
        for match in matches_pagination.items:
            match_dict = match.to_dict(include_outcomes=include_outcomes)
            matches_data.append(match_dict)
        
        return jsonify({
            'matches': matches_data,
            'pagination': {
                'page': page,
                'pages': matches_pagination.pages,
                'per_page': per_page,
                'total': matches_pagination.total,
                'has_next': matches_pagination.has_next,
                'has_prev': matches_pagination.has_prev
            },
            'filters': {
                'status': status_filter,
                'search': search_query,
                'active_only': active_only,
                'sort_by': sort_by,
                'sort_order': sort_order
            }
        }), 200
        
    except Exception as e:
        logger.error(f"API get matches error: {str(e)}")
        return jsonify({'error': 'Failed to retrieve matches'}), 500

@bp.route('/matches/<int:match_id>', methods=['GET'])
@csrf.exempt
@jwt_required()
def api_get_match(match_id):
    """Get specific match details"""
    try:
        from app.models import User, Match, FileUpload
        user_id = get_jwt_identity()
        user = User.query.get(user_id)
        
        if not user or not user.is_active:
            return jsonify({'error': 'User not found or inactive'}), 404
        
        # Get match
        if user.is_admin:
            match = Match.query.get(match_id)
        else:
            match = Match.query.filter_by(id=match_id, created_by=user_id).first()
        
        if not match:
            return jsonify({'error': 'Match not found'}), 404
        
        # Get associated uploads
        uploads = FileUpload.query.filter_by(match_id=match_id).all()
        
        return jsonify({
            'match': match.to_dict(include_outcomes=True),
            'uploads': [upload.to_dict() for upload in uploads]
        }), 200
        
    except Exception as e:
        logger.error(f"API get match error: {str(e)}")
        return jsonify({'error': 'Failed to retrieve match'}), 500

@bp.route('/matches/<int:match_id>', methods=['PUT'])
@csrf.exempt
@jwt_required()
def api_update_match(match_id):
    """Update match details"""
    try:
        from app.models import User, Match
        user_id = get_jwt_identity()
        user = User.query.get(user_id)
        
        if not user or not user.is_active:
            return jsonify({'error': 'User not found or inactive'}), 404
        
        # Get match
        if user.is_admin:
            match = Match.query.get(match_id)
        else:
            match = Match.query.filter_by(id=match_id, created_by=user_id).first()
        
        if not match:
            return jsonify({'error': 'Match not found'}), 404
        
        data = request.get_json()
        if not data:
            return jsonify({'error': 'No data provided'}), 400
        
        # Update allowed fields
        updatable_fields = ['start_time', 'end_time', 'result']
        updated_fields = []
        
        for field in updatable_fields:
            if field in data:
                if field in ['start_time', 'end_time'] and data[field]:
                    try:
                        setattr(match, field, datetime.fromisoformat(data[field]))
                        updated_fields.append(field)
                    except ValueError:
                        return jsonify({'error': f'Invalid datetime format for {field}'}), 400
                else:
                    setattr(match, field, data[field])
                    updated_fields.append(field)
        
        if updated_fields:
            match.updated_at = datetime.utcnow()
            db.session.commit()
            
            return jsonify({
                'message': 'Match updated successfully',
                'updated_fields': updated_fields,
                'match': match.to_dict()
            }), 200
        else:
            return jsonify({'message': 'No fields to update'}), 200
        
    except Exception as e:
        logger.error(f"API update match error: {str(e)}")
        db.session.rollback()
        return jsonify({'error': 'Failed to update match'}), 500

@bp.route('/matches/<int:match_id>', methods=['DELETE'])
@csrf.exempt
@jwt_required()
def api_delete_match(match_id):
    """Delete match (admin only)"""
    try:
        from app.models import User, Match, FileUpload
        user_id = get_jwt_identity()
        user = User.query.get(user_id)
        
        if not user or not user.is_admin:
            return jsonify({'error': 'Admin privileges required'}), 403
        
        match = Match.query.get(match_id)
        if not match:
            return jsonify({'error': 'Match not found'}), 404
        
        # Delete associated files
        uploads = FileUpload.query.filter_by(match_id=match_id).all()
        for upload in uploads:
            try:
                import os
                if os.path.exists(upload.file_path):
                    os.remove(upload.file_path)
            except Exception as e:
                logger.warning(f"Failed to delete file {upload.file_path}: {str(e)}")
        
        # Delete match (cascades to outcomes and uploads)
        db.session.delete(match)
        db.session.commit()
        
        return jsonify({'message': 'Match deleted successfully'}), 200
        
    except Exception as e:
        logger.error(f"API delete match error: {str(e)}")
        db.session.rollback()
        return jsonify({'error': 'Failed to delete match'}), 500

@bp.route('/statistics', methods=['GET'])
@csrf.exempt
@jwt_required()
def api_get_statistics():
    """Get comprehensive statistics"""
    try:
        from app.models import User, Match, FileUpload
        user_id = get_jwt_identity()
        user = User.query.get(user_id)
        
        if not user or not user.is_active:
            return jsonify({'error': 'User not found or inactive'}), 404
        
        # User statistics
        user_stats = {
            'total_matches': Match.query.filter_by(created_by=user_id).count(),
            'active_matches': Match.query.filter_by(
                created_by=user_id, active_status=True
            ).count(),
            'total_uploads': FileUpload.query.filter_by(uploaded_by=user_id).count(),
            'completed_uploads': FileUpload.query.filter_by(
                uploaded_by=user_id, upload_status='completed'
            ).count(),
            'failed_uploads': FileUpload.query.filter_by(
                uploaded_by=user_id, upload_status='failed'
            ).count(),
            'pending_zip_uploads': Match.query.filter_by(
                created_by=user_id, zip_upload_status='pending'
            ).count()
        }
        
        # Global statistics (if admin)
        global_stats = {}
        if user.is_admin:
            from app.database import get_db_manager
            db_manager = get_db_manager()
            global_stats = db_manager.get_database_stats()
            
            file_handler = get_file_upload_handler()
            fixture_parser = get_fixture_parser()
            
            upload_stats = file_handler.get_upload_statistics()
            parsing_stats = fixture_parser.get_parsing_statistics()
            
            global_stats.update({
                'upload_stats': upload_stats,
                'parsing_stats': parsing_stats
            })
        
        # Recent activity
        recent_matches = Match.query.filter_by(created_by=user_id)\
            .order_by(Match.created_at.desc()).limit(5).all()
        
        recent_uploads = FileUpload.query.filter_by(uploaded_by=user_id)\
            .order_by(FileUpload.created_at.desc()).limit(5).all()
        
        return jsonify({
            'user_stats': user_stats,
            'global_stats': global_stats,
            'recent_activity': {
                'matches': [match.to_dict(include_outcomes=False) for match in recent_matches],
                'uploads': [upload.to_dict() for upload in recent_uploads]
            }
        }), 200
        
    except Exception as e:
        logger.error(f"API statistics error: {str(e)}")
        return jsonify({'error': 'Failed to retrieve statistics'}), 500

@bp.route('/admin/users', methods=['GET'])
@csrf.exempt
@jwt_required()
@require_admin
def api_admin_get_users():
    """Get users list (admin only)"""
    try:
        page = request.args.get('page', 1, type=int)
        per_page = min(request.args.get('per_page', 20, type=int), 100)
        
        search_query = request.args.get('search', '').strip()
        status_filter = request.args.get('status')
        
        from app.models import User
        # Base query
        query = User.query
        
        # Apply filters
        if status_filter == 'active':
            query = query.filter_by(is_active=True)
        elif status_filter == 'inactive':
            query = query.filter_by(is_active=False)
        elif status_filter == 'admin':
            query = query.filter_by(is_admin=True)
        
        # Search functionality
        if search_query:
            search_pattern = f"%{search_query}%"
            query = query.filter(
                db.or_(
                    User.username.ilike(search_pattern),
                    User.email.ilike(search_pattern)
                )
            )
        
        # Pagination
        users_pagination = query.order_by(User.created_at.desc()).paginate(
            page=page, per_page=per_page, error_out=False
        )
        
        return jsonify({
            'users': [user.to_dict() for user in users_pagination.items],
            'pagination': {
                'page': page,
                'pages': users_pagination.pages,
                'per_page': per_page,
                'total': users_pagination.total,
                'has_next': users_pagination.has_next,
                'has_prev': users_pagination.has_prev
            }
        }), 200
        
    except Exception as e:
        logger.error(f"API admin users error: {str(e)}")
        return jsonify({'error': 'Failed to retrieve users'}), 500

@bp.route('/admin/users/<int:user_id>', methods=['PUT'])
@csrf.exempt
@jwt_required()
@require_admin
def api_admin_update_user(user_id):
    """Update user (admin only)"""
    try:
        from app.models import User
        user = User.query.get(user_id)
        if not user:
            return jsonify({'error': 'User not found'}), 404
        
        data = request.get_json()
        if not data:
            return jsonify({'error': 'No data provided'}), 400
        
        # Update allowed fields
        updatable_fields = ['is_active', 'is_admin']
        updated_fields = []
        
        for field in updatable_fields:
            if field in data:
                setattr(user, field, bool(data[field]))
                updated_fields.append(field)
        
        if updated_fields:
            user.updated_at = datetime.utcnow()
            db.session.commit()
            
            return jsonify({
                'message': 'User updated successfully',
                'updated_fields': updated_fields,
                'user': user.to_dict()
            }), 200
        else:
            return jsonify({'message': 'No fields to update'}), 200
        
    except Exception as e:
        logger.error(f"API admin update user error: {str(e)}")
        db.session.rollback()
        return jsonify({'error': 'Failed to update user'}), 500

@bp.route('/admin/logs', methods=['GET'])
@csrf.exempt
@jwt_required()
@require_admin
def api_admin_get_logs():
    """Get system logs (admin only)"""
    try:
        page = request.args.get('page', 1, type=int)
        per_page = min(request.args.get('per_page', 50, type=int), 200)
        
        level_filter = request.args.get('level')
        module_filter = request.args.get('module')
        
        from app.models import SystemLog
        # Base query
        query = SystemLog.query
        
        # Apply filters
        if level_filter:
            query = query.filter_by(level=level_filter)
        
        if module_filter:
            query = query.filter_by(module=module_filter)
        
        # Pagination
        logs_pagination = query.order_by(SystemLog.created_at.desc()).paginate(
            page=page, per_page=per_page, error_out=False
        )
        
        return jsonify({
            'logs': [log.to_dict() for log in logs_pagination.items],
            'pagination': {
                'page': page,
                'pages': logs_pagination.pages,
                'per_page': per_page,
                'total': logs_pagination.total,
                'has_next': logs_pagination.has_next,
                'has_prev': logs_pagination.has_prev
            }
        }), 200
        
    except Exception as e:
        logger.error(f"API admin logs error: {str(e)}")
        return jsonify({'error': 'Failed to retrieve logs'}), 500

@bp.route('/admin/system-info', methods=['GET'])
@csrf.exempt
@jwt_required()
@require_admin
def api_admin_system_info():
    """Get system information (admin only)"""
    try:
        from app.database import get_db_manager
        db_manager = get_db_manager()
        
        # Database statistics
        db_stats = db_manager.get_database_stats()
        
        # System health
        db_healthy = db_manager.test_connection()
        
        # Upload statistics
        file_handler = get_file_upload_handler()
        upload_stats = file_handler.get_upload_statistics()
        
        # Parsing statistics
        fixture_parser = get_fixture_parser()
        parsing_stats = fixture_parser.get_parsing_statistics()
        
        from app.models import UserSession
        # Active sessions
        active_sessions = UserSession.query.filter_by(is_active=True).count()
        
        system_info = {
            'database': {
                'healthy': db_healthy,
                'statistics': db_stats
            },
            'uploads': upload_stats,
            'parsing': parsing_stats,
            'sessions': {
                'active_sessions': active_sessions
            },
            'timestamp': datetime.utcnow().isoformat()
        }
        
        return jsonify(system_info), 200
        
    except Exception as e:
        logger.error(f"API system info error: {str(e)}")
        return jsonify({'error': 'Failed to retrieve system information'}), 500

@bp.route('/admin/cleanup', methods=['POST'])
@csrf.exempt
@jwt_required()
@require_admin
def api_admin_cleanup():
    """Perform system cleanup (admin only)"""
    try:
        cleanup_type = request.json.get('type', 'all') if request.json else 'all'
        
        results = {}
        
        if cleanup_type in ['all', 'sessions']:
            # Clean up expired sessions
            from app.database import get_db_manager
            db_manager = get_db_manager()
            expired_sessions = db_manager.cleanup_expired_sessions()
            results['expired_sessions_cleaned'] = expired_sessions
        
        if cleanup_type in ['all', 'logs']:
            # Clean up old logs (older than 30 days)
            days = request.json.get('log_retention_days', 30) if request.json else 30
            from app.database import get_db_manager
            db_manager = get_db_manager()
            old_logs = db_manager.cleanup_old_logs(days)
            results['old_logs_cleaned'] = old_logs
        
        if cleanup_type in ['all', 'uploads']:
            # Clean up failed uploads
            file_handler = get_file_upload_handler()
            file_handler.cleanup_failed_uploads()
            results['failed_uploads_cleaned'] = True
        
        return jsonify({
            'message': 'Cleanup completed successfully',
            'results': results
        }), 200
        
    except Exception as e:
        logger.error(f"API cleanup error: {str(e)}")
        return jsonify({'error': 'Cleanup failed'}), 500

# Error handlers for API
@bp.errorhandler(404)
def api_not_found(error):
    return jsonify({'error': 'Endpoint not found'}), 404

@bp.errorhandler(405)
def api_method_not_allowed(error):
    return jsonify({'error': 'Method not allowed'}), 405

@bp.errorhandler(500)
def api_internal_error(error):
    return jsonify({'error': 'Internal server error'}), 500

@bp.route('/updates', methods=['GET', 'POST'])
@csrf.exempt
def api_get_updates():
    """Get fixtures with updates since a specific timestamp - supports both JWT and API tokens"""
    try:
        from app.models import Match, User
        from flask_jwt_extended import jwt_required, get_jwt_identity, verify_jwt_in_request
        from app.auth.jwt_utils import validate_api_token, extract_token_from_request
        from flask import request

        # Debug logging for request
        user_agent = request.headers.get('User-Agent')
        logger.info(f"api_get_updates: Request received - Method: {request.method}, User-Agent: '{user_agent}', Headers: {dict(request.headers)}")

        user = None
        auth_method = None
        api_token = None
        
        # Try JWT authentication first (short-lived session tokens)
        try:
            verify_jwt_in_request()
            user_id = get_jwt_identity()
            user = User.query.get(user_id)
            auth_method = "JWT"
            logger.info(f"API access via JWT token by user {user.username if user else 'unknown'}")
        except Exception as jwt_error:
            logger.debug(f"JWT authentication failed: {str(jwt_error)}")
            
            # If JWT fails, try API token authentication (long-lived tokens)
            try:
                from app.auth.jwt_utils import validate_api_token, extract_token_from_request
                token = extract_token_from_request()
                if not token:
                    return jsonify({
                        'error': 'Authentication required',
                        'message': 'Either JWT token or API token required'
                    }), 401
                
                user, api_token = validate_api_token(token)
                auth_method = f"API Token ({api_token.name if api_token else 'JWT'})"
                logger.info(f"API access via API token by user {user.username}")
            except Exception as api_error:
                logger.debug(f"API token authentication failed: {str(api_error)}")
                return jsonify({
                    'error': 'Authentication failed',
                    'message': 'Invalid JWT token or API token'
                }), 401
        
        if not user or not user.is_active:
            return jsonify({'error': 'User not found or inactive'}), 404
        
        logger.info(f"API updates accessed via {auth_method} by user {user.username} (ID: {user.id})")
        
        # Track client activity if using API token
        if api_token:
            if request.method == 'GET':
                rustdesk_id = request.args.get('rustdesk_id')
            else:  # POST
                data = request.get_json() or {}
                rustdesk_id = data.get('rustdesk_id')
            track_client_activity(api_token, rustdesk_id)
        
        # Get 'from' parameter (unix timestamp) - now optional
        # Support both GET (query params) and POST (JSON body)
        if request.method == 'GET':
            from_timestamp = request.args.get('from')
        else:  # POST
            data = request.get_json() or {}
            from_timestamp = data.get('from')
        
        # Get the default fixtures count for both cases
        from app.models import SystemSettings
        default_fixtures_count = SystemSettings.get_setting('api_updates_default_count', 10)
        
        if from_timestamp is not None:
            # If 'from' parameter is provided, validate it and get fixtures after that timestamp
            try:
                from_timestamp = int(from_timestamp)
            except (ValueError, TypeError):
                return jsonify({'error': 'Parameter "from" must be a valid unix timestamp'}), 400
            
            # Get fixtures with active time after the given timestamp (limited to max count)
            fixtures_data = Match.get_fixtures_with_active_time(from_timestamp, limit=default_fixtures_count)
        else:
            # If 'from' parameter is not provided, get the last N activated fixtures
            fixtures_data = Match.get_last_activated_fixtures(default_fixtures_count)
            
            # Fallback: If no fixtures have fixture_active_time set, get active fixtures by active_status
            if not fixtures_data:
                logger.warning("No fixtures with fixture_active_time found, falling back to active_status")
                from sqlalchemy import func
                
                fallback_query = db.session.query(
                    Match.fixture_id,
                    Match.fixture_active_time,
                    Match.filename,
                    func.min(Match.created_at).label('created_at')
                ).filter(
                    Match.active_status == True
                ).group_by(
                    Match.fixture_id,
                    Match.fixture_active_time,
                    Match.filename
                )
                
                if from_timestamp is not None:
                    # For 'from' queries, order by created_at ascending and limit
                    fallback_query = fallback_query.order_by(func.min(Match.created_at).asc())
                else:
                    # For default queries, get most recent fixtures
                    fallback_query = fallback_query.order_by(func.min(Match.created_at).desc())
                
                fixtures_data = fallback_query.limit(default_fixtures_count).all()
        
        if not fixtures_data:
            return jsonify({
                'fixtures': [],
                'count': 0,
                'from_timestamp': from_timestamp
            }), 200
        
        result_fixtures = []
        
        for fixture_row in fixtures_data:
            fixture_id = fixture_row.fixture_id
            fixture_active_time = fixture_row.fixture_active_time
            filename = fixture_row.filename
            created_at = fixture_row.created_at
            
            # Get all matches for this fixture
            matches = Match.query.filter_by(fixture_id=fixture_id).all()
            
            # Prepare match data with zip download links
            matches_data = []
            for match in matches:
                match_dict = match.to_dict(include_outcomes=True)
                
                # Add zip file download link if available
                if match.zip_filename and match.zip_upload_status == 'completed':
                    from flask import url_for, current_app
                    import os
                    
                    # Construct download link using authenticated API endpoint
                    # Use the persistent ZIP uploads directory
                    zip_path = os.path.join(current_app.config.get('ZIP_UPLOADS_DIR', current_app.config.get('UPLOAD_FOLDER', 'uploads')), match.zip_filename)
                    if os.path.exists(zip_path):
                        match_dict['zip_download_url'] = url_for('api.api_download_zip',
                                                               match_id=match.id,
                                                               _external=True)
                    else:
                        match_dict['zip_download_url'] = None
                else:
                    match_dict['zip_download_url'] = None
                
                matches_data.append(match_dict)
            
            # Create fixture summary
            fixture_summary = {
                'fixture_id': fixture_id,
                'filename': filename,
                'fixture_active_time': fixture_active_time,
                'created_at': created_at.isoformat() if created_at else None,
                'matches_count': len(matches_data),
                'matches': matches_data
            }
            
            result_fixtures.append(fixture_summary)
        
        return jsonify({
            'fixtures': result_fixtures,
            'count': len(result_fixtures),
            'from_timestamp': from_timestamp
        }), 200
        
    except Exception as e:
        logger.error(f"API updates error: {str(e)}")
        return jsonify({'error': 'Failed to retrieve updates'}), 500

@bp.route('/download/zip/<int:match_id>', methods=['GET'])
def api_download_zip(match_id):
    """Download ZIP file for specific match - supports both JWT and API tokens"""
    try:
        from app.models import Match, User
        from flask_jwt_extended import jwt_required, get_jwt_identity, verify_jwt_in_request
        from app.auth.jwt_utils import validate_api_token, extract_token_from_request
        from flask import send_file
        import os
        
        user = None
        auth_method = None
        
        # Try JWT authentication first (short-lived session tokens)
        try:
            verify_jwt_in_request()
            user_id = get_jwt_identity()
            user = User.query.get(user_id)
            auth_method = "JWT"
            logger.info(f"ZIP download via JWT token by user {user.username if user else 'unknown'}")
        except Exception as jwt_error:
            logger.debug(f"JWT authentication failed: {str(jwt_error)}")
            
            # If JWT fails, try API token authentication (long-lived tokens)
            try:
                token = extract_token_from_request()
                if not token:
                    return jsonify({
                        'error': 'Authentication required',
                        'message': 'Either JWT token or API token required'
                    }), 401
                
                user, api_token = validate_api_token(token)
                auth_method = f"API Token ({api_token.name if api_token else 'JWT'})"
                logger.info(f"ZIP download via API token by user {user.username}")
            except Exception as api_error:
                logger.debug(f"API token authentication failed: {str(api_error)}")
                return jsonify({
                    'error': 'Authentication failed',
                    'message': 'Invalid JWT token or API token'
                }), 401
        
        if not user or not user.is_active:
            return jsonify({'error': 'User not found or inactive'}), 404
        
        # Get match and verify access
        match = Match.query.get(match_id)
        if not match:
            return jsonify({'error': 'Match not found'}), 404
        
        # Check if user has access to this match (admin or creator)
        if not user.is_admin and match.created_by != user.id:
            return jsonify({'error': 'Access denied - you can only download ZIP files for your own matches'}), 403
        
        # Check if ZIP file exists
        if not match.zip_filename or match.zip_upload_status != 'completed':
            return jsonify({'error': 'ZIP file not available for this match'}), 404
        
        # Construct file path using persistent ZIP directory
        zip_path = os.path.join(current_app.config.get('ZIP_UPLOADS_DIR', current_app.config.get('UPLOAD_FOLDER', 'uploads')), match.zip_filename)
        
        if not os.path.exists(zip_path):
            return jsonify({'error': 'ZIP file not found on disk'}), 404
        
        logger.info(f"ZIP file download successful via {auth_method} by user {user.username} for match {match_id}")
        
        return send_file(zip_path, as_attachment=True, download_name=match.zip_filename)
        
    except Exception as e:
        logger.error(f"API ZIP download error: {str(e)}")
        return jsonify({'error': 'ZIP download failed'}), 500

def track_client_activity(api_token, rustdesk_id=None):
    """Track client activity for online status"""
    try:
        if not api_token or not api_token.is_valid():
            logger.debug(f"track_client_activity: Invalid API token")
            return

        from app.models import ClientActivity
        from flask import request

        # Get client info
        ip_address = request.headers.get('X-Forwarded-For', request.remote_addr)
        user_agent = request.headers.get('User-Agent')

        logger.info(f"track_client_activity: api_token_id={api_token.id}, rustdesk_id={rustdesk_id}, ip={ip_address}, user_agent='{user_agent}'")

        if rustdesk_id:
            # If rustdesk_id is provided, use specific client
            client = ClientActivity.query.filter_by(
                api_token_id=api_token.id,
                rustdesk_id=rustdesk_id
            ).first()

            if client:
                # Update existing client
                old_user_agent = client.user_agent
                client.last_seen = datetime.utcnow()
                client.ip_address = ip_address
                client.user_agent = user_agent
                logger.info(f"track_client_activity: Updated existing client {client.id}, user_agent changed from '{old_user_agent}' to '{user_agent}'")
            else:
                # Create new client
                client = ClientActivity(
                    api_token_id=api_token.id,
                    rustdesk_id=rustdesk_id,
                    ip_address=ip_address,
                    user_agent=user_agent
                )
                db.session.add(client)
                logger.info(f"track_client_activity: Created new client with rustdesk_id {rustdesk_id}")
        else:
            # If no rustdesk_id provided, update most recent client for this API token
            client = ClientActivity.query.filter_by(
                api_token_id=api_token.id
            ).order_by(ClientActivity.last_seen.desc()).first()

            if client:
                # Update most recent client
                old_user_agent = client.user_agent
                client.last_seen = datetime.utcnow()
                client.ip_address = ip_address
                client.user_agent = user_agent
                logger.info(f"track_client_activity: Updated most recent client {client.id} (rustdesk_id: {client.rustdesk_id}), user_agent changed from '{old_user_agent}' to '{user_agent}'")
            else:
                # Create new client with unknown rustdesk_id
                client = ClientActivity(
                    api_token_id=api_token.id,
                    rustdesk_id='unknown',
                    ip_address=ip_address,
                    user_agent=user_agent
                )
                db.session.add(client)
                logger.info(f"track_client_activity: Created new client with unknown rustdesk_id")

        db.session.commit()
        logger.info(f"track_client_activity: Database commit successful")

    except Exception as e:
        logger.error(f"Failed to track client activity: {str(e)}")
        db.session.rollback()

@bp.route('/track', methods=['POST'])
@csrf.exempt
def api_track_client():
    """Track client activity with rustdesk_id"""
    try:
        from app.auth.jwt_utils import validate_api_token, extract_token_from_request
        
        token = extract_token_from_request()
        if not token:
            return jsonify({'error': 'API token required'}), 401
        
        user, api_token = validate_api_token(token)
        if not user or not user.is_active:
            return jsonify({'error': 'User not found or inactive'}), 404
        
        # Get rustdesk_id from request
        data = request.get_json() or {}
        rustdesk_id = data.get('rustdesk_id')
        
        if not rustdesk_id:
            return jsonify({'error': 'rustdesk_id is required'}), 400
        
        # Track client activity
        track_client_activity(api_token, rustdesk_id)
        
        return jsonify({
            'message': 'Client activity tracked successfully',
            'rustdesk_id': rustdesk_id,
            'last_seen': datetime.utcnow().isoformat()
        }), 200
        
    except Exception as e:
        logger.error(f"API track client error: {str(e)}")
        return jsonify({'error': 'Failed to track client'}), 500

@bp.route('/reports/sync', methods=['POST'])
@csrf.exempt
def api_reports_sync():
    """Synchronize report data from clients"""
    import time
    start_time = time.time()
    
    try:
        from app.models import ReportSync, Bet, BetDetail, ExtractionStats, APIToken, ReportSyncLog, MatchReport, ClientActivity
        from app.auth.jwt_utils import validate_api_token, extract_token_from_request
        from datetime import datetime
        import uuid as uuid_lib
        
        # Get request metadata for logging
        ip_address = request.headers.get('X-Forwarded-For', request.remote_addr)
        user_agent = request.headers.get('User-Agent')
        request_size = len(request.data) if request.data else 0
        
        # Log incoming request details
        logger.info(f"Reports sync request received from {ip_address}")
        logger.info(f"Request headers: {dict(request.headers)}")
        logger.info(f"Request size: {request_size} bytes")
        logger.info(f"Request data (raw, first 1000 bytes): {request.data[:1000] if request.data else 'None'}")
        logger.info(f"Content-Type: {request.headers.get('Content-Type', 'Not set')}")
        
        # Authenticate using API token
        token = extract_token_from_request()
        if not token:
            # Log failed authentication
            log_entry = ReportSyncLog(
                sync_id='unknown',
                client_id='unknown',
                sync_timestamp=datetime.utcnow(),
                operation_type='error',
                status='failed',
                error_message='Authentication required',
                error_details={'reason': 'API token required'},
                ip_address=ip_address,
                user_agent=user_agent,
                request_size=request_size,
                processing_time_ms=int((time.time() - start_time) * 1000)
            )
            db.session.add(log_entry)
            db.session.commit()
            
            return jsonify({
                'success': False,
                'error': 'Authentication required',
                'details': 'API token required'
            }), 401
        
        user, api_token = validate_api_token(token)
        if not user or not user.is_active:
            # Log failed authentication
            log_entry = ReportSyncLog(
                sync_id='unknown',
                client_id='unknown',
                sync_timestamp=datetime.utcnow(),
                operation_type='error',
                status='failed',
                error_message='Authentication failed',
                error_details={'reason': 'Invalid or expired API token'},
                ip_address=ip_address,
                user_agent=user_agent,
                request_size=request_size,
                processing_time_ms=int((time.time() - start_time) * 1000)
            )
            db.session.add(log_entry)
            db.session.commit()
            
            return jsonify({
                'success': False,
                'error': 'Authentication failed',
                'details': 'Invalid or expired API token'
            }), 401
        
        # Log authenticated request with full content for debugging
        logger.info(f"Authenticated reports sync request from user {user.username} (ID: {user.id})")
        
        # Get request data - use force=True to parse JSON even if Content-Type header is missing
        data = request.get_json(force=True, silent=True)
        
        # Log full request content for debugging
        if data:
            logger.info(f"Reports sync request content: {json.dumps(data, indent=2, default=str)}")
        else:
            logger.error("Reports sync request content: No JSON data provided")
            logger.error(f"Content-Type header: {request.headers.get('Content-Type', 'Not set')}")
            logger.error(f"Request data (raw): {request.data[:1000] if request.data else 'None'}")
        
        if not data:
            # Log invalid request
            log_entry = ReportSyncLog(
                sync_id='unknown',
                client_id='unknown',
                sync_timestamp=datetime.utcnow(),
                operation_type='error',
                status='failed',
                error_message='Invalid request format',
                error_details={'reason': 'No JSON data provided'},
                ip_address=ip_address,
                user_agent=user_agent,
                request_size=request_size,
                processing_time_ms=int((time.time() - start_time) * 1000)
            )
            db.session.add(log_entry)
            db.session.commit()
            
            return jsonify({
                'success': False,
                'error': 'Invalid request format',
                'details': 'No JSON data provided'
            }), 400
        
        # Validate required fields (client_id can be derived from auth token if missing)
        required_fields = ['sync_id', 'sync_timestamp', 'date_range', 'start_date', 'end_date', 'bets', 'extraction_stats', 'summary']
        for field in required_fields:
            if field not in data:
                return jsonify({
                    'success': False,
                    'error': 'Invalid request format',
                    'details': f'Missing required field: {field}'
                }), 400
            # Also check that required fields are not None/null
            if data[field] is None:
                return jsonify({
                    'success': False,
                    'error': 'Invalid request format',
                    'details': f'Required field {field} cannot be null'
                }), 400
        
        # client_id is optional - if missing or null, derive from auth token
        client_id = data.get('client_id')
        if not client_id:
            # Derive client_id from ClientActivity using API token
            most_recent_client = ClientActivity.query.filter_by(api_token_id=api_token.id)\
                .order_by(desc(ClientActivity.last_seen))\
                .first()
            
            if most_recent_client:
                client_id = most_recent_client.rustdesk_id
                logger.info(f"Reports sync: Auto-detected client_id={client_id} from API token {api_token.name} (ID: {api_token.id}) via ClientActivity")
            else:
                logger.warning(f"Reports sync: No client activity found for API token {api_token.name}")
                return jsonify({
                    'success': False,
                    'error': 'Cannot identify client',
                    'details': 'No client activity found for this API token. Please ensure client has tracked activity first.'
                }), 400
        
        # Get cap compensation balance (optional field)
        cap_compensation_balance = data.get('cap_compensation_balance', 0.00)
        
        # Check for duplicate sync_id (idempotency)
        existing_sync = ReportSync.query.filter_by(sync_id=data['sync_id']).first()
        if existing_sync:
            # Log duplicate sync
            log_entry = ReportSyncLog(
                sync_id=data['sync_id'],
                client_id=data['client_id'],
                sync_timestamp=sync_timestamp,
                operation_type='duplicate_sync',
                status='success',
                bets_processed=0,
                bets_new=0,
                bets_duplicate=0,
                stats_processed=0,
                stats_new=0,
                stats_updated=0,
                total_payin=existing_sync.total_payin,
                total_payout=existing_sync.total_payout,
                net_profit=existing_sync.net_profit,
                total_bets=existing_sync.total_bets,
                total_matches=existing_sync.total_matches,
                ip_address=ip_address,
                user_agent=user_agent,
                request_size=request_size,
                processing_time_ms=int((time.time() - start_time) * 1000)
            )
            db.session.add(log_entry)
            db.session.commit()
            
            logger.info(f"Duplicate sync_id {data['sync_id']} detected, returning success")
            return jsonify({
                'success': True,
                'synced_count': existing_sync.total_bets + existing_sync.total_matches,
                'message': 'Report data already synchronized',
                'server_timestamp': datetime.utcnow().isoformat()
            }), 200
        
        # Parse timestamps
        try:
            sync_timestamp = datetime.fromisoformat(data['sync_timestamp'])
            start_date = datetime.fromisoformat(data['start_date'])
            end_date = datetime.fromisoformat(data['end_date'])
        except ValueError as e:
            return jsonify({
                'success': False,
                'error': 'Invalid request format',
                'details': f'Invalid datetime format: {str(e)}'
            }), 400
        
        # Validate summary data
        summary = data.get('summary')
        if not all(key in summary for key in ['total_payin', 'total_payout', 'net_profit', 'total_bets', 'total_matches']):
            return jsonify({
                'success': False,
                'error': 'Invalid request format',
                'details': 'Missing required fields in summary'
            }), 400
        
        # Create report sync record
        report_sync = ReportSync(
            sync_id=data['sync_id'],
            client_id=client_id,
            sync_timestamp=sync_timestamp,
            date_range=data['date_range'],
            start_date=start_date,
            end_date=end_date,
            total_payin=summary['total_payin'],
            total_payout=summary['total_payout'],
            net_profit=summary['net_profit'],
            total_bets=summary['total_bets'],
            total_matches=summary['total_matches'],
            cap_compensation_balance=cap_compensation_balance
        )
        db.session.add(report_sync)
        
        # Process bets
        bets_count = 0
        bets_new = 0
        bets_duplicate = 0
        
        for bet_data in data['bets']:
            # Validate bet UUID
            try:
                uuid_lib.UUID(bet_data['uuid'])
            except ValueError:
                return jsonify({
                    'success': False,
                    'error': 'Invalid request format',
                    'details': f'Invalid UUID format: {bet_data.get("uuid", "unknown")}'
                }), 400
            
            # Check for duplicate bet UUID
            existing_bet = Bet.query.filter_by(uuid=bet_data['uuid']).first()
            if existing_bet:
                # Update existing bet if it has changed
                bet_updated = False
                if existing_bet.total_amount != bet_data['total_amount']:
                    existing_bet.total_amount = bet_data['total_amount']
                    bet_updated = True
                if existing_bet.paid != bet_data.get('paid', False):
                    existing_bet.paid = bet_data.get('paid', False)
                    bet_updated = True
                if existing_bet.paid_out != bet_data.get('paid_out', False):
                    existing_bet.paid_out = bet_data.get('paid_out', False)
                    bet_updated = True
                
                if bet_updated:
                    existing_bet.sync_id = report_sync.id
                    bets_count += 1
                    bets_duplicate += 1
                    logger.info(f"Updated existing bet {bet_data['uuid']}")
                else:
                    logger.warning(f"Duplicate bet UUID {bet_data['uuid']} detected, skipping (no changes)")
                    bets_duplicate += 1
                continue
            
            # Parse bet datetime
            try:
                bet_datetime = datetime.fromisoformat(bet_data['bet_datetime'])
            except ValueError:
                return jsonify({
                    'success': False,
                    'error': 'Invalid request format',
                    'details': f'Invalid bet_datetime format: {bet_data.get("bet_datetime", "unknown")}'
                }), 400
            
            # Create bet record with match_id and match_number from first detail
            first_detail = bet_data.get('details', [{}])[0] if bet_data.get('details') else {}
            bet = Bet(
                uuid=bet_data['uuid'],
                sync_id=report_sync.id,
                client_id=client_id,
                fixture_id=bet_data['fixture_id'],
                match_id=first_detail.get('match_id'),
                match_number=first_detail.get('match_number'),
                bet_datetime=bet_datetime,
                paid=bet_data.get('paid', False),
                paid_out=bet_data.get('paid_out', False),
                total_amount=bet_data['total_amount'],
                bet_count=bet_data['bet_count']
            )
            db.session.add(bet)
            # Flush to get bet.id before creating bet details
            db.session.flush()
            bets_count += 1
            bets_new += 1
            
            # Process bet details
            for detail_data in bet_data.get('details', []):
                bet_detail = BetDetail(
                    bet_id=bet.id,
                    match_id=detail_data['match_id'],
                    match_number=detail_data['match_number'],
                    outcome=detail_data['outcome'],
                    amount=detail_data['amount'],
                    win_amount=detail_data.get('win_amount', 0.00),
                    result=detail_data['result']
                )
                db.session.add(bet_detail)
        
        # Process extraction stats
        stats_count = 0
        stats_new = 0
        stats_updated = 0
        
        for stats_data in data['extraction_stats']:
            # Check for duplicate stats (match_id + client_id)
            existing_stats = ExtractionStats.query.filter_by(
                match_id=stats_data['match_id'],
                client_id=client_id
            ).first()
            
            if existing_stats:
                # Update existing stats
                existing_stats.sync_id = report_sync.id
                existing_stats.fixture_id = stats_data['fixture_id']
                existing_stats.match_datetime = datetime.fromisoformat(stats_data['match_datetime'])
                existing_stats.total_bets = stats_data['total_bets']
                existing_stats.total_amount_collected = stats_data['total_amount_collected']
                existing_stats.total_redistributed = stats_data['total_redistributed']
                existing_stats.actual_result = stats_data['actual_result']
                existing_stats.extraction_result = stats_data['extraction_result']
                existing_stats.cap_applied = stats_data.get('cap_applied', False)
                existing_stats.cap_percentage = stats_data.get('cap_percentage')
                existing_stats.accumulated_shortfall = stats_data.get('accumulated_shortfall', 0.00)
                existing_stats.under_bets = stats_data.get('under_bets', 0)
                existing_stats.under_amount = stats_data.get('under_amount', 0.00)
                existing_stats.over_bets = stats_data.get('over_bets', 0)
                existing_stats.over_amount = stats_data.get('over_amount', 0.00)
                existing_stats.result_breakdown = stats_data.get('result_breakdown')
                
                # Update match_number from bet details
                from sqlalchemy import func
                bet_details_for_match = db.session.query(
                    BetDetail.match_number
                ).join(Bet).filter(
                    Bet.client_id == client_id,
                    BetDetail.match_id == stats_data['match_id']
                ).first()
                
                if bet_details_for_match:
                    existing_stats.match_number = bet_details_for_match.match_number
                
                stats_count += 1
                stats_updated += 1
            else:
                # Create new stats record
                try:
                    match_datetime = datetime.fromisoformat(stats_data['match_datetime'])
                except ValueError:
                    return jsonify({
                        'success': False,
                        'error': 'Invalid request format',
                        'details': f'Invalid match_datetime format: {stats_data.get("match_datetime", "unknown")}'
                    }), 400
                
                # Get match_number from bet details for this match
                match_number = 0
                from sqlalchemy import func
                bet_details_for_match = db.session.query(
                    BetDetail.match_number
                ).join(Bet).filter(
                    Bet.client_id == client_id,
                    BetDetail.match_id == stats_data['match_id']
                ).first()
                
                if bet_details_for_match:
                    match_number = bet_details_for_match.match_number
                
                extraction_stats = ExtractionStats(
                    sync_id=report_sync.id,
                    client_id=client_id,
                    match_id=stats_data['match_id'],
                    match_number=match_number,
                    fixture_id=stats_data['fixture_id'],
                    match_datetime=match_datetime,
                    total_bets=stats_data['total_bets'],
                    total_amount_collected=stats_data['total_amount_collected'],
                    total_redistributed=stats_data['total_redistributed'],
                    actual_result=stats_data['actual_result'],
                    extraction_result=stats_data['extraction_result'],
                    cap_applied=stats_data.get('cap_applied', False),
                    cap_percentage=stats_data.get('cap_percentage'),
                    accumulated_shortfall=stats_data.get('accumulated_shortfall', 0.00),
                    under_bets=stats_data.get('under_bets', 0),
                    under_amount=stats_data.get('under_amount', 0.00),
                    over_bets=stats_data.get('over_bets', 0),
                    over_amount=stats_data.get('over_amount', 0.00),
                    result_breakdown=stats_data.get('result_breakdown')
                )
                db.session.add(extraction_stats)
                stats_count += 1
                stats_new += 1
        
        # Create MatchReport records for comprehensive match-level data
        match_reports_count = 0
        
        # First, process extraction_stats if provided
        for stats_data in data['extraction_stats']:
            # Get client token name
            client_activity = ClientActivity.query.filter_by(rustdesk_id=client_id).first()
            client_token_name = client_activity.api_token.name if client_activity and client_activity.api_token else 'Unknown'
            
            # Calculate winning/losing/pending bets from bet details
            match_id = stats_data['match_id']
            winning_bets = 0
            losing_bets = 0
            pending_bets = 0
            match_number = 0
            
            # Query all bet details for this match to get match_number
            from sqlalchemy import func
            bet_details_query = db.session.query(
                BetDetail.result,
                BetDetail.match_number,
                func.count(BetDetail.id).label('count')
            ).join(Bet).filter(
                Bet.client_id == client_id,
                BetDetail.match_id == match_id
            ).group_by(BetDetail.result, BetDetail.match_number)
            
            for result, match_num, count in bet_details_query.all():
                if result == 'won':
                    winning_bets = count
                elif result == 'lost':
                    losing_bets = count
                elif result == 'pending':
                    pending_bets = count
                # Get match_number from first bet detail
                if match_num and match_number == 0:
                    match_number = match_num
            
            # Calculate balance (payin - payout)
            total_payin = stats_data['total_amount_collected']
            total_payout = stats_data['total_redistributed']
            balance = total_payin - total_payout
            
            # Create MatchReport record
            match_report = MatchReport(
                sync_id=report_sync.id,
                client_id=client_id,
                client_token_name=client_token_name,
                match_id=match_id,
                match_number=match_number,
                fixture_id=stats_data['fixture_id'],
                match_datetime=datetime.fromisoformat(stats_data['match_datetime']),
                total_bets=stats_data['total_bets'],
                winning_bets=winning_bets,
                losing_bets=losing_bets,
                pending_bets=pending_bets,
                total_payin=total_payin,
                total_payout=total_payout,
                balance=balance,
                actual_result=stats_data['actual_result'],
                extraction_result=stats_data['extraction_result'],
                cap_applied=stats_data.get('cap_applied', False),
                cap_percentage=stats_data.get('cap_percentage'),
                cap_compensation_balance=cap_compensation_balance,
                accumulated_shortfall=stats_data.get('accumulated_shortfall', 0.00),
                under_bets=stats_data.get('under_bets', 0),
                under_amount=stats_data.get('under_amount', 0.00),
                over_bets=stats_data.get('over_bets', 0),
                over_amount=stats_data.get('over_amount', 0.00),
                result_breakdown=stats_data.get('result_breakdown')
            )
            db.session.add(match_report)
            match_reports_count += 1
        
        # If extraction_stats is empty but we have bets, create MatchReport records from bets
        # This handles incremental updates where only bets are sent
        if not data['extraction_stats'] and data['bets']:
            logger.info(f"Creating MatchReport records from bets (extraction_stats is empty)")
            
            # Get client token name
            client_activity = ClientActivity.query.filter_by(rustdesk_id=client_id).first()
            client_token_name = client_activity.api_token.name if client_activity and client_activity.api_token else 'Unknown'
            
            # Group bets by match_id to create match reports
            from sqlalchemy import func
            bets_by_match = db.session.query(
                Bet.match_id,
                Bet.match_number,
                Bet.fixture_id,
                func.min(Bet.bet_datetime).label('match_datetime'),
                func.count(Bet.id).label('total_bets'),
                func.sum(Bet.total_amount).label('total_payin')
            ).filter(
                Bet.client_id == client_id,
                Bet.sync_id == report_sync.id
            ).group_by(Bet.match_id, Bet.match_number, Bet.fixture_id).all()
            
            for match_id, match_number, fixture_id, match_datetime, total_bets, total_payin in bets_by_match:
                # Calculate winning/losing/pending bets from bet details
                winning_bets = 0
                losing_bets = 0
                pending_bets = 0
                total_payout = 0.0
                
                bet_details_query = db.session.query(
                    BetDetail.result,
                    func.count(BetDetail.id).label('count'),
                    func.sum(BetDetail.win_amount).label('total_win')
                ).join(Bet).filter(
                    Bet.client_id == client_id,
                    Bet.match_id == match_id,
                    Bet.sync_id == report_sync.id
                ).group_by(BetDetail.result)
                
                for result, count, total_win in bet_details_query.all():
                    if result == 'won':
                        winning_bets = count
                        total_payout += total_win or 0.0
                    elif result == 'lost':
                        losing_bets = count
                    elif result == 'pending':
                        pending_bets = count
                
                balance = total_payin - total_payout
                
                # Check if MatchReport already exists for this match and client
                existing_match_report = MatchReport.query.filter_by(
                    client_id=client_id,
                    match_id=match_id
                ).first()
                
                if existing_match_report:
                    # Update existing MatchReport
                    existing_match_report.sync_id = report_sync.id
                    existing_match_report.total_bets = total_bets
                    existing_match_report.winning_bets = winning_bets
                    existing_match_report.losing_bets = losing_bets
                    existing_match_report.pending_bets = pending_bets
                    existing_match_report.total_payin = total_payin
                    existing_match_report.total_payout = total_payout
                    existing_match_report.balance = balance
                    existing_match_report.cap_compensation_balance = cap_compensation_balance
                    logger.info(f"Updated existing MatchReport for match {match_id}")
                else:
                    # Create new MatchReport
                    match_report = MatchReport(
                        sync_id=report_sync.id,
                        client_id=client_id,
                        client_token_name=client_token_name,
                        match_id=match_id,
                        match_number=match_number,
                        fixture_id=fixture_id,
                        match_datetime=match_datetime,
                        total_bets=total_bets,
                        winning_bets=winning_bets,
                        losing_bets=losing_bets,
                        pending_bets=pending_bets,
                        total_payin=total_payin,
                        total_payout=total_payout,
                        balance=balance,
                        cap_compensation_balance=cap_compensation_balance
                    )
                    db.session.add(match_report)
                    logger.info(f"Created new MatchReport for match {match_id}")
                
                match_reports_count += 1
        
        # Commit all changes
        db.session.commit()
        
        # Log successful sync
        log_entry = ReportSyncLog(
            sync_id=data['sync_id'],
            client_id=client_id,
            sync_timestamp=sync_timestamp,
            operation_type='new_sync',
            status='success',
            bets_processed=bets_count,
            bets_new=bets_new,
            bets_duplicate=bets_duplicate,
            stats_processed=stats_count,
            stats_new=stats_new,
            stats_updated=stats_updated,
            total_payin=summary['total_payin'],
            total_payout=summary['total_payout'],
            net_profit=summary['net_profit'],
            total_bets=summary['total_bets'],
            total_matches=summary['total_matches'],
            ip_address=ip_address,
            user_agent=user_agent,
            request_size=request_size,
            processing_time_ms=int((time.time() - start_time) * 1000)
        )
        db.session.add(log_entry)
        db.session.commit()
        
        # Update API token last used
        if api_token:
            api_token.update_last_used(request.remote_addr)
        
        logger.info(f"Report sync {data['sync_id']} completed successfully from client {client_id}: {bets_count} bets, {stats_count} stats ({stats_new} new, {stats_updated} updated)")
        
        # Check if this was the first sync for this client (server database was empty)
        # Check if ANY records exist for this client across all report tables
        total_bets_for_client = Bet.query.filter_by(client_id=client_id).count()
        total_stats_for_client = ExtractionStats.query.filter_by(client_id=client_id).count()
        total_match_reports_for_client = MatchReport.query.filter_by(client_id=client_id).count()
        
        # If no records exist in any table, this is effectively a first sync and requires full sync
        requires_full_sync = (total_bets_for_client == 0 and total_stats_for_client == 0 and total_match_reports_for_client == 0)
        
        return jsonify({
            'success': True,
            'synced_count': bets_count + stats_count,
            'message': 'Report data synchronized successfully',
            'requires_full_sync': requires_full_sync,
            'server_timestamp': datetime.utcnow().isoformat(),
            'client_id_used': client_id  # Include the client_id that was used
        }), 200
        
    except Exception as e:
        db.session.rollback()
        logger.error(f"Reports sync error: {str(e)}")
        return jsonify({
            'success': False,
            'error': 'Internal server error',
            'details': 'An unexpected error occurred while processing sync'
        }), 500

@bp.route('/reports/last-sync', methods=['GET'])
@csrf.exempt
def api_get_last_sync():
    """Get last sync information for a client - allows client to verify server state"""
    try:
        from app.models import ReportSync, ReportSyncLog, APIToken, ClientActivity, Bet, ExtractionStats, MatchReport
        from app.auth.jwt_utils import validate_api_token, extract_token_from_request
        from sqlalchemy import desc
        
        # Authenticate using API token
        token = extract_token_from_request()
        if not token:
            return jsonify({
                'success': False,
                'error': 'Authentication required',
                'details': 'API token required'
            }), 401
        
        user, api_token = validate_api_token(token)
        if not user or not user.is_active:
            return jsonify({
                'success': False,
                'error': 'Authentication failed',
                'details': 'Invalid or expired API token'
            }), 401
        
        # Get client_id from query parameter (optional)
        client_id = request.args.get('client_id')
        
        logger.info(f"API get last sync: user={user.username}, client_id={client_id}, api_token={api_token.name if api_token else 'None'}")
        
        # If client_id is not provided, derive it from ClientActivity using API token
        if not client_id:
            # Get most recent client activity for this API token to find rustdesk_id
            if api_token:
                most_recent_client = ClientActivity.query.filter_by(api_token_id=api_token.id)\
                    .order_by(desc(ClientActivity.last_seen))\
                    .first()
                
                if most_recent_client:
                    client_id = most_recent_client.rustdesk_id
                    logger.info(f"API get last sync: Auto-detected client_id={client_id} from API token {api_token.name} (ID: {api_token.id}) via ClientActivity")
                else:
                    logger.warning(f"API get last sync: No client activity found for API token {api_token.name}")
                    return jsonify({
                        'success': True,
                        'message': 'No client activity found for this API token',
                        'client_id': None,
                        'last_sync_id': None,
                        'last_sync_timestamp': None,
                        'last_sync_type': None,
                        'total_syncs': 0,
                        'requires_full_sync': True,
                        'server_timestamp': datetime.utcnow().isoformat()
                    }), 200
            else:
                logger.warning(f"API get last sync: No API token provided")
                return jsonify({
                    'success': False,
                    'error': 'No API token provided',
                    'details': 'Cannot identify client without API token'
                }), 401
        
        # Verify user has access to this specific client
        if not user.is_admin:
            # Check if this client belongs to user's API tokens
            user_token_ids = [t.id for t in APIToken.query.filter_by(user_id=user.id).all()]
            if user_token_ids:
                client_ids = [c.rustdesk_id for c in ClientActivity.query.filter(
                    ClientActivity.api_token_id.in_(user_token_ids)
                ).all()]
                if client_id not in client_ids:
                    return jsonify({
                        'success': False,
                        'error': 'Access denied',
                        'details': 'You do not have access to this client'
                    }), 403
            else:
                return jsonify({
                    'success': False,
                    'error': 'Access denied',
                    'details': 'You do not have access to this client'
                }), 403
        
        logger.info(f"API get last sync: Querying specific client {client_id}")
        
        # Get most recent sync for this client
        last_sync = ReportSync.query.filter_by(client_id=client_id)\
            .order_by(desc(ReportSync.sync_timestamp))\
            .first()
        
        if not last_sync:
            # Check if ANY records exist for this client across all report tables
            total_bets_for_client = Bet.query.filter_by(client_id=client_id).count()
            total_stats_for_client = ExtractionStats.query.filter_by(client_id=client_id).count()
            total_match_reports_for_client = MatchReport.query.filter_by(client_id=client_id).count()
            
            # If no records exist in any table, signal that full sync is required
            requires_full_sync = (total_bets_for_client == 0 and total_stats_for_client == 0 and total_match_reports_for_client == 0)
            
            logger.info(f"API get last sync: Client {client_id} - NO sync records found, bets={total_bets_for_client}, stats={total_stats_for_client}, match_reports={total_match_reports_for_client}, requires_full_sync={requires_full_sync}")
            
            response_data = {
                'success': True,
                'message': 'No sync records found for this client',
                'client_id': client_id,
                'last_sync_id': None,
                'last_sync_timestamp': None,
                'last_sync_type': None,
                'total_syncs': 0,
                'requires_full_sync': requires_full_sync,  # Signal to client that full sync is needed
                'server_timestamp': datetime.utcnow().isoformat()
            }
            
            logger.info(f"API get last sync: Client {client_id} - Returning response: {json.dumps(response_data, indent=2)}")
            
            return jsonify(response_data), 200
        
        # Get total sync count for this client
        total_syncs = ReportSync.query.filter_by(client_id=client_id).count()
        
        logger.info(f"API get last sync: Client {client_id} - Found {total_syncs} sync(s), last_sync_id={last_sync.sync_id}")
        
        # Get most recent sync log for additional details
        last_sync_log = ReportSyncLog.query.filter_by(client_id=client_id)\
            .order_by(desc(ReportSyncLog.created_at))\
            .first()
        
        # Build response
        response_data = {
            'success': True,
            'client_id': client_id,
            'last_sync_id': last_sync.sync_id,
            'last_sync_timestamp': last_sync.sync_timestamp.isoformat() if last_sync.sync_timestamp else None,
            'last_sync_type': last_sync.sync_type if hasattr(last_sync, 'sync_type') else 'unknown',
            'last_date_range': last_sync.date_range,
            'last_start_date': last_sync.start_date.isoformat() if last_sync.start_date else None,
            'last_end_date': last_sync.end_date.isoformat() if last_sync.end_date else None,
            'total_syncs': total_syncs,
            'requires_full_sync': False,
            'last_sync_summary': {
                'total_payin': float(last_sync.total_payin) if last_sync.total_payin else 0.0,
                'total_payout': float(last_sync.total_payout) if last_sync.total_payout else 0.0,
                'net_profit': float(last_sync.net_profit) if last_sync.net_profit else 0.0,
                'total_bets': last_sync.total_bets,
                'total_matches': last_sync.total_matches,
                'cap_compensation_balance': float(last_sync.cap_compensation_balance) if last_sync.cap_compensation_balance else 0.0
            },
            'server_timestamp': datetime.utcnow().isoformat()
        }
        
        logger.info(f"API get last sync: Client {client_id} - Returning response with requires_full_sync=False (has sync records)")
        logger.info(f"API get last sync: Client {client_id} - Full response: {json.dumps(response_data, indent=2)}")
        
        # Add sync log details if available
        if last_sync_log:
            response_data['last_sync_log'] = {
                'operation_type': last_sync_log.operation_type,
                'status': last_sync_log.status,
                'bets_processed': last_sync_log.bets_processed,
                'bets_new': last_sync_log.bets_new,
                'bets_duplicate': last_sync_log.bets_duplicate,
                'stats_processed': last_sync_log.stats_processed,
                'stats_new': last_sync_log.stats_new,
                'stats_updated': last_sync_log.stats_updated,
                'created_at': last_sync_log.created_at.isoformat() if last_sync_log.created_at else None
            }
        
        logger.info(f"Last sync query for client {client_id} by user {user.username}: last_sync_id={last_sync.sync_id}")
        
        return jsonify(response_data), 200
        
    except Exception as e:
        logger.error(f"API get last sync error: {str(e)}")
        return jsonify({
            'success': False,
            'error': 'Internal server error',
            'details': 'Failed to retrieve last sync information'
        }), 500