import logging
from flask import render_template, request, jsonify, redirect, url_for, flash, current_app
from flask_login import login_required, current_user
from app.main import bp
from app import db
from app.models import Match, FileUpload, User, SystemLog, MatchOutcome
from app.upload.file_handler import get_file_upload_handler
from app.upload.fixture_parser import get_fixture_parser
from app.utils.security import require_admin, require_active_user
from app.database import get_db_manager

logger = logging.getLogger(__name__)

@bp.route('/')
def index():
    """Home page"""
    if current_user.is_authenticated:
        return redirect(url_for('main.dashboard'))
    return render_template('main/index.html')

@bp.route('/dashboard')
@login_required
@require_active_user
def dashboard():
    """Main dashboard"""
    try:
        # Get user statistics
        user_matches = Match.query.filter_by(created_by=current_user.id).count()
        user_uploads = FileUpload.query.filter_by(uploaded_by=current_user.id).count()
        
        # Get recent matches
        recent_matches = Match.query.filter_by(created_by=current_user.id)\
            .order_by(Match.created_at.desc()).limit(5).all()
        
        # Get recent uploads
        recent_uploads = FileUpload.query.filter_by(uploaded_by=current_user.id)\
            .order_by(FileUpload.created_at.desc()).limit(5).all()
        
        # Get system statistics (for admins)
        system_stats = {}
        if current_user.is_admin:
            db_manager = get_db_manager()
            system_stats = db_manager.get_database_stats()
            
            file_handler = get_file_upload_handler()
            upload_stats = file_handler.get_upload_statistics()
            system_stats.update(upload_stats)
        
        return render_template('main/dashboard.html',
                             user_matches=user_matches,
                             user_uploads=user_uploads,
                             recent_matches=recent_matches,
                             recent_uploads=recent_uploads,
                             system_stats=system_stats)
    
    except Exception as e:
        logger.error(f"Dashboard error: {str(e)}")
        flash('Error loading dashboard', 'error')
        return render_template('main/dashboard.html')

@bp.route('/matches')
@login_required
@require_active_user
def matches():
    """List matches with pagination and filtering"""
    try:
        page = request.args.get('page', 1, type=int)
        per_page = min(request.args.get('per_page', 20, type=int), 100)
        
        # Filtering options
        status_filter = request.args.get('status')
        search_query = request.args.get('search', '').strip()
        
        # Base query
        if current_user.is_admin:
            query = Match.query
        else:
            query = Match.query.filter_by(created_by=current_user.id)
        
        # Apply filters
        if status_filter == 'active':
            query = query.filter_by(active_status=True)
        elif status_filter == 'pending':
            query = query.filter_by(active_status=False)
        elif status_filter == 'zip_pending':
            query = query.filter_by(zip_upload_status='pending')
        elif status_filter == 'zip_completed':
            query = query.filter_by(zip_upload_status='completed')
        
        # Search functionality
        if search_query:
            search_pattern = f"%{search_query}%"
            query = query.filter(
                db.or_(
                    Match.fighter1_township.ilike(search_pattern),
                    Match.fighter2_township.ilike(search_pattern),
                    Match.venue_kampala_township.ilike(search_pattern),
                    Match.match_number.like(search_pattern)
                )
            )
        
        # Pagination
        matches_pagination = query.order_by(Match.created_at.desc()).paginate(
            page=page, per_page=per_page, error_out=False
        )
        
        return render_template('main/matches.html',
                             matches=matches_pagination.items,
                             pagination=matches_pagination,
                             status_filter=status_filter,
                             search_query=search_query)
    
    except Exception as e:
        logger.error(f"Matches list error: {str(e)}")
        flash('Error loading matches', 'error')
        return render_template('main/matches.html', matches=[], pagination=None)

@bp.route('/match/<int:id>')
@login_required
@require_active_user
def match_detail(id):
    """Match detail page"""
    try:
        if current_user.is_admin:
            match = Match.query.get_or_404(id)
        else:
            match = Match.query.filter_by(id=id, created_by=current_user.id).first_or_404()
        
        # Get match outcomes
        outcomes = match.outcomes.all()
        
        # Get associated uploads
        uploads = FileUpload.query.filter_by(match_id=id).all()
        
        return render_template('main/match_detail.html',
                             match=match,
                             outcomes=outcomes,
                             uploads=uploads)
    
    except Exception as e:
        logger.error(f"Match detail error: {str(e)}")
        flash('Error loading match details', 'error')
        return redirect(url_for('main.matches'))

@bp.route('/uploads')
@login_required
@require_active_user
def uploads():
    """List uploads with pagination and filtering"""
    try:
        page = request.args.get('page', 1, type=int)
        per_page = min(request.args.get('per_page', 20, type=int), 100)
        
        # Filtering options
        file_type_filter = request.args.get('file_type')
        status_filter = request.args.get('status')
        
        # Base query
        if current_user.is_admin:
            query = FileUpload.query
        else:
            query = FileUpload.query.filter_by(uploaded_by=current_user.id)
        
        # Apply filters
        if file_type_filter:
            query = query.filter_by(file_type=file_type_filter)
        
        if status_filter:
            query = query.filter_by(upload_status=status_filter)
        
        # Pagination
        uploads_pagination = query.order_by(FileUpload.created_at.desc()).paginate(
            page=page, per_page=per_page, error_out=False
        )
        
        return render_template('main/uploads.html',
                             uploads=uploads_pagination.items,
                             pagination=uploads_pagination,
                             file_type_filter=file_type_filter,
                             status_filter=status_filter)
    
    except Exception as e:
        logger.error(f"Uploads list error: {str(e)}")
        flash('Error loading uploads', 'error')
        return render_template('main/uploads.html', uploads=[], pagination=None)

@bp.route('/statistics')
@login_required
@require_active_user
def statistics():
    """Statistics page"""
    try:
        # Get file handler and parser statistics
        file_handler = get_file_upload_handler()
        fixture_parser = get_fixture_parser()
        
        upload_stats = file_handler.get_upload_statistics()
        parsing_stats = fixture_parser.get_parsing_statistics()
        
        # User-specific statistics
        user_stats = {
            'total_uploads': FileUpload.query.filter_by(uploaded_by=current_user.id).count(),
            'total_matches': Match.query.filter_by(created_by=current_user.id).count(),
            'active_matches': Match.query.filter_by(
                created_by=current_user.id, active_status=True
            ).count(),
            'pending_zip_uploads': Match.query.filter_by(
                created_by=current_user.id, zip_upload_status='pending'
            ).count()
        }
        
        # System statistics (admin only)
        system_stats = {}
        if current_user.is_admin:
            db_manager = get_db_manager()
            system_stats = db_manager.get_database_stats()
        
        return render_template('main/statistics.html',
                             upload_stats=upload_stats,
                             parsing_stats=parsing_stats,
                             user_stats=user_stats,
                             system_stats=system_stats)
    
    except Exception as e:
        logger.error(f"Statistics error: {str(e)}")
        flash('Error loading statistics', 'error')
        return render_template('main/statistics.html')

@bp.route('/admin')
@login_required
@require_admin
def admin_panel():
    """Admin panel"""
    try:
        # Get system overview
        db_manager = get_db_manager()
        system_stats = db_manager.get_database_stats()
        
        # Get recent system logs
        recent_logs = SystemLog.query.order_by(SystemLog.created_at.desc()).limit(20).all()
        
        # Get user statistics
        total_users = User.query.count()
        active_users = User.query.filter_by(is_active=True).count()
        admin_users = User.query.filter_by(is_admin=True).count()
        
        user_stats = {
            'total_users': total_users,
            'active_users': active_users,
            'admin_users': admin_users,
            'inactive_users': total_users - active_users
        }
        
        # Get upload statistics
        file_handler = get_file_upload_handler()
        upload_stats = file_handler.get_upload_statistics()
        
        return render_template('main/admin.html',
                             system_stats=system_stats,
                             user_stats=user_stats,
                             upload_stats=upload_stats,
                             recent_logs=recent_logs)
    
    except Exception as e:
        logger.error(f"Admin panel error: {str(e)}")
        flash('Error loading admin panel', 'error')
        return render_template('main/admin.html')

@bp.route('/admin/users')
@login_required
@require_admin
def admin_users():
    """Admin user management"""
    try:
        page = request.args.get('page', 1, type=int)
        per_page = min(request.args.get('per_page', 20, type=int), 100)
        
        search_query = request.args.get('search', '').strip()
        status_filter = request.args.get('status')
        
        # Base query
        query = User.query
        
        # Apply filters
        if status_filter == 'active':
            query = query.filter_by(is_active=True)
        elif status_filter == 'inactive':
            query = query.filter_by(is_active=False)
        elif status_filter == 'admin':
            query = query.filter_by(is_admin=True)
        
        # Search functionality
        if search_query:
            search_pattern = f"%{search_query}%"
            query = query.filter(
                db.or_(
                    User.username.ilike(search_pattern),
                    User.email.ilike(search_pattern)
                )
            )
        
        # Pagination
        users_pagination = query.order_by(User.created_at.desc()).paginate(
            page=page, per_page=per_page, error_out=False
        )
        
        return render_template('main/admin_users.html',
                             users=users_pagination.items,
                             pagination=users_pagination,
                             search_query=search_query,
                             status_filter=status_filter)
    
    except Exception as e:
        logger.error(f"Admin users error: {str(e)}")
        flash('Error loading users', 'error')
        return render_template('main/admin_users.html', users=[], pagination=None)

@bp.route('/admin/logs')
@login_required
@require_admin
def admin_logs():
    """Admin system logs"""
    try:
        page = request.args.get('page', 1, type=int)
        per_page = min(request.args.get('per_page', 50, type=int), 200)
        
        level_filter = request.args.get('level')
        module_filter = request.args.get('module')
        
        # Base query
        query = SystemLog.query
        
        # Apply filters
        if level_filter:
            query = query.filter_by(level=level_filter)
        
        if module_filter:
            query = query.filter_by(module=module_filter)
        
        # Pagination
        logs_pagination = query.order_by(SystemLog.created_at.desc()).paginate(
            page=page, per_page=per_page, error_out=False
        )
        
        # Get available modules for filter
        modules = db.session.query(SystemLog.module).distinct().all()
        modules = [m[0] for m in modules if m[0]]
        
        return render_template('main/admin_logs.html',
                             logs=logs_pagination.items,
                             pagination=logs_pagination,
                             level_filter=level_filter,
                             module_filter=module_filter,
                             modules=modules)
    
    except Exception as e:
        logger.error(f"Admin logs error: {str(e)}")
        flash('Error loading logs', 'error')
        return render_template('main/admin_logs.html', logs=[], pagination=None)

@bp.route('/health')
def health_check():
    """Health check endpoint"""
    try:
        # Test database connection
        db_manager = get_db_manager()
        db_healthy = db_manager.test_connection()
        
        # Get basic statistics
        stats = db_manager.get_database_stats()
        
        health_status = {
            'status': 'healthy' if db_healthy else 'unhealthy',
            'database': 'connected' if db_healthy else 'disconnected',
            'timestamp': db.func.now(),
            'statistics': stats
        }
        
        status_code = 200 if db_healthy else 503
        return jsonify(health_status), status_code
    
    except Exception as e:
        logger.error(f"Health check error: {str(e)}")
        return jsonify({
            'status': 'unhealthy',
            'error': str(e),
            'timestamp': db.func.now()
        }), 503

@bp.route('/api/dashboard-data')
@login_required
@require_active_user
def api_dashboard_data():
    """API endpoint for dashboard data"""
    try:
        # User statistics
        user_stats = {
            'total_matches': Match.query.filter_by(created_by=current_user.id).count(),
            'active_matches': Match.query.filter_by(
                created_by=current_user.id, active_status=True
            ).count(),
            'total_uploads': FileUpload.query.filter_by(uploaded_by=current_user.id).count(),
            'pending_zip_uploads': Match.query.filter_by(
                created_by=current_user.id, zip_upload_status='pending'
            ).count()
        }
        
        # Recent activity
        recent_matches = Match.query.filter_by(created_by=current_user.id)\
            .order_by(Match.created_at.desc()).limit(5).all()
        
        recent_uploads = FileUpload.query.filter_by(uploaded_by=current_user.id)\
            .order_by(FileUpload.created_at.desc()).limit(5).all()
        
        return jsonify({
            'user_stats': user_stats,
            'recent_matches': [match.to_dict() for match in recent_matches],
            'recent_uploads': [upload.to_dict() for upload in recent_uploads]
        }), 200
    
    except Exception as e:
        logger.error(f"Dashboard API error: {str(e)}")
        return jsonify({'error': 'Failed to load dashboard data'}), 500