"""
Test script to verify that incremental reports sync updates existing MatchReport records
"""
import sys
import os
sys.path.insert(0, os.path.dirname(os.path.abspath(__file__)))

from datetime import datetime, timedelta
import uuid
from app import create_app, db
from app.models import ReportSync, Bet, BetDetail, ExtractionStats, MatchReport, ClientActivity, APIToken, User

def create_test_data():
    """Create test data to verify the fix"""
    app = create_app()
    app.config['SQLALCHEMY_DATABASE_URI'] = 'sqlite:///:memory:'
    app.config['SQLALCHEMY_TRACK_MODIFICATIONS'] = False
    app.config['TESTING'] = True
    
    with app.app_context():
        db.create_all()
        
        # Create test user
        user = User(username='testuser', email='test@example.com', is_active=True, is_admin=True)
        user.set_password('testpassword')
        db.session.add(user)
        db.session.commit()
        
        # Create test API token
        api_token = user.generate_api_token('Test Token')
        db.session.commit()
        
        # Create test client activity
        client_activity = ClientActivity(
            api_token_id=api_token.id,
            rustdesk_id='test_client_001',
            last_seen=datetime.utcnow()
        )
        db.session.add(client_activity)
        db.session.commit()
        
        # Create first sync (full sync)
        sync1_data = {
            'sync_id': f'sync_test_1_{uuid.uuid4()}',
            'client_id': 'test_client_001',
            'sync_timestamp': datetime.utcnow().isoformat(),
            'date_range': 'all',
            'start_date': (datetime.utcnow() - timedelta(days=7)).isoformat(),
            'end_date': datetime.utcnow().isoformat(),
            'bets': [
                {
                    'uuid': str(uuid.uuid4()),
                    'fixture_id': 'fixture_test_001',
                    'bet_datetime': (datetime.utcnow() - timedelta(hours=2)).isoformat(),
                    'paid': False,
                    'paid_out': False,
                    'total_amount': 500.00,
                    'bet_count': 1,
                    'details': [
                        {
                            'match_id': 123,
                            'match_number': 1,
                            'outcome': 'WIN1',
                            'amount': 500.00,
                            'win_amount': 0.00,
                            'result': 'pending'
                        }
                    ]
                }
            ],
            'extraction_stats': [
                {
                    'match_id': 123,
                    'fixture_id': 'fixture_test_001',
                    'match_datetime': (datetime.utcnow() - timedelta(hours=1)).isoformat(),
                    'total_bets': 10,
                    'total_amount_collected': 2000.00,
                    'total_redistributed': 1500.00,
                    'actual_result': 'WIN1',
                    'extraction_result': 'WIN1',
                    'cap_applied': False,
                    'under_bets': 5,
                    'under_amount': 1000.00,
                    'over_bets': 5,
                    'over_amount': 1000.00,
                    'result_breakdown': {
                        'WIN1': {'bets': 3, 'amount': 600.00},
                        'X1': {'bets': 2, 'amount': 400.00},
                        'WIN2': {'bets': 5, 'amount': 1000.00}
                    }
                }
            ],
            'cap_compensation_balance': 0.00,
            'summary': {
                'total_payin': 2000.00,
                'total_payout': 1500.00,
                'net_profit': 500.00,
                'total_bets': 10,
                'total_matches': 1
            }
        }
        
        return app, user, api_token, client_activity, sync1_data

def test_matchreport_update():
    """Test that existing MatchReport records are updated in incremental sync"""
    print("=" * 80)
    print("Testing MatchReport Update in Incremental Sync")
    print("=" * 80)
    
    app, user, api_token, client_activity, sync1_data = create_test_data()
    
    with app.app_context():
        # Process first sync (full sync)
        print("\n1. Processing first sync (full sync)...")
        
        # Create report sync record
        sync1 = ReportSync(
            sync_id=sync1_data['sync_id'],
            client_id=sync1_data['client_id'],
            sync_timestamp=datetime.fromisoformat(sync1_data['sync_timestamp']),
            date_range=sync1_data['date_range'],
            start_date=datetime.fromisoformat(sync1_data['start_date']),
            end_date=datetime.fromisoformat(sync1_data['end_date']),
            total_payin=sync1_data['summary']['total_payin'],
            total_payout=sync1_data['summary']['total_payout'],
            net_profit=sync1_data['summary']['net_profit'],
            total_bets=sync1_data['summary']['total_bets'],
            total_matches=sync1_data['summary']['total_matches'],
            cap_compensation_balance=sync1_data['cap_compensation_balance']
        )
        db.session.add(sync1)
        db.session.commit()
        
        # Process bets
        for bet_data in sync1_data['bets']:
            bet = Bet(
                uuid=bet_data['uuid'],
                sync_id=sync1.id,
                client_id=sync1_data['client_id'],
                fixture_id=bet_data['fixture_id'],
                match_id=bet_data['details'][0]['match_id'],
                match_number=bet_data['details'][0]['match_number'],
                bet_datetime=datetime.fromisoformat(bet_data['bet_datetime']),
                paid=bet_data['paid'],
                paid_out=bet_data['paid_out'],
                total_amount=bet_data['total_amount'],
                bet_count=bet_data['bet_count']
            )
            db.session.add(bet)
            db.session.flush()
            
            for detail_data in bet_data['details']:
                bet_detail = BetDetail(
                    bet_id=bet.id,
                    match_id=detail_data['match_id'],
                    match_number=detail_data['match_number'],
                    outcome=detail_data['outcome'],
                    amount=detail_data['amount'],
                    win_amount=detail_data['win_amount'],
                    result=detail_data['result']
                )
                db.session.add(bet_detail)
        
        # Process extraction stats and create MatchReport
        for stats_data in sync1_data['extraction_stats']:
            # Create extraction stats
            extraction_stats = ExtractionStats(
                sync_id=sync1.id,
                client_id=sync1_data['client_id'],
                match_id=stats_data['match_id'],
                match_number=sync1_data['bets'][0]['details'][0]['match_number'],
                fixture_id=stats_data['fixture_id'],
                match_datetime=datetime.fromisoformat(stats_data['match_datetime']),
                total_bets=stats_data['total_bets'],
                total_amount_collected=stats_data['total_amount_collected'],
                total_redistributed=stats_data['total_redistributed'],
                actual_result=stats_data['actual_result'],
                extraction_result=stats_data['extraction_result'],
                cap_applied=stats_data['cap_applied'],
                under_bets=stats_data['under_bets'],
                under_amount=stats_data['under_amount'],
                over_bets=stats_data['over_bets'],
                over_amount=stats_data['over_amount'],
                result_breakdown=stats_data['result_breakdown']
            )
            db.session.add(extraction_stats)
            
            # Create MatchReport (this is what we fixed)
            match_report = MatchReport(
                sync_id=sync1.id,
                client_id=sync1_data['client_id'],
                client_token_name=api_token.name,
                match_id=stats_data['match_id'],
                match_number=sync1_data['bets'][0]['details'][0]['match_number'],
                fixture_id=stats_data['fixture_id'],
                match_datetime=datetime.fromisoformat(stats_data['match_datetime']),
                total_bets=stats_data['total_bets'],
                winning_bets=3,
                losing_bets=5,
                pending_bets=2,
                total_payin=stats_data['total_amount_collected'],
                total_payout=stats_data['total_redistributed'],
                balance=stats_data['total_amount_collected'] - stats_data['total_redistributed'],
                actual_result=stats_data['actual_result'],
                extraction_result=stats_data['extraction_result'],
                cap_applied=stats_data['cap_applied'],
                cap_compensation_balance=sync1_data['cap_compensation_balance'],
                under_bets=stats_data['under_bets'],
                under_amount=stats_data['under_amount'],
                over_bets=stats_data['over_bets'],
                over_amount=stats_data['over_amount'],
                result_breakdown=stats_data['result_breakdown']
            )
            db.session.add(match_report)
        
        db.session.commit()
        
        # Verify MatchReport was created
        match_report_count = MatchReport.query.count()
        print(f"   MatchReport count after first sync: {match_report_count}")
        assert match_report_count == 1, "Should have created 1 MatchReport"
        
        original_match_report = MatchReport.query.first()
        print(f"   Original MatchReport sync_id: {original_match_report.sync_id}")
        print(f"   Original MatchReport total_bets: {original_match_report.total_bets}")
        print(f"   Original MatchReport winning_bets: {original_match_report.winning_bets}")
        
        # Create incremental sync with updated data
        print("\n2. Creating incremental sync with updated data...")
        sync2_data = sync1_data.copy()
        sync2_data['sync_id'] = f'sync_test_2_{uuid.uuid4()}'
        sync2_data['sync_timestamp'] = (datetime.utcnow() + timedelta(minutes=10)).isoformat()
        sync2_data['extraction_stats'][0]['total_bets'] = 15  # Updated value
        sync2_data['extraction_stats'][0]['total_amount_collected'] = 2500.00  # Updated value
        sync2_data['extraction_stats'][0]['total_redistributed'] = 1800.00  # Updated value
        sync2_data['summary']['total_bets'] = 15
        sync2_data['summary']['total_payin'] = 2500.00
        sync2_data['summary']['total_payout'] = 1800.00
        sync2_data['summary']['net_profit'] = 700.00
        
        # Now simulate processing the incremental sync through the API endpoint
        from app.api.routes import api_reports_sync
        from flask import Flask, request, jsonify
        
        # Create a test client
        test_client = app.test_client()
        
        # Set up headers with authentication
        headers = {
            'Content-Type': 'application/json',
            'Authorization': f'Bearer {api_token.plain_token}'
        }
        
        # Send the sync request
        response = test_client.post(
            '/api/reports/sync',
            json=sync2_data,
            headers=headers
        )
        
        print(f"   API Response status code: {response.status_code}")
        assert response.status_code == 200, "Sync should succeed"
        
        # Verify the response
        response_json = response.get_json()
        print(f"   Response success: {response_json['success']}")
        assert response_json['success'] == True, "Sync should be successful"
        
        print(f"   Synced count: {response_json['synced_count']}")
        
        # Check MatchReport was updated, not duplicated
        print("\n3. Verifying MatchReport was updated...")
        match_report_count = MatchReport.query.count()
        print(f"   MatchReport count after incremental sync: {match_report_count}")
        assert match_report_count == 1, "Should not create duplicate MatchReport"
        
        updated_match_report = MatchReport.query.first()
        print(f"   Updated MatchReport sync_id: {updated_match_report.sync_id}")
        print(f"   Updated MatchReport total_bets: {updated_match_report.total_bets}")
        assert updated_match_report.total_bets == 15, "Should update total_bets"
        print(f"   Updated MatchReport total_payin: {updated_match_report.total_payin}")
        assert updated_match_report.total_payin == 2500.00, "Should update total_payin"
        print(f"   Updated MatchReport total_payout: {updated_match_report.total_payout}")
        assert updated_match_report.total_payout == 1800.00, "Should update total_payout"
        
        print("\n✓ SUCCESS: MatchReport updated correctly in incremental sync!")
        
        # Cleanup
        db.session.remove()
        db.drop_all()

if __name__ == "__main__":
    test_matchreport_update()