#!/usr/bin/env python3
"""
Test script to verify accumulated_shortfall and cap_percentage implementation
"""

import sys
import os
sys.path.insert(0, os.path.dirname(os.path.abspath(__file__)))

from app import create_app, db
from app.models import ExtractionStats, MatchReport, ReportSync
from datetime import datetime

def test_accumulated_shortfall():
    """Test accumulated_shortfall field in models"""
    print("Testing accumulated_shortfall implementation...")
    
    app = create_app()
    
    with app.app_context():
        # Test ExtractionStats model
        print("\n1. Testing ExtractionStats model...")
        try:
            # Check if accumulated_shortfall column exists
            columns = [col.name for col in ExtractionStats.__table__.columns]
            if 'accumulated_shortfall' in columns:
                print("   ✓ accumulated_shortfall column exists in ExtractionStats")
            else:
                print("   ✗ accumulated_shortfall column NOT found in ExtractionStats")
                return False
        except Exception as e:
            print(f"   ✗ Error checking ExtractionStats: {e}")
            return False
        
        # Test MatchReport model
        print("\n2. Testing MatchReport model...")
        try:
            columns = [col.name for col in MatchReport.__table__.columns]
            if 'accumulated_shortfall' in columns:
                print("   ✓ accumulated_shortfall column exists in MatchReport")
            else:
                print("   ✗ accumulated_shortfall column NOT found in MatchReport")
                return False
        except Exception as e:
            print(f"   ✗ Error checking MatchReport: {e}")
            return False
        
        # Test to_dict methods include accumulated_shortfall
        print("\n3. Testing to_dict methods...")
        try:
            # Create a dummy ExtractionStats object
            dummy_stats = ExtractionStats(
                sync_id=1,
                client_id='test_client',
                match_id=1,
                match_number=1,
                fixture_id='test_fixture',
                match_datetime=datetime.utcnow(),
                total_bets=10,
                total_amount_collected=100.00,
                total_redistributed=90.00,
                actual_result='under',
                extraction_result='under',
                accumulated_shortfall=50.00
            )
            stats_dict = dummy_stats.to_dict()
            if 'accumulated_shortfall' in stats_dict:
                print("   ✓ ExtractionStats.to_dict() includes accumulated_shortfall")
            else:
                print("   ✗ ExtractionStats.to_dict() missing accumulated_shortfall")
                return False
            
            # Create a dummy MatchReport object
            dummy_report = MatchReport(
                sync_id=1,
                client_id='test_client',
                client_token_name='Test Token',
                match_id=1,
                match_number=1,
                fixture_id='test_fixture',
                match_datetime=datetime.utcnow(),
                total_bets=10,
                total_payin=100.00,
                total_payout=90.00,
                balance=10.00,
                actual_result='under',
                extraction_result='under',
                accumulated_shortfall=50.00
            )
            report_dict = dummy_report.to_dict()
            if 'accumulated_shortfall' in report_dict:
                print("   ✓ MatchReport.to_dict() includes accumulated_shortfall")
            else:
                print("   ✗ MatchReport.to_dict() missing accumulated_shortfall")
                return False
        except Exception as e:
            print(f"   ✗ Error testing to_dict: {e}")
            return False
        
        print("\n✓ All tests passed!")
        return True

def test_cap_percentage():
    """Test cap_percentage field in models"""
    print("\nTesting cap_percentage implementation...")
    
    app = create_app()
    
    with app.app_context():
        # Test ExtractionStats model
        print("\n1. Testing ExtractionStats model...")
        try:
            # Check if cap_percentage column exists
            columns = [column.name for column in ExtractionStats.__table__.columns]
            if 'cap_percentage' in columns:
                print("   ✓ cap_percentage column exists in ExtractionStats")
            else:
                print("   ✗ cap_percentage column missing in ExtractionStats")
                return False
            
            # Test to_dict method includes cap_percentage
            dummy_stat = ExtractionStats(
                sync_id=1,
                client_id='test_client',
                match_id=1,
                match_number=1,
                fixture_id='test-fixture',
                match_datetime=datetime.utcnow(),
                total_bets=10,
                total_amount_collected=100.00,
                total_redistributed=90.00,
                actual_result='under',
                extraction_result='under',
                cap_applied=True,
                cap_percentage=15.50,
                accumulated_shortfall=50.00
            )
            stat_dict = dummy_stat.to_dict()
            if 'cap_percentage' in stat_dict:
                print("   ✓ ExtractionStats.to_dict() includes cap_percentage")
                if stat_dict['cap_percentage'] == 15.50:
                    print("   ✓ cap_percentage value is correct")
                else:
                    print(f"   ✗ cap_percentage value is incorrect: {stat_dict['cap_percentage']}")
                    return False
            else:
                print("   ✗ ExtractionStats.to_dict() missing cap_percentage")
                return False
        except Exception as e:
            print(f"   ✗ Error testing to_dict: {e}")
            return False
        
        # Test MatchReport model
        print("\n2. Testing MatchReport model...")
        try:
            # Check if cap_percentage column exists
            columns = [column.name for column in MatchReport.__table__.columns]
            if 'cap_percentage' in columns:
                print("   ✓ cap_percentage column exists in MatchReport")
            else:
                print("   ✗ cap_percentage column missing in MatchReport")
                return False
            
            # Test to_dict method includes cap_percentage
            dummy_report = MatchReport(
                sync_id=1,
                client_id='test_client',
                client_token_name='Test Token',
                match_id=1,
                match_number=1,
                fixture_id='test-fixture',
                match_datetime=datetime.utcnow(),
                total_bets=10,
                winning_bets=5,
                losing_bets=5,
                pending_bets=0,
                total_payin=100.00,
                total_payout=90.00,
                balance=10.00,
                actual_result='under',
                extraction_result='under',
                cap_applied=True,
                cap_percentage=15.50,
                cap_compensation_balance=100.00,
                accumulated_shortfall=50.00
            )
            report_dict = dummy_report.to_dict()
            if 'cap_percentage' in report_dict:
                print("   ✓ MatchReport.to_dict() includes cap_percentage")
                if report_dict['cap_percentage'] == 15.50:
                    print("   ✓ cap_percentage value is correct")
                else:
                    print(f"   ✗ cap_percentage value is incorrect: {report_dict['cap_percentage']}")
                    return False
            else:
                print("   ✗ MatchReport.to_dict() missing cap_percentage")
                return False
        except Exception as e:
            print(f"   ✗ Error testing to_dict: {e}")
            return False
        
        print("\n✓ All cap_percentage tests passed!")
        return True

if __name__ == '__main__':
    success1 = test_accumulated_shortfall()
    success2 = test_cap_percentage()
    
    if success1 and success2:
        print("\n" + "="*50)
        print("✓ ALL TESTS PASSED!")
        print("="*50)
        sys.exit(0)
    else:
        print("\n" + "="*50)
        print("✗ SOME TESTS FAILED!")
        print("="*50)
        sys.exit(1)