# Proxy Support Documentation

## Overview

The mbetterd web dashboard now supports running behind an nginx reverse proxy with the `X-Forwarded-Prefix` header. This allows the application to be served from a subpath (e.g., `/mbetterc`) while maintaining correct URL generation for all links, forms, and JavaScript requests.

## Nginx Configuration

The nginx proxy should be configured with the following headers:

```nginx
# Headers
proxy_set_header Host $host;
proxy_set_header X-Real-IP $remote_addr;
proxy_set_header X-Forwarded-For $proxy_add_x_forwarded_for;
proxy_set_header X-Forwarded-Proto $scheme;
proxy_set_header X-Forwarded-Host $host;
proxy_set_header X-Forwarded-Port $server_port;
proxy_set_header X-Forwarded-Prefix /mbetterc;
proxy_set_header Connection "";
```

## Implementation Details

### 1. Flask ProxyFix Configuration

The application uses Werkzeug's `ProxyFix` middleware to handle proxy headers:

```python
app.wsgi_app = ProxyFix(app.wsgi_app, x_for=1, x_proto=1, x_host=1, x_port=1, x_prefix=1)
```

This configuration:
- `x_for=1`: Trusts `X-Forwarded-For` header (1 proxy)
- `x_proto=1`: Trusts `X-Forwarded-Proto` header (1 proxy)
- `x_host=1`: Trusts `X-Forwarded-Host` header (1 proxy)
- `x_port=1`: Trusts `X-Forwarded-Port` header (1 proxy)
- `x_prefix=1`: Trusts `X-Forwarded-Prefix` header (1 proxy)

### 2. Template Context Processor

A template context processor makes `script_root` and `url_root` available in all templates:

```python
@app.context_processor
def inject_script_root():
    """Make script_root available in all templates for proxy support"""
    return {
        'script_root': request.script_root if request else '',
        'url_root': request.url_root if request else ''
    }
```

### 3. JavaScript URL Helper Functions

The base template includes JavaScript helper functions for proxy-aware URL generation:

```javascript
// Get the script root from Flask (includes X-Forwarded-Prefix if behind proxy)
const SCRIPT_ROOT = {{ request.script_root | tojson | safe }};

// Build a URL with the correct proxy prefix
function buildUrl(path) { ... }

// Make a fetch request with proxy-aware URL
function proxyFetch(path, options = {}) { ... }

// Navigate to a URL with proxy prefix
function navigateTo(path) { ... }

// Get current URL with proxy prefix
function getCurrentUrl() { ... }

// Build URL with query parameters
function buildUrlWithParams(path, params = {}) { ... }
```

## Usage

### In Templates (Jinja2)

Flask's `url_for()` function automatically handles the proxy prefix:

```jinja2
<a href="{{ url_for('main.dashboard') }}">Dashboard</a>
<form action="{{ url_for('upload.upload_fixture') }}" method="POST">
```

### In JavaScript

Use the provided helper functions instead of hardcoded URLs:

```javascript
// Instead of: fetch('/api/dashboard-data')
// Use:
proxyFetch('/api/dashboard-data')

// Instead of: window.location.href = '/clients'
// Use:
navigateTo('/clients')

// Instead of: window.location.href
// Use:
getCurrentUrl()

// Build URLs with parameters:
const url = buildUrlWithParams('/reports', { page: 2, per_page: 20 });
```

## Updated Templates

The following templates have been updated to use proxy-aware URL generation:

- `app/templates/base.html` - Added JavaScript helper functions
- `app/templates/main/admin_settings.html` - Updated fetch calls
- `app/templates/main/admin_users.html` - Updated fetch calls
- `app/templates/main/admin_migrations.html` - Updated fetch calls
- `app/templates/main/match_detail.html` - Updated fetch calls
- `app/templates/main/fixture_detail.html` - Updated fetch calls
- `app/templates/upload/zip.html` - Updated fetch calls
- `app/templates/main/user_tokens.html` - Updated fetch calls
- `app/templates/main/clients.html` - Updated URL building
- `app/templates/main/reports.html` - Updated URL building
- `app/templates/main/client_report_detail.html` - Updated URL building
- `app/templates/main/match_report_detail.html` - Updated URL building
- `app/templates/main/bet_detail.html` - Updated URL building

## Testing

To test the proxy support:

1. Configure nginx with the `X-Forwarded-Prefix` header
2. Access the application through the proxy URL (e.g., `https://example.com/mbetterc`)
3. Verify that:
   - All navigation links work correctly
   - Forms submit to the correct URLs
   - JavaScript fetch requests work properly
   - Pagination links work correctly
   - Export functionality works

## Troubleshooting

### URLs not working behind proxy

1. Verify nginx is sending the `X-Forwarded-Prefix` header
2. Check that `ProxyFix` is configured with `x_prefix=1`
3. Ensure JavaScript code uses `proxyFetch()` instead of `fetch()`

### Static files not loading

1. Verify nginx is configured to serve static files or proxy them correctly
2. Check that the `X-Forwarded-Host` header is set correctly

### Redirects not working

1. Flask's `redirect()` and `url_for()` should work automatically
2. If using manual redirects, use `buildUrl()` or `navigateTo()` functions

## Notes

- The `SCRIPT_ROOT` variable is injected into the JavaScript from Flask's `request.script_root`
- This value automatically includes the `X-Forwarded-Prefix` when behind a proxy
- All new JavaScript code should use the provided helper functions for URL generation
- The implementation is backward compatible - it works with or without a proxy

## References

- [Flask ProxyFix Documentation](https://werkzeug.palletsprojects.com/en/stable/middleware/proxy_fix/)
- [Nginx Proxy Headers](https://nginx.org/en/docs/http/ngx_http_proxy_module.html#proxy_set_header)