# Minimal Prompt: Client-Side Last Sync Query Implementation

## What Changed on Server

Server now has a new endpoint to query last sync information:

**Endpoint**: `GET /api/reports/last-sync?client_id=<client_id>`

**Authentication**: Bearer token (API token)

**Response Format**:
```json
{
  "success": true,
  "client_id": "client_unique_identifier",
  "last_sync_id": "sync_20260201_214327_abc12345",
  "last_sync_timestamp": "2026-02-01T21:43:27.249Z",
  "last_sync_type": "incremental",
  "total_syncs": 25,
  "last_sync_summary": {
    "total_payin": 100000.0,
    "total_payout": 95000.0,
    "net_profit": 5000.0,
    "total_bets": 50,
    "total_matches": 10,
    "cap_compensation_balance": 5000.0
  },
  "server_timestamp": "2026-02-01T21:43:27.249Z"
}
```

## What You Need to Implement

### 1. Add Function to Query Server

```python
def query_server_last_sync(api_token, client_id):
    """Query server for last sync information"""
    import requests
    
    url = "https://your-server.com/api/reports/last-sync"
    headers = {"Authorization": f"Bearer {api_token}"}
    params = {"client_id": client_id}
    
    response = requests.get(url, headers=headers, params=params)
    return response.json()
```

### 2. Call Before Each Sync

```python
# Before performing sync
server_info = query_server_last_sync(api_token, client_id)

if server_info.get('success'):
    last_sync_id = server_info.get('last_sync_id')
    last_sync_time = server_info.get('last_sync_timestamp')
    
    # Compare with your local tracking
    # If mismatch detected, perform full sync instead of incremental
```

### 3. Handle Recovery

If your local tracking is corrupted or lost:

```python
# If no local tracking exists
if not local_tracking_exists():
    # Query server for last sync
    server_info = query_server_last_sync(api_token, client_id)
    
    # Recover local tracking from server state
    if server_info.get('last_sync_id'):
        update_local_tracking(
            sync_id=server_info['last_sync_id'],
            timestamp=server_info['last_sync_timestamp']
        )
```

## Key Benefits

1. **Verify Server State**: Check what server has before syncing
2. **Detect Corruption**: Compare local tracking with server
3. **Auto-Recovery**: Restore local tracking from server if lost
4. **Prevent Data Loss**: Ensure no syncs are missed

## Integration Point

Add this call to your existing sync flow:

```python
# Existing sync flow
def perform_sync():
    # NEW: Query server first
    server_info = query_server_last_sync(api_token, client_id)
    
    # Verify and recover if needed
    if needs_recovery(server_info):
        recover_from_server(server_info)
    
    # Continue with normal sync
    send_sync_data()
```

That's it! Just add the query call before your existing sync logic.