"""
Test script for reports synchronization API
"""
import requests
import json
from datetime import datetime, timedelta

# Configuration
API_URL = "http://localhost:5000/api/reports/sync"
API_TOKEN = "your-api-token-here"  # Replace with actual token

def create_test_sync_data():
    """Create test sync data matching the API specification"""
    now = datetime.utcnow()
    
    sync_data = {
        "sync_id": f"sync_{now.strftime('%Y%m%d_%H%M%S')}_test1234",
        "client_id": "test_client_001",
        "sync_timestamp": now.isoformat(),
        "date_range": "today",
        "start_date": now.replace(hour=0, minute=0, second=0).isoformat(),
        "end_date": now.isoformat(),
        "bets": [
            {
                "uuid": "550e8400-e29b-41d4-a716-4466554400001",
                "fixture_id": "fixture_test_001",
                "bet_datetime": (now - timedelta(minutes=30)).isoformat(),
                "paid": False,
                "paid_out": False,
                "total_amount": 500.00,
                "bet_count": 3,
                "details": [
                    {
                        "match_id": 123,
                        "match_number": 1,
                        "outcome": "WIN1",
                        "amount": 200.00,
                        "win_amount": 0.00,
                        "result": "pending"
                    },
                    {
                        "match_id": 124,
                        "match_number": 2,
                        "outcome": "X1",
                        "amount": 150.00,
                        "win_amount": 0.00,
                        "result": "pending"
                    },
                    {
                        "match_id": 125,
                        "match_number": 3,
                        "outcome": "WIN2",
                        "amount": 150.00,
                        "win_amount": 0.00,
                        "result": "pending"
                    }
                ]
            }
        ],
        "extraction_stats": [
            {
                "match_id": 123,
                "fixture_id": "fixture_test_001",
                "match_datetime": (now - timedelta(hours=1)).isoformat(),
                "total_bets": 45,
                "total_amount_collected": 15000.00,
                "total_redistributed": 10500.00,
                "actual_result": "WIN1",
                "extraction_result": "WIN1",
                "cap_applied": True,
                "cap_percentage": 70.0,
                "under_bets": 20,
                "under_amount": 6000.00,
                "over_bets": 25,
                "over_amount": 9000.00,
                "result_breakdown": {
                    "WIN1": {"bets": 20, "amount": 6000.00, "coefficient": 2.5},
                    "X1": {"bets": 10, "amount": 3000.00, "coefficient": 3.0},
                    "WIN2": {"bets": 15, "amount": 6000.00, "coefficient": 2.0}
                }
            }
        ],
        "summary": {
            "total_payin": 15000.00,
            "total_payout": 10500.00,
            "net_profit": 4500.00,
            "total_bets": 45,
            "total_matches": 1
        }
    }
    
    return sync_data

def test_sync_endpoint():
    """Test the reports sync endpoint"""
    print("=" * 80)
    print("Testing Reports Synchronization API")
    print("=" * 80)
    
    # Create test data
    sync_data = create_test_sync_data()
    
    print(f"\n1. Preparing sync data:")
    print(f"   Sync ID: {sync_data['sync_id']}")
    print(f"   Client ID: {sync_data['client_id']}")
    print(f"   Bets: {len(sync_data['bets'])}")
    print(f"   Extraction Stats: {len(sync_data['extraction_stats'])}")
    print(f"   Total Payin: {sync_data['summary']['total_payin']}")
    print(f"   Total Payout: {sync_data['summary']['total_payout']}")
    print(f"   Net Profit: {sync_data['summary']['net_profit']}")
    
    # Prepare headers
    headers = {
        "Content-Type": "application/json",
        "Authorization": f"Bearer {API_TOKEN}"
    }
    
    print(f"\n2. Sending POST request to: {API_URL}")
    print(f"   Headers: Content-Type: application/json, Authorization: Bearer ***")
    
    try:
        # Send request
        response = requests.post(API_URL, json=sync_data, headers=headers, timeout=60)
        
        print(f"\n3. Response received:")
        print(f"   Status Code: {response.status_code}")
        print(f"   Response Time: {response.elapsed.total_seconds():.2f}s")
        
        # Parse response
        try:
            response_data = response.json()
            print(f"\n4. Response Data:")
            print(json.dumps(response_data, indent=2))
            
            # Check if successful
            if response.status_code == 200 and response_data.get('success'):
                print(f"\n✓ SUCCESS: Report synchronized successfully!")
                print(f"  Synced Count: {response_data.get('synced_count', 0)}")
                print(f"  Server Timestamp: {response_data.get('server_timestamp')}")
                return True
            else:
                print(f"\n✗ FAILED: {response_data.get('error', 'Unknown error')}")
                print(f"  Details: {response_data.get('details', 'No details')}")
                return False
                
        except json.JSONDecodeError:
            print(f"\n✗ FAILED: Could not parse JSON response")
            print(f"  Response Body: {response.text[:500]}")
            return False
            
    except requests.exceptions.Timeout:
        print(f"\n✗ FAILED: Request timed out after 60 seconds")
        return False
    except requests.exceptions.ConnectionError:
        print(f"\n✗ FAILED: Could not connect to server")
        print(f"  Make sure the server is running at {API_URL}")
        return False
    except Exception as e:
        print(f"\n✗ FAILED: Unexpected error occurred")
        print(f"  Error: {str(e)}")
        return False

def test_idempotency():
    """Test that duplicate sync_id is handled correctly"""
    print("\n" + "=" * 80)
    print("Testing Idempotency (Duplicate Sync ID)")
    print("=" * 80)
    
    # Create test data
    sync_data = create_test_sync_data()
    
    headers = {
        "Content-Type": "application/json",
        "Authorization": f"Bearer {API_TOKEN}"
    }
    
    print("\n1. Sending first sync request...")
    try:
        response1 = requests.post(API_URL, json=sync_data, headers=headers, timeout=60)
        print(f"   Status: {response1.status_code}")
        
        if response1.status_code == 200:
            print("   ✓ First sync successful")
            
            print("\n2. Sending duplicate sync request (same sync_id)...")
            response2 = requests.post(API_URL, json=sync_data, headers=headers, timeout=60)
            print(f"   Status: {response2.status_code}")
            
            if response2.status_code == 200:
                data2 = response2.json()
                if data2.get('success') and 'already' in data2.get('message', '').lower():
                    print("   ✓ Duplicate detected and handled correctly")
                    print(f"   Message: {data2.get('message')}")
                    return True
                else:
                    print("   ✗ Duplicate not detected properly")
                    return False
            else:
                print("   ✗ Second request failed")
                return False
        else:
            print("   ✗ First sync failed")
            return False
            
    except Exception as e:
        print(f"   ✗ Error: {str(e)}")
        return False

def test_validation():
    """Test input validation"""
    print("\n" + "=" * 80)
    print("Testing Input Validation")
    print("=" * 80)
    
    headers = {
        "Content-Type": "application/json",
        "Authorization": f"Bearer {API_TOKEN}"
    }
    
    # Test 1: Missing required field
    print("\n1. Testing missing required field (sync_id)...")
    invalid_data = create_test_sync_data()
    del invalid_data['sync_id']
    
    response = requests.post(API_URL, json=invalid_data, headers=headers, timeout=60)
    if response.status_code == 400:
        print("   ✓ Correctly rejected (400 Bad Request)")
        data = response.json()
        print(f"   Error: {data.get('error')}")
    else:
        print(f"   ✗ Expected 400, got {response.status_code}")
    
    # Test 2: Invalid UUID format
    print("\n2. Testing invalid UUID format...")
    invalid_data = create_test_sync_data()
    invalid_data['bets'][0]['uuid'] = 'invalid-uuid'
    
    response = requests.post(API_URL, json=invalid_data, headers=headers, timeout=60)
    if response.status_code == 400:
        print("   ✓ Correctly rejected (400 Bad Request)")
        data = response.json()
        print(f"   Error: {data.get('error')}")
    else:
        print(f"   ✗ Expected 400, got {response.status_code}")
    
    # Test 3: Invalid datetime format
    print("\n3. Testing invalid datetime format...")
    invalid_data = create_test_sync_data()
    invalid_data['sync_timestamp'] = 'invalid-datetime'
    
    response = requests.post(API_URL, json=invalid_data, headers=headers, timeout=60)
    if response.status_code == 400:
        print("   ✓ Correctly rejected (400 Bad Request)")
        data = response.json()
        print(f"   Error: {data.get('error')}")
    else:
        print(f"   ✗ Expected 400, got {response.status_code}")

def test_authentication():
    """Test authentication requirements"""
    print("\n" + "=" * 80)
    print("Testing Authentication")
    print("=" * 80)
    
    sync_data = create_test_sync_data()
    
    # Test 1: No authentication
    print("\n1. Testing without authentication...")
    headers = {"Content-Type": "application/json"}
    response = requests.post(API_URL, json=sync_data, headers=headers, timeout=60)
    
    if response.status_code == 401:
        print("   ✓ Correctly rejected (401 Unauthorized)")
        data = response.json()
        print(f"   Error: {data.get('error')}")
    else:
        print(f"   ✗ Expected 401, got {response.status_code}")
    
    # Test 2: Invalid token
    print("\n2. Testing with invalid token...")
    headers = {
        "Content-Type": "application/json",
        "Authorization": "Bearer invalid-token-12345"
    }
    response = requests.post(API_URL, json=sync_data, headers=headers, timeout=60)
    
    if response.status_code == 401:
        print("   ✓ Correctly rejected (401 Unauthorized)")
        data = response.json()
        print(f"   Error: {data.get('error')}")
    else:
        print(f"   ✗ Expected 401, got {response.status_code}")

def main():
    """Run all tests"""
    print("\n" + "=" * 80)
    print("REPORTS SYNCHRONIZATION API TEST SUITE")
    print("=" * 80)
    print(f"\nAPI URL: {API_URL}")
    print(f"API Token: {API_TOKEN[:20]}..." if len(API_TOKEN) > 20 else f"API Token: {API_TOKEN}")
    print(f"Test Time: {datetime.utcnow().isoformat()}")
    
    # Run tests
    results = []
    
    # Test authentication first
    results.append(("Authentication", test_authentication()))
    
    # Test validation
    results.append(("Validation", test_validation()))
    
    # Test main sync endpoint
    results.append(("Sync Endpoint", test_sync_endpoint()))
    
    # Test idempotency
    results.append(("Idempotency", test_idempotency()))
    
    # Print summary
    print("\n" + "=" * 80)
    print("TEST SUMMARY")
    print("=" * 80)
    
    for test_name, passed in results:
        status = "✓ PASSED" if passed else "✗ FAILED"
        print(f"{test_name:.<50} {status}")
    
    total_tests = len(results)
    passed_tests = sum(1 for _, passed in results if passed)
    
    print(f"\nTotal: {passed_tests}/{total_tests} tests passed")
    
    if passed_tests == total_tests:
        print("\n🎉 All tests passed!")
        return 0
    else:
        print(f"\n⚠️  {total_tests - passed_tests} test(s) failed")
        return 1

if __name__ == "__main__":
    import sys
    sys.exit(main())