import logging
import os
from datetime import datetime
from flask import render_template, request, jsonify, redirect, url_for, flash, current_app
from flask_login import login_required, current_user
from app.main import bp
from app import db, csrf
from app.upload.file_handler import get_file_upload_handler
from app.upload.fixture_parser import get_fixture_parser
from app.utils.security import require_admin, require_active_user
import pandas as pd
from io import BytesIO, StringIO
from flask import Response, make_response

logger = logging.getLogger(__name__)

@csrf.exempt
@bp.route('/')
def index():
    """Home page"""
    # Check if database is available
    try:
        from app.database import get_db_manager
        db_manager = get_db_manager()
        if not db_manager.test_connection():
            # Add logging to debug template loading
            logger.info(f"Template folder path: {current_app.template_folder}")
            logger.info(f"Template folder exists: {os.path.exists(current_app.template_folder) if current_app.template_folder else 'None'}")
            template_path = os.path.join(current_app.template_folder or '', 'setup', 'database.html')
            logger.info(f"Full template path: {template_path}")
            logger.info(f"Template file exists: {os.path.exists(template_path)}")
            return render_template('setup/database.html')
    except Exception as e:
        logger.warning(f"Database check failed: {str(e)}")
        # Add logging to debug template loading
        logger.info(f"Template folder path: {current_app.template_folder}")
        logger.info(f"Template folder exists: {os.path.exists(current_app.template_folder) if current_app.template_folder else 'None'}")
        template_path = os.path.join(current_app.template_folder or '', 'setup', 'database.html')
        logger.info(f"Full template path: {template_path}")
        logger.info(f"Template file exists: {os.path.exists(template_path)}")
        return render_template('setup/database.html')
    
    if current_user.is_authenticated:
        return redirect(url_for('main.dashboard'))
    return render_template('main/index.html')

@csrf.exempt
@bp.route('/dashboard')
@login_required
@require_active_user
def dashboard():
    """Main dashboard"""
    try:
        # Check if database is available
        from app.database import get_db_manager
        db_manager = get_db_manager()
        if not db_manager.test_connection():
            flash('Database connection failed. Please check database setup.', 'error')
            return render_template('setup/database.html')
        
        from app.models import Match, FileUpload
        from sqlalchemy import func
        
        # Get user statistics
        user_matches = Match.query.filter_by(created_by=current_user.id).count()
        user_uploads = FileUpload.query.filter_by(uploaded_by=current_user.id).count()
        
        # Get recent fixtures (grouped by fixture_id)
        recent_fixtures_query = db.session.query(
            Match.fixture_id,
            Match.filename,
            func.count(Match.id).label('match_count'),
            func.min(Match.created_at).label('upload_date')
        ).filter_by(created_by=current_user.id)\
         .group_by(Match.fixture_id, Match.filename)\
         .order_by(func.min(Match.created_at).desc())\
         .limit(5)
        
        recent_fixtures = recent_fixtures_query.all()
        
        # Get system statistics (for admins)
        system_stats = {}
        if current_user.is_admin:
            db_manager = get_db_manager()
            system_stats = db_manager.get_database_stats()
            
            file_handler = get_file_upload_handler()
            upload_stats = file_handler.get_upload_statistics()
            system_stats.update(upload_stats)
        
        return render_template('main/dashboard.html',
                             user_matches=user_matches,
                             user_uploads=user_uploads,
                             recent_fixtures=recent_fixtures,
                             system_stats=system_stats)
    
    except Exception as e:
        logger.error(f"Dashboard error: {str(e)}")
        flash('Error loading dashboard', 'error')
        return render_template('main/dashboard.html')

@csrf.exempt
@bp.route('/fixtures')
@login_required
@require_active_user
def fixtures():
    """List fixtures with pagination and filtering"""
    try:
        page = request.args.get('page', 1, type=int)
        per_page = min(request.args.get('per_page', 20, type=int), 100)
        
        # Filtering options
        status_filter = request.args.get('status')
        search_query = request.args.get('search', '').strip()
        
        from app.models import Match
        from sqlalchemy import func, and_
        
        # Base query for fixtures (grouped by fixture_id)
        base_query = db.session.query(
            Match.fixture_id,
            Match.filename,
            func.count(Match.id).label('match_count'),
            func.min(Match.created_at).label('upload_date'),
            func.sum(Match.active_status.cast(db.Integer)).label('active_matches'),
            func.max(Match.created_by).label('created_by')  # Assuming all matches in fixture have same creator
        ).group_by(Match.fixture_id, Match.filename)
        
        # Apply user filter
        if not current_user.is_admin:
            base_query = base_query.filter(Match.created_by == current_user.id)
        
        # Apply search filter
        if search_query:
            search_pattern = f"%{search_query}%"
            base_query = base_query.filter(Match.filename.ilike(search_pattern))
        
        # Get all fixtures
        fixtures_query = base_query.order_by(func.min(Match.created_at).desc())
        
        # Apply status filter after grouping
        fixtures = fixtures_query.all()
        
        if status_filter:
            filtered_fixtures = []
            for fixture in fixtures:
                if status_filter == 'active' and fixture.active_matches > 0:
                    filtered_fixtures.append(fixture)
                elif status_filter == 'inactive' and fixture.active_matches == 0:
                    filtered_fixtures.append(fixture)
                elif status_filter == 'complete' and fixture.active_matches == fixture.match_count:
                    filtered_fixtures.append(fixture)
                elif not status_filter:
                    filtered_fixtures.append(fixture)
            fixtures = filtered_fixtures
        
        # Manual pagination
        total = len(fixtures)
        start = (page - 1) * per_page
        end = start + per_page
        fixtures_page = fixtures[start:end]
        
        # Create pagination object
        class SimplePagination:
            def __init__(self, page, per_page, total, items):
                self.page = page
                self.per_page = per_page
                self.total = total
                self.items = items
                self.pages = (total + per_page - 1) // per_page
                self.has_prev = page > 1
                self.has_next = page < self.pages
                self.prev_num = page - 1 if self.has_prev else None
                self.next_num = page + 1 if self.has_next else None
        
        pagination = SimplePagination(page, per_page, total, fixtures_page)
        
        return render_template('main/matches.html',
                             fixtures=fixtures_page,
                             pagination=pagination,
                             status_filter=status_filter,
                             search_query=search_query)
    
    except Exception as e:
        logger.error(f"Fixtures list error: {str(e)}")
        flash('Error loading fixtures', 'error')
        return render_template('main/matches.html', fixtures=[], pagination=None)

# Keep the old /matches route for backward compatibility
@csrf.exempt
@bp.route('/matches')
@login_required
@require_active_user
def matches():
    """Redirect to fixtures for backward compatibility"""
    return redirect(url_for('main.fixtures'))

@csrf.exempt
@bp.route('/fixture/<fixture_id>')
@login_required
@require_active_user
def fixture_detail(fixture_id):
    """Fixture detail page showing all matches in the fixture"""
    try:
        from app.models import Match, FileUpload
        
        # Get all matches for this fixture
        if current_user.is_admin:
            matches = Match.query.filter_by(fixture_id=fixture_id).all()
        else:
            matches = Match.query.filter_by(fixture_id=fixture_id, created_by=current_user.id).all()
        
        if not matches:
            flash('Fixture not found', 'error')
            return redirect(url_for('main.matches'))
        
        # Get fixture info from first match
        fixture_info = {
            'fixture_id': fixture_id,
            'filename': matches[0].filename,
            'upload_date': matches[0].created_at,
            'total_matches': len(matches),
            'active_matches': sum(1 for m in matches if m.active_status),
            'created_by': matches[0].created_by
        }
        
        # Get associated uploads for the fixture
        match_ids = [m.id for m in matches]
        uploads = FileUpload.query.filter(FileUpload.match_id.in_(match_ids)).all() if match_ids else []
        
        return render_template('main/fixture_detail.html',
                             fixture_info=fixture_info,
                             matches=matches,
                             uploads=uploads)
    
    except Exception as e:
        logger.error(f"Fixture detail error: {str(e)}")
        flash('Error loading fixture details', 'error')
        return redirect(url_for('main.matches'))

@csrf.exempt
@bp.route('/fixture/<fixture_id>/delete', methods=['POST'])
@login_required
@require_active_user
def delete_fixture(fixture_id):
    """Delete entire fixture and all related matches"""
    try:
        from app.models import Match, MatchOutcome, FileUpload
        
        # Get all matches for this fixture
        if current_user.is_admin:
            matches = Match.query.filter_by(fixture_id=fixture_id).all()
        else:
            matches = Match.query.filter_by(fixture_id=fixture_id, created_by=current_user.id).all()
        
        if not matches:
            flash('Fixture not found', 'error')
            return redirect(url_for('main.matches'))
        
        fixture_filename = matches[0].filename
        match_count = len(matches)
        
        # Delete all related data in correct order to handle foreign key constraints
        match_ids = [m.id for m in matches]
        
        # First, get file upload IDs that will be deleted
        file_uploads = FileUpload.query.filter(FileUpload.match_id.in_(match_ids)).all()
        upload_ids = [upload.id for upload in file_uploads]
        
        # Delete system logs that reference uploads (to avoid foreign key constraint)
        if upload_ids:
            from app.models import SystemLog
            SystemLog.query.filter(SystemLog.upload_id.in_(upload_ids)).delete(synchronize_session=False)
        
        # Delete system logs that reference matches
        if match_ids:
            from app.models import SystemLog
            SystemLog.query.filter(SystemLog.match_id.in_(match_ids)).delete(synchronize_session=False)
        
        # Delete match outcomes
        MatchOutcome.query.filter(MatchOutcome.match_id.in_(match_ids)).delete(synchronize_session=False)
        
        # Delete file uploads associated with matches
        FileUpload.query.filter(FileUpload.match_id.in_(match_ids)).delete(synchronize_session=False)
        
        # Delete matches
        Match.query.filter_by(fixture_id=fixture_id).delete(synchronize_session=False)
        
        db.session.commit()
        
        logger.info(f"Fixture {fixture_id} ({fixture_filename}) with {match_count} matches deleted by user {current_user.username}")
        flash(f'Fixture "{fixture_filename}" and all {match_count} related matches have been deleted successfully.', 'success')
        
        return redirect(url_for('main.matches'))
    
    except Exception as e:
        db.session.rollback()
        logger.error(f"Fixture deletion error: {str(e)}")
        flash('Error deleting fixture', 'error')
        return redirect(url_for('main.matches'))

@csrf.exempt
@bp.route('/match/<int:id>')
@login_required
@require_active_user
def match_detail(id):
    """Individual match detail page"""
    try:
        from app.models import Match, FileUpload
        if current_user.is_admin:
            match = Match.query.get_or_404(id)
        else:
            match = Match.query.filter_by(id=id, created_by=current_user.id).first_or_404()
        
        # Get match outcomes
        outcomes = match.outcomes.all()
        
        # Get associated uploads
        uploads = FileUpload.query.filter_by(match_id=id).all()
        
        return render_template('main/match_detail.html',
                             match=match,
                             outcomes=outcomes,
                             uploads=uploads)
    
    except Exception as e:
        logger.error(f"Match detail error: {str(e)}")
        flash('Error loading match details', 'error')
        return redirect(url_for('main.fixture_detail', fixture_id=request.args.get('fixture_id', '')))

@csrf.exempt
@bp.route('/match/<int:match_id>/outcomes', methods=['POST'])
@login_required
@require_active_user
def update_match_outcomes(match_id):
    """Update match outcomes via AJAX"""
    try:
        from app.models import Match, MatchOutcome
        
        # Get the match
        if current_user.is_admin:
            match = Match.query.get_or_404(match_id)
        else:
            match = Match.query.filter_by(id=match_id, created_by=current_user.id).first_or_404()
        
        data = request.get_json()
        if not data or 'outcomes' not in data:
            return jsonify({'error': 'No outcomes data provided'}), 400
        
        outcomes_data = data['outcomes']
        updated_outcomes = []
        
        # Update or create outcomes
        for column_name, float_value in outcomes_data.items():
            try:
                # Validate the float value
                float_val = float(float_value)
                
                # Find existing outcome or create new one
                outcome = MatchOutcome.query.filter_by(
                    match_id=match_id,
                    column_name=column_name
                ).first()
                
                if outcome:
                    outcome.float_value = float_val
                    outcome.updated_at = datetime.utcnow()
                else:
                    outcome = MatchOutcome(
                        match_id=match_id,
                        column_name=column_name,
                        float_value=float_val
                    )
                    db.session.add(outcome)
                
                updated_outcomes.append({
                    'column_name': column_name,
                    'float_value': float_val
                })
                
            except ValueError:
                return jsonify({'error': f'Invalid numeric value for {column_name}'}), 400
        
        # Update match timestamp
        match.updated_at = datetime.utcnow()
        db.session.commit()
        
        logger.info(f"Match {match_id} outcomes updated by user {current_user.username}")
        
        return jsonify({
            'message': 'Outcomes updated successfully',
            'updated_outcomes': updated_outcomes
        }), 200
    
    except Exception as e:
        db.session.rollback()
        logger.error(f"Update match outcomes error: {str(e)}")
        return jsonify({'error': 'Failed to update outcomes'}), 500

@csrf.exempt
@bp.route('/match/<int:match_id>/outcomes/<int:outcome_id>', methods=['DELETE'])
@login_required
@require_active_user
def delete_match_outcome(match_id, outcome_id):
    """Delete a specific match outcome"""
    try:
        from app.models import Match, MatchOutcome
        
        # Get the match to verify ownership
        if current_user.is_admin:
            match = Match.query.get_or_404(match_id)
        else:
            match = Match.query.filter_by(id=match_id, created_by=current_user.id).first_or_404()
        
        # Get the outcome
        outcome = MatchOutcome.query.filter_by(
            id=outcome_id,
            match_id=match_id
        ).first_or_404()
        
        column_name = outcome.column_name
        db.session.delete(outcome)
        
        # Update match timestamp
        match.updated_at = datetime.utcnow()
        db.session.commit()
        
        logger.info(f"Outcome {column_name} deleted from match {match_id} by user {current_user.username}")
        
        return jsonify({
            'message': f'Outcome "{column_name}" deleted successfully'
        }), 200
    
    except Exception as e:
        db.session.rollback()
        logger.error(f"Delete match outcome error: {str(e)}")
        return jsonify({'error': 'Failed to delete outcome'}), 500

@csrf.exempt
@bp.route('/uploads')
@login_required
@require_active_user
def uploads():
    """List uploads with pagination and filtering"""
    try:
        page = request.args.get('page', 1, type=int)
        per_page = min(request.args.get('per_page', 20, type=int), 100)
        
        # Filtering options
        file_type_filter = request.args.get('file_type')
        status_filter = request.args.get('status')
        
        from app.models import FileUpload
        # Base query
        if current_user.is_admin:
            query = FileUpload.query
        else:
            query = FileUpload.query.filter_by(uploaded_by=current_user.id)
        
        # Apply filters
        if file_type_filter:
            query = query.filter_by(file_type=file_type_filter)
        
        if status_filter:
            query = query.filter_by(upload_status=status_filter)
        
        # Pagination
        uploads_pagination = query.order_by(FileUpload.created_at.desc()).paginate(
            page=page, per_page=per_page, error_out=False
        )
        
        return render_template('main/uploads.html',
                             uploads=uploads_pagination.items,
                             pagination=uploads_pagination,
                             file_type_filter=file_type_filter,
                             status_filter=status_filter)
    
    except Exception as e:
        logger.error(f"Uploads list error: {str(e)}")
        flash('Error loading uploads', 'error')
        return render_template('main/uploads.html', uploads=[], pagination=None)

@csrf.exempt
@bp.route('/statistics')
@login_required
@require_active_user
def statistics():
    """Statistics page"""
    try:
        # Get file handler and parser statistics
        file_handler = get_file_upload_handler()
        fixture_parser = get_fixture_parser()
        
        upload_stats = file_handler.get_upload_statistics()
        parsing_stats = fixture_parser.get_parsing_statistics()
        
        from app.models import FileUpload, Match
        # User-specific statistics
        user_stats = {
            'total_uploads': FileUpload.query.filter_by(uploaded_by=current_user.id).count(),
            'total_matches': Match.query.filter_by(created_by=current_user.id).count(),
            'active_matches': Match.query.filter_by(
                created_by=current_user.id, active_status=True
            ).count(),
            'pending_zip_uploads': Match.query.filter_by(
                created_by=current_user.id, zip_upload_status='pending'
            ).count()
        }
        
        # System statistics (admin only)
        system_stats = {}
        if current_user.is_admin:
            from app.database import get_db_manager
            db_manager = get_db_manager()
            system_stats = db_manager.get_database_stats()
        
        return render_template('main/statistics.html',
                             upload_stats=upload_stats,
                             parsing_stats=parsing_stats,
                             user_stats=user_stats,
                             system_stats=system_stats)
    
    except Exception as e:
        logger.error(f"Statistics error: {str(e)}")
        flash('Error loading statistics', 'error')
        return render_template('main/statistics.html')

@csrf.exempt
@bp.route('/admin')
@login_required
@require_admin
def admin_panel():
    """Admin panel"""
    try:
        # Get system overview
        from app.database import get_db_manager
        db_manager = get_db_manager()
        system_stats = db_manager.get_database_stats()
        
        from app.models import SystemLog, User
        # Get recent system logs
        recent_logs = SystemLog.query.order_by(SystemLog.created_at.desc()).limit(20).all()
        
        # Get user statistics
        total_users = User.query.count()
        active_users = User.query.filter_by(is_active=True).count()
        admin_users = User.query.filter_by(is_admin=True).count()
        
        user_stats = {
            'total_users': total_users,
            'active_users': active_users,
            'admin_users': admin_users,
            'inactive_users': total_users - active_users
        }
        
        # Get upload statistics
        file_handler = get_file_upload_handler()
        upload_stats = file_handler.get_upload_statistics()
        
        return render_template('main/admin.html',
                             system_stats=system_stats,
                             user_stats=user_stats,
                             upload_stats=upload_stats,
                             recent_logs=recent_logs)
    
    except Exception as e:
        logger.error(f"Admin panel error: {str(e)}")
        flash('Error loading admin panel', 'error')
        return render_template('main/admin.html')

@csrf.exempt
@bp.route('/admin/users')
@login_required
@require_admin
def admin_users():
    """Admin user management"""
    try:
        page = request.args.get('page', 1, type=int)
        per_page = min(request.args.get('per_page', 20, type=int), 100)
        
        search_query = request.args.get('search', '').strip()
        status_filter = request.args.get('status')
        
        from app.models import User
        # Base query
        query = User.query
        
        # Apply filters
        if status_filter == 'active':
            query = query.filter_by(is_active=True)
        elif status_filter == 'inactive':
            query = query.filter_by(is_active=False)
        elif status_filter == 'admin':
            query = query.filter_by(is_admin=True)
        
        # Search functionality
        if search_query:
            search_pattern = f"%{search_query}%"
            query = query.filter(
                db.or_(
                    User.username.ilike(search_pattern),
                    User.email.ilike(search_pattern)
                )
            )
        
        # Pagination
        users_pagination = query.order_by(User.created_at.desc()).paginate(
            page=page, per_page=per_page, error_out=False
        )
        
        return render_template('main/admin_users.html',
                             users=users_pagination.items,
                             pagination=users_pagination,
                             search_query=search_query,
                             status_filter=status_filter)
    
    except Exception as e:
        logger.error(f"Admin users error: {str(e)}")
        flash('Error loading users', 'error')
        return render_template('main/admin_users.html', users=[], pagination=None)

@csrf.exempt
@bp.route('/admin/logs')
@login_required
@require_admin
def admin_logs():
    """Admin system logs"""
    try:
        page = request.args.get('page', 1, type=int)
        per_page = min(request.args.get('per_page', 50, type=int), 200)
        
        level_filter = request.args.get('level')
        module_filter = request.args.get('module')
        
        from app.models import SystemLog
        # Base query
        query = SystemLog.query
        
        # Apply filters
        if level_filter:
            query = query.filter_by(level=level_filter)
        
        if module_filter:
            query = query.filter_by(module=module_filter)
        
        # Pagination
        logs_pagination = query.order_by(SystemLog.created_at.desc()).paginate(
            page=page, per_page=per_page, error_out=False
        )
        
        # Get available modules for filter
        modules = db.session.query(SystemLog.module).distinct().all()
        modules = [m[0] for m in modules if m[0]]
        
        return render_template('main/admin_logs.html',
                             logs=logs_pagination.items,
                             pagination=logs_pagination,
                             level_filter=level_filter,
                             module_filter=module_filter,
                             modules=modules)
    
    except Exception as e:
        logger.error(f"Admin logs error: {str(e)}")
        flash('Error loading logs', 'error')
        return render_template('main/admin_logs.html', logs=[], pagination=None)

@csrf.exempt
@bp.route('/admin/users/<int:user_id>/edit', methods=['POST'])
@login_required
@require_admin
def admin_edit_user(user_id):
    """Edit user details"""
    try:
        from app.models import User
        user = User.query.get_or_404(user_id)
        
        # Prevent editing current admin user's admin status
        if user.id == current_user.id and request.json.get('is_admin') == False:
            return jsonify({'error': 'Cannot remove admin status from your own account'}), 400
        
        data = request.get_json()
        if not data:
            return jsonify({'error': 'No data provided'}), 400
        
        # Update user fields
        if 'username' in data:
            # Check if username is already taken by another user
            existing_user = User.query.filter_by(username=data['username']).first()
            if existing_user and existing_user.id != user.id:
                return jsonify({'error': 'Username already exists'}), 400
            user.username = data['username']
        
        if 'email' in data:
            # Check if email is already taken by another user
            existing_user = User.query.filter_by(email=data['email']).first()
            if existing_user and existing_user.id != user.id:
                return jsonify({'error': 'Email already exists'}), 400
            user.email = data['email']
        
        if 'is_active' in data:
            user.is_active = bool(data['is_active'])
        
        if 'is_admin' in data:
            user.is_admin = bool(data['is_admin'])
        
        user.updated_at = datetime.utcnow()
        db.session.commit()
        
        logger.info(f"User {user.username} updated by admin {current_user.username}")
        return jsonify({'message': 'User updated successfully', 'user': user.to_dict()}), 200
    
    except Exception as e:
        logger.error(f"Admin edit user error: {str(e)}")
        return jsonify({'error': 'Failed to update user'}), 500

@csrf.exempt
@bp.route('/admin/users/<int:user_id>/delete', methods=['DELETE'])
@login_required
@require_admin
def admin_delete_user(user_id):
    """Delete user"""
    try:
        from app.models import User
        user = User.query.get_or_404(user_id)
        
        # Prevent deleting current admin user
        if user.id == current_user.id:
            return jsonify({'error': 'Cannot delete your own account'}), 400
        
        username = user.username
        db.session.delete(user)
        db.session.commit()
        
        logger.info(f"User {username} deleted by admin {current_user.username}")
        return jsonify({'message': 'User deleted successfully'}), 200
    
    except Exception as e:
        logger.error(f"Admin delete user error: {str(e)}")
        return jsonify({'error': 'Failed to delete user'}), 500

@csrf.exempt
@bp.route('/admin/users/create', methods=['POST'])
@login_required
@require_admin
def admin_create_user():
    """Create new user"""
    try:
        data = request.get_json()
        if not data:
            return jsonify({'error': 'No data provided'}), 400
        
        # Validate required fields
        required_fields = ['username', 'email', 'password']
        for field in required_fields:
            if field not in data or not data[field]:
                return jsonify({'error': f'{field.title()} is required'}), 400
        
        from app.models import User
        # Check if username already exists
        if User.query.filter_by(username=data['username']).first():
            return jsonify({'error': 'Username already exists'}), 400
        
        # Check if email already exists
        if User.query.filter_by(email=data['email']).first():
            return jsonify({'error': 'Email already exists'}), 400
        
        # Validate password strength
        from app.utils.security import validate_password_strength
        if not validate_password_strength(data['password']):
            return jsonify({'error': 'Password does not meet security requirements'}), 400
        
        # Create new user
        user = User(
            username=data['username'],
            email=data['email'],
            is_active=data.get('is_active', True),
            is_admin=data.get('is_admin', False)
        )
        user.set_password(data['password'])
        
        db.session.add(user)
        db.session.commit()
        
        logger.info(f"User {user.username} created by admin {current_user.username}")
        return jsonify({'message': 'User created successfully', 'user': user.to_dict()}), 201
    
    except Exception as e:
        logger.error(f"Admin create user error: {str(e)}")
        return jsonify({'error': 'Failed to create user'}), 500

@csrf.exempt
@bp.route('/admin/users/<int:user_id>/reset-password', methods=['POST'])
@login_required
@require_admin
def admin_reset_user_password(user_id):
    """Reset user password"""
    try:
        from app.models import User
        user = User.query.get_or_404(user_id)
        
        data = request.get_json()
        if not data or not data.get('new_password'):
            return jsonify({'error': 'New password is required'}), 400
        
        # Validate password strength
        from app.utils.security import validate_password_strength
        if not validate_password_strength(data['new_password']):
            return jsonify({'error': 'Password does not meet security requirements'}), 400
        
        user.set_password(data['new_password'])
        user.updated_at = datetime.utcnow()
        db.session.commit()
        
        logger.info(f"Password reset for user {user.username} by admin {current_user.username}")
        return jsonify({'message': 'Password reset successfully'}), 200
    
    except Exception as e:
        logger.error(f"Admin reset password error: {str(e)}")
        return jsonify({'error': 'Failed to reset password'}), 500

@csrf.exempt
@bp.route('/admin/settings')
@login_required
@require_admin
def admin_system_settings():
    """System settings management page"""
    try:
        from app.models import SystemSettings
        
        # Initialize default settings if they don't exist
        SystemSettings.initialize_default_settings()
        
        # Get all system settings
        settings = SystemSettings.query.all()
        settings_dict = {setting.key: setting for setting in settings}
        
        return render_template('main/admin_settings.html', settings=settings_dict)
    
    except Exception as e:
        logger.error(f"Admin system settings error: {str(e)}")
        flash('Error loading system settings', 'error')
        return render_template('main/admin_settings.html', settings={})

@csrf.exempt
@bp.route('/admin/settings/registration', methods=['GET', 'POST'])
@login_required
@require_admin
def admin_registration_settings():
    """Manage registration settings"""
    try:
        from app.models import SystemSettings
        
        if request.method == 'GET':
            # Get current registration status from database
            registration_enabled = SystemSettings.is_registration_enabled()
            return jsonify({'registration_enabled': registration_enabled}), 200
        
        elif request.method == 'POST':
            data = request.get_json()
            if not data:
                return jsonify({'error': 'No data provided'}), 400
            
            # Update registration setting in database
            registration_enabled = bool(data.get('registration_enabled', False))
            SystemSettings.set_registration_enabled(registration_enabled)
            
            status = 'enabled' if registration_enabled else 'disabled'
            logger.info(f"Registration {status} by admin {current_user.username}")
            
            return jsonify({
                'message': f'Registration {status} successfully',
                'registration_enabled': registration_enabled
            }), 200
    
    except Exception as e:
        logger.error(f"Admin registration settings error: {str(e)}")
        return jsonify({'error': 'Failed to update registration settings'}), 500

@csrf.exempt
@bp.route('/admin/settings/<setting_key>', methods=['GET', 'POST'])
@login_required
@require_admin
def admin_setting_detail(setting_key):
    """Get or update a specific system setting"""
    try:
        from app.models import SystemSettings
        
        if request.method == 'GET':
            setting = SystemSettings.query.filter_by(key=setting_key).first()
            if not setting:
                return jsonify({'error': 'Setting not found'}), 404
            
            return jsonify({'setting': setting.to_dict()}), 200
        
        elif request.method == 'POST':
            data = request.get_json()
            if not data or 'value' not in data:
                return jsonify({'error': 'Value is required'}), 400
            
            value = data['value']
            value_type = data.get('value_type', 'string')
            description = data.get('description')
            
            # Validate value type
            if value_type not in ['string', 'boolean', 'integer', 'float', 'json']:
                return jsonify({'error': 'Invalid value type'}), 400
            
            # Type validation
            if value_type == 'boolean' and not isinstance(value, bool):
                return jsonify({'error': 'Value must be boolean'}), 400
            elif value_type == 'integer':
                try:
                    int(value)
                except (ValueError, TypeError):
                    return jsonify({'error': 'Value must be integer'}), 400
            elif value_type == 'float':
                try:
                    float(value)
                except (ValueError, TypeError):
                    return jsonify({'error': 'Value must be float'}), 400
            
            # Update setting
            setting = SystemSettings.set_setting(setting_key, value, value_type, description)
            
            logger.info(f"System setting '{setting_key}' updated to '{value}' by admin {current_user.username}")
            
            return jsonify({
                'message': f'Setting "{setting_key}" updated successfully',
                'setting': setting.to_dict()
            }), 200
    
    except Exception as e:
        logger.error(f"Admin setting detail error: {str(e)}")
        return jsonify({'error': 'Failed to update setting'}), 500

@csrf.exempt
@bp.route('/admin/database/migrations')
@login_required
@require_admin
def admin_database_migrations():
    """Database migrations management"""
    try:
        from app.database import get_migration_status
        migration_status = get_migration_status()
        
        return render_template('main/admin_migrations.html',
                             migration_status=migration_status)
    
    except Exception as e:
        logger.error(f"Admin migrations error: {str(e)}")
        flash('Error loading migration status', 'error')
        return render_template('main/admin_migrations.html', migration_status={})

@csrf.exempt
@bp.route('/admin/database/migrations/run', methods=['POST'])
@login_required
@require_admin
def admin_run_migrations():
    """Run pending database migrations"""
    try:
        from app.database import run_migrations
        result = run_migrations()
        
        if result['status'] == 'success':
            logger.info(f"Migrations run successfully by admin {current_user.username}: {result['message']}")
            return jsonify({
                'success': True,
                'message': result['message'],
                'applied_count': result['applied_count']
            }), 200
        elif result['status'] == 'partial':
            logger.warning(f"Migrations partially completed by admin {current_user.username}: {result['message']}")
            return jsonify({
                'success': False,
                'message': result['message'],
                'applied_count': result['applied_count'],
                'failed_count': result['failed_count'],
                'failed_migrations': result.get('failed_migrations', [])
            }), 207  # Multi-status
        else:
            logger.error(f"Migration failed by admin {current_user.username}: {result['message']}")
            return jsonify({
                'success': False,
                'message': result['message']
            }), 500
    
    except Exception as e:
        logger.error(f"Admin run migrations error: {str(e)}")
        return jsonify({
            'success': False,
            'message': f'Failed to run migrations: {str(e)}'
        }), 500

@csrf.exempt
@bp.route('/admin/database/migrations/status')
@login_required
@require_admin
def admin_migration_status():
    """Get current migration status"""
    try:
        from app.database import get_migration_status
        migration_status = get_migration_status()
        
        return jsonify({
            'success': True,
            'status': migration_status
        }), 200
    
    except Exception as e:
        logger.error(f"Admin migration status error: {str(e)}")
        return jsonify({
            'success': False,
            'message': f'Failed to get migration status: {str(e)}'
        }), 500

@csrf.exempt
@bp.route('/health')
def health_check():
    """Health check endpoint"""
    try:
        # Test database connection
        from app.database import get_db_manager
        db_manager = get_db_manager()
        db_healthy = db_manager.test_connection()
        
        # Get basic statistics
        stats = db_manager.get_database_stats()
        
        health_status = {
            'status': 'healthy' if db_healthy else 'unhealthy',
            'database': 'connected' if db_healthy else 'disconnected',
            'timestamp': db.func.now(),
            'statistics': stats
        }
        
        status_code = 200 if db_healthy else 503
        return jsonify(health_status), status_code
    
    except Exception as e:
        logger.error(f"Health check error: {str(e)}")
        return jsonify({
            'status': 'unhealthy',
            'error': str(e),
            'timestamp': db.func.now()
        }), 503

@csrf.exempt
@bp.route('/api/dashboard-data')
@login_required
@require_active_user
def api_dashboard_data():
    """API endpoint for dashboard data"""
    try:
        from app.models import Match, FileUpload
        # User statistics
        user_stats = {
            'total_matches': Match.query.filter_by(created_by=current_user.id).count(),
            'active_matches': Match.query.filter_by(
                created_by=current_user.id, active_status=True
            ).count(),
            'total_uploads': FileUpload.query.filter_by(uploaded_by=current_user.id).count(),
            'pending_zip_uploads': Match.query.filter_by(
                created_by=current_user.id, zip_upload_status='pending'
            ).count()
        }
        
        # Recent activity
        recent_matches = Match.query.filter_by(created_by=current_user.id)\
            .order_by(Match.created_at.desc()).limit(5).all()
        
        recent_uploads = FileUpload.query.filter_by(uploaded_by=current_user.id)\
            .order_by(FileUpload.created_at.desc()).limit(5).all()
        
        return jsonify({
            'user_stats': user_stats,
            'recent_matches': [match.to_dict() for match in recent_matches],
            'recent_uploads': [upload.to_dict() for upload in recent_uploads]
        }), 200
    
    except Exception as e:
        logger.error(f"Dashboard API error: {str(e)}")
        return jsonify({'error': 'Failed to load dashboard data'}), 500

# API Token Management Routes

@csrf.exempt
@bp.route('/profile/tokens')
@login_required
@require_active_user
def user_tokens():
    """User API token management page"""
    try:
        from app.models import APIToken
        
        # Get user's API tokens
        tokens = APIToken.query.filter_by(user_id=current_user.id)\
            .order_by(APIToken.created_at.desc()).all()
        
        return render_template('main/user_tokens.html', tokens=tokens)
    
    except Exception as e:
        logger.error(f"User tokens page error: {str(e)}")
        flash('Error loading API tokens', 'error')
        return render_template('main/user_tokens.html', tokens=[])

@csrf.exempt
@bp.route('/profile/tokens/create', methods=['POST'])
@login_required
@require_active_user
def create_api_token():
    """Create a new API token"""
    try:
        data = request.get_json()
        if not data or not data.get('name'):
            return jsonify({'error': 'Token name is required'}), 400
        
        token_name = data['name'].strip()
        if not token_name:
            return jsonify({'error': 'Token name cannot be empty'}), 400
        
        # Check if user already has a token with this name
        from app.models import APIToken
        existing_token = APIToken.query.filter_by(
            user_id=current_user.id,
            name=token_name
        ).first()
        
        if existing_token:
            return jsonify({'error': 'A token with this name already exists'}), 400
        
        # Generate token
        api_token = current_user.generate_api_token(token_name)
        
        logger.info(f"API token '{token_name}' created by user {current_user.username}")
        
        return jsonify({
            'message': 'API token created successfully',
            'token': api_token.to_dict(include_token=True)
        }), 201
    
    except Exception as e:
        logger.error(f"Create API token error: {str(e)}")
        return jsonify({'error': 'Failed to create API token'}), 500

@csrf.exempt
@bp.route('/profile/tokens/<int:token_id>/delete', methods=['DELETE'])
@login_required
@require_active_user
def delete_api_token(token_id):
    """Delete an API token"""
    try:
        from app.models import APIToken
        
        # Get token (ensure it belongs to current user)
        token = APIToken.query.filter_by(
            id=token_id,
            user_id=current_user.id
        ).first_or_404()
        
        token_name = token.name
        db.session.delete(token)
        db.session.commit()
        
        logger.info(f"API token '{token_name}' deleted by user {current_user.username}")
        
        return jsonify({
            'message': f'API token "{token_name}" deleted successfully'
        }), 200
    
    except Exception as e:
        logger.error(f"Delete API token error: {str(e)}")
        return jsonify({'error': 'Failed to delete API token'}), 500

@csrf.exempt
@bp.route('/profile/tokens/<int:token_id>/revoke', methods=['POST'])
@login_required
@require_active_user
def revoke_api_token(token_id):
    """Revoke (deactivate) an API token"""
    try:
        from app.models import APIToken
        
        # Get token (ensure it belongs to current user)
        token = APIToken.query.filter_by(
            id=token_id,
            user_id=current_user.id
        ).first_or_404()
        
        if not token.is_active:
            return jsonify({'error': 'Token is already revoked'}), 400
        
        token.revoke()
        
        logger.info(f"API token '{token.name}' revoked by user {current_user.username}")
        
        return jsonify({
            'message': f'API token "{token_name}" revoked successfully',
            'token': token.to_dict()
        }), 200
    
    except Exception as e:
        logger.error(f"Revoke API token error: {str(e)}")
        return jsonify({'error': 'Failed to revoke API token'}), 500

@csrf.exempt
@bp.route('/profile/tokens/<int:token_id>/extend', methods=['POST'])
@login_required
@require_active_user
def extend_api_token(token_id):
    """Extend API token expiration"""
    try:
        from app.models import APIToken
        
        # Get token (ensure it belongs to current user)
        token = APIToken.query.filter_by(
            id=token_id,
            user_id=current_user.id
        ).first_or_404()
        
        data = request.get_json()
        days = data.get('days', 365) if data else 365
        
        # Validate days (between 1 and 1095 - 3 years max)
        if not isinstance(days, int) or days < 1 or days > 1095:
            return jsonify({'error': 'Days must be between 1 and 1095'}), 400
        
        token.extend_expiration(days)
        
        logger.info(f"API token '{token.name}' extended by {days} days by user {current_user.username}")
        
        return jsonify({
            'message': f'API token "{token.name}" extended by {days} days',
            'token': token.to_dict()
        }), 200
    
    except Exception as e:
        logger.error(f"Extend API token error: {str(e)}")
        return jsonify({'error': 'Failed to extend API token'}), 500

# API Routes with Token Authentication

@csrf.exempt
@bp.route('/api/matches')
def api_matches():
    """API endpoint to list matches (requires API token)"""
    from app.auth.jwt_utils import require_api_token, get_current_api_user
    
    @require_api_token
    def _api_matches():
        try:
            from app.models import Match
            
            # Get current API user
            api_user = get_current_api_user()
            
            # Pagination
            page = request.args.get('page', 1, type=int)
            per_page = min(request.args.get('per_page', 20, type=int), 100)
            
            # Base query
            if api_user.is_admin:
                query = Match.query
            else:
                query = Match.query.filter_by(created_by=api_user.id)
            
            # Apply filters
            fixture_id = request.args.get('fixture_id')
            if fixture_id:
                query = query.filter_by(fixture_id=fixture_id)
            
            active_only = request.args.get('active_only', '').lower() == 'true'
            if active_only:
                query = query.filter_by(active_status=True)
            
            # Pagination
            matches_pagination = query.order_by(Match.created_at.desc()).paginate(
                page=page, per_page=per_page, error_out=False
            )
            
            return jsonify({
                'matches': [match.to_dict() for match in matches_pagination.items],
                'pagination': {
                    'page': matches_pagination.page,
                    'pages': matches_pagination.pages,
                    'per_page': matches_pagination.per_page,
                    'total': matches_pagination.total,
                    'has_next': matches_pagination.has_next,
                    'has_prev': matches_pagination.has_prev
                }
            }), 200
        
        except Exception as e:
            logger.error(f"API matches error: {str(e)}")
            return jsonify({'error': 'Failed to retrieve matches'}), 500
    
    return _api_matches()

@csrf.exempt
@bp.route('/api/fixtures')
def api_fixtures():
    """API endpoint to list fixtures (requires API token)"""
    from app.auth.jwt_utils import require_api_token, get_current_api_user
    
    @require_api_token
    def _api_fixtures():
        try:
            from app.models import Match
            from sqlalchemy import func
            
            # Get current API user
            api_user = get_current_api_user()
            
            # Base query for fixtures (grouped by fixture_id)
            base_query = db.session.query(
                Match.fixture_id,
                Match.filename,
                func.count(Match.id).label('match_count'),
                func.min(Match.created_at).label('upload_date'),
                func.sum(Match.active_status.cast(db.Integer)).label('active_matches')
            ).group_by(Match.fixture_id, Match.filename)
            
            # Apply user filter
            if not api_user.is_admin:
                base_query = base_query.filter(Match.created_by == api_user.id)
            
            # Get fixtures
            fixtures = base_query.order_by(func.min(Match.created_at).desc()).all()
            
            # Convert to dict format
            fixtures_data = []
            for fixture in fixtures:
                fixtures_data.append({
                    'fixture_id': fixture.fixture_id,
                    'filename': fixture.filename,
                    'match_count': fixture.match_count,
                    'active_matches': fixture.active_matches,
                    'upload_date': fixture.upload_date.isoformat() if fixture.upload_date else None,
                    'status': 'complete' if fixture.active_matches == fixture.match_count else 'partial'
                })
            
            return jsonify({
                'fixtures': fixtures_data,
                'total': len(fixtures_data)
            }), 200
        
        except Exception as e:
            logger.error(f"API fixtures error: {str(e)}")
            return jsonify({'error': 'Failed to retrieve fixtures'}), 500
    
    return _api_fixtures()

@csrf.exempt
@bp.route('/api/match/<int:match_id>')
def api_match_detail(match_id):
    """API endpoint to get match details (requires API token)"""
    from app.auth.jwt_utils import require_api_token, get_current_api_user
    
    @require_api_token
    def _api_match_detail():
        try:
            from app.models import Match
            
            # Get current API user
            api_user = get_current_api_user()
            
            # Get match
            if api_user.is_admin:
                match = Match.query.get_or_404(match_id)
            else:
                match = Match.query.filter_by(id=match_id, created_by=api_user.id).first_or_404()
            
            return jsonify({
                'match': match.to_dict(include_outcomes=True)
            }), 200
        
        except Exception as e:
            logger.error(f"API match detail error: {str(e)}")
            return jsonify({'error': 'Failed to retrieve match details'}), 500
    
    return _api_match_detail()

@csrf.exempt
@bp.route('/download/zip/<int:match_id>')
@login_required
@require_active_user
def download_zip(match_id):
    """Download ZIP file for a match"""
    try:
        from app.models import Match
        from flask import send_file, abort
        
        # Get match
        if current_user.is_admin:
            match = Match.query.get_or_404(match_id)
        else:
            match = Match.query.filter_by(id=match_id, created_by=current_user.id).first_or_404()
        
        # Check if ZIP file exists and is completed
        if not match.zip_filename or match.zip_upload_status != 'completed':
            flash('ZIP file not available for this match', 'error')
            abort(404)
        
        # Construct file path
        zip_path = os.path.join(current_app.config.get('ZIP_UPLOADS_DIR', current_app.config.get('UPLOAD_FOLDER', 'uploads')), match.zip_filename)
        
        if not os.path.exists(zip_path):
            flash('ZIP file not found on disk', 'error')
            abort(404)
        
        # Log the download
        logger.info(f"ZIP file downloaded: {match.zip_filename} by user {current_user.username}")
        
        return send_file(zip_path, as_attachment=True, download_name=match.zip_filename)
        
    except Exception as e:
        logger.error(f"ZIP download error: {str(e)}")
        flash('Error downloading ZIP file', 'error')
        abort(500)

@csrf.exempt
@bp.route('/clients')
@login_required
@require_active_user
def clients():
    """Clients page showing connected clients with filtering and search"""
    try:
        from app.models import ClientActivity, SystemSettings, APIToken
        from datetime import datetime, timedelta
        
        # Clean up clients with inactive or revoked tokens
        # Delete ClientActivity records where the associated token is inactive or doesn't exist
        inactive_clients = ClientActivity.query.filter(
            db.or_(
                ~ClientActivity.api_token_id.in_(
                    db.session.query(APIToken.id).filter(APIToken.is_active == True)
                ),
                ClientActivity.api_token_id.is_(None)
            )
        ).all()

        if inactive_clients:
            for client in inactive_clients:
                db.session.delete(client)
            db.session.commit()
            logger.info(f"Cleaned up {len(inactive_clients)} client records with inactive/revoked tokens")

        # Get filter and search parameters
        status_filter = request.args.get('status')  # 'online', 'offline', or None for all
        search_query = request.args.get('search', '').strip()

        # Get remote domain setting
        remote_domain = SystemSettings.get_setting('remote_domain', 'townshipscombatleague.com')

        # Get clients with their associated token and user info
        clients_query = db.session.query(
            ClientActivity,
            APIToken.name.label('token_name'),
            APIToken.user_id,
            APIToken.created_at.label('token_created_at')
        ).join(
            APIToken, ClientActivity.api_token_id == APIToken.id
        ).filter(
            APIToken.is_active == True
        ).order_by(
            ClientActivity.last_seen.desc()
        )

        clients_data = []
        for client_activity, token_name, user_id, token_created_at in clients_query.all():
            # Get user info
            from app.models import User
            user = User.query.get(user_id)

            # Calculate if client is online (last seen within 30 minutes)
            now = datetime.utcnow()
            time_diff = now - client_activity.last_seen
            is_online = time_diff.total_seconds() <= 1800  # 30 minutes = 1800 seconds

            # Format last seen time
            last_seen_formatted = client_activity.last_seen.strftime('%Y-%m-%d %H:%M:%S')

            # Calculate time ago
            if time_diff.total_seconds() < 60:
                last_seen_ago = f"{int(time_diff.total_seconds())} seconds ago"
            elif time_diff.total_seconds() < 3600:
                last_seen_ago = f"{int(time_diff.total_seconds() / 60)} minutes ago"
            elif time_diff.total_seconds() < 86400:
                last_seen_ago = f"{int(time_diff.total_seconds() / 3600)} hours ago"
            else:
                last_seen_ago = f"{int(time_diff.total_seconds() / 86400)} days ago"

            clients_data.append({
                'rustdesk_id': client_activity.rustdesk_id,
                'token_name': token_name,
                'username': user.username if user else 'Unknown',
                'is_online': is_online,
                'last_seen': client_activity.last_seen,
                'last_seen_formatted': last_seen_formatted,
                'last_seen_ago': last_seen_ago,
                'ip_address': client_activity.ip_address,
                'user_agent': client_activity.user_agent,
                'remote_domain': remote_domain
            })

        # Apply search filter
        if search_query:
            search_pattern = search_query.lower()
            clients_data = [
                client for client in clients_data
                if (search_pattern in client['token_name'].lower() or
                    search_pattern in client['username'].lower() or
                    search_pattern in client['rustdesk_id'].lower() or
                    (client['ip_address'] and search_pattern in client['ip_address'].lower()))
            ]

        # Apply status filter
        if status_filter == 'online':
            clients_data = [client for client in clients_data if client['is_online']]
        elif status_filter == 'offline':
            clients_data = [client for client in clients_data if not client['is_online']]

        # Sort: online clients first, then offline clients by last seen
        clients_data.sort(key=lambda x: (not x['is_online'], x['last_seen']), reverse=True)

        return render_template('main/clients.html',
                             clients=clients_data,
                             status_filter=status_filter,
                             search_query=search_query)

    except Exception as e:
        logger.error(f"Clients page error: {str(e)}")
        flash('Error loading clients', 'error')
        return render_template('main/clients.html', clients=[])

# Reports Routes

@csrf.exempt
@bp.route('/reports')
@login_required
@require_active_user
def reports():
    """Reports page with filtering, pagination, and export"""
    try:
        from app.models import ReportSync, Bet, ExtractionStats, APIToken, ClientActivity
        from sqlalchemy import func, and_, or_
        
        # Get filter parameters
        client_id_filter = request.args.get('client_id', '').strip()
        date_range_filter = request.args.get('date_range', 'today').strip()
        start_date_filter = request.args.get('start_date', '').strip()
        end_date_filter = request.args.get('end_date', '').strip()
        start_time_filter = request.args.get('start_time', '').strip()
        end_time_filter = request.args.get('end_time', '').strip()
        sort_by = request.args.get('sort_by', 'sync_timestamp')
        sort_order = request.args.get('sort_order', 'desc')
        export_format = request.args.get('export', '').strip()
        
        # Pagination
        page = request.args.get('page', 1, type=int)
        per_page = min(request.args.get('per_page', 20, type=int), 100)
        
        # Calculate date range based on filter
        now = datetime.utcnow()
        start_date = None
        end_date = None
        
        if date_range_filter == 'today':
            start_date = now.replace(hour=0, minute=0, second=0, microsecond=0)
            end_date = now
        elif date_range_filter == 'yesterday':
            yesterday = now - timedelta(days=1)
            start_date = yesterday.replace(hour=0, minute=0, second=0, microsecond=0)
            end_date = yesterday.replace(hour=23, minute=59, second=59, microsecond=999999)
        elif date_range_filter == 'this_week':
            # Start of current week (Monday)
            start_date = now - timedelta(days=now.weekday())
            start_date = start_date.replace(hour=0, minute=0, second=0, microsecond=0)
            end_date = now
        elif date_range_filter == 'last_week':
            # Start of last week (Monday)
            last_week_end = now - timedelta(days=now.weekday() + 1)
            last_week_start = last_week_end - timedelta(days=6)
            start_date = last_week_start.replace(hour=0, minute=0, second=0, microsecond=0)
            end_date = last_week_end.replace(hour=23, minute=59, second=59, microsecond=999999)
        elif date_range_filter == 'this_month':
            start_date = now.replace(day=1, hour=0, minute=0, second=0, microsecond=0)
            end_date = now
        elif date_range_filter == 'all':
            start_date = None
            end_date = None
        elif date_range_filter == 'custom':
            # Use custom date range
            if start_date_filter:
                try:
                    start_date = datetime.strptime(start_date_filter, '%Y-%m-%d')
                    if start_time_filter:
                        hour, minute = map(int, start_time_filter.split(':'))
                        start_date = start_date.replace(hour=hour, minute=minute, second=0, microsecond=0)
                    else:
                        start_date = start_date.replace(hour=0, minute=0, second=0, microsecond=0)
                except ValueError:
                    pass
            if end_date_filter:
                try:
                    end_date = datetime.strptime(end_date_filter, '%Y-%m-%d')
                    if end_time_filter:
                        hour, minute = map(int, end_time_filter.split(':'))
                        end_date = end_date.replace(hour=hour, minute=minute, second=59, microsecond=999999)
                    else:
                        end_date = end_date.replace(hour=23, minute=59, second=59, microsecond=999999)
                except ValueError:
                    pass
        
        # Base query
        if current_user.is_admin:
            query = ReportSync.query
        else:
            # Non-admin users can only see reports from their own clients
            user_token_ids = [t.id for t in APIToken.query.filter_by(user_id=current_user.id).all()]
            if user_token_ids:
                # Get client_ids from ClientActivity for this user's tokens
                client_ids = [c.rustdesk_id for c in ClientActivity.query.filter(
                    ClientActivity.api_token_id.in_(user_token_ids)
                ).all()]
                if client_ids:
                    query = ReportSync.query.filter(ReportSync.client_id.in_(client_ids))
                else:
                    query = ReportSync.query.filter(ReportSync.client_id == 'none')
            else:
                query = ReportSync.query.filter(ReportSync.client_id == 'none')
        
        # Apply filters
        if client_id_filter:
            query = query.filter(ReportSync.client_id == client_id_filter)
        
        if start_date:
            query = query.filter(ReportSync.start_date >= start_date)
        
        if end_date:
            query = query.filter(ReportSync.end_date <= end_date)
        
        # Sorting
        if hasattr(ReportSync, sort_by):
            sort_column = getattr(ReportSync, sort_by)
            if sort_order == 'asc':
                query = query.order_by(sort_column.asc())
            else:
                query = query.order_by(sort_column.desc())
        else:
            query = query.order_by(ReportSync.sync_timestamp.desc())
        
        # Handle export
        if export_format:
            return export_reports(query, export_format)
        
        # Aggregate data by client for the selected period
        # Build base query with filters
        base_query = ReportSync.query
        
        # Apply user filter
        if not current_user.is_admin:
            user_token_ids = [t.id for t in APIToken.query.filter_by(user_id=current_user.id).all()]
            if user_token_ids:
                client_ids = [c.rustdesk_id for c in ClientActivity.query.filter(
                    ClientActivity.api_token_id.in_(user_token_ids)
                ).all()]
                if client_ids:
                    base_query = base_query.filter(ReportSync.client_id.in_(client_ids))
                else:
                    base_query = base_query.filter(ReportSync.client_id == 'none')
            else:
                base_query = base_query.filter(ReportSync.client_id == 'none')
        
        # Apply filters
        if client_id_filter:
            base_query = base_query.filter(ReportSync.client_id == client_id_filter)
        
        if start_date:
            base_query = base_query.filter(ReportSync.start_date >= start_date)
        
        if end_date:
            base_query = base_query.filter(ReportSync.end_date <= end_date)
        
        # Get all matching syncs for aggregation
        matching_syncs = base_query.all()
        
        # Aggregate by client
        client_aggregates = {}
        for sync in matching_syncs:
            client_id = sync.client_id
            if client_id not in client_aggregates:
                # Get token name for this client
                client_activity = ClientActivity.query.filter_by(rustdesk_id=client_id).first()
                token_name = client_activity.api_token.name if client_activity and client_activity.api_token else 'Unknown'
                
                client_aggregates[client_id] = {
                    'client_id': client_id,
                    'token_name': token_name,
                    'total_payin': 0.0,
                    'total_payout': 0.0,
                    'total_bets': 0,
                    'total_matches': 0,
                    'cap_balance': 0.0,
                    'last_sync_timestamp': sync.sync_timestamp
                }
            
            client_aggregates[client_id]['total_payin'] += float(sync.total_payin) if sync.total_payin else 0.0
            client_aggregates[client_id]['total_payout'] += float(sync.total_payout) if sync.total_payout else 0.0
            client_aggregates[client_id]['total_bets'] += sync.total_bets
            client_aggregates[client_id]['total_matches'] += sync.total_matches
            
            # Use the most recent CAP balance for this client
            if sync.sync_timestamp >= client_aggregates[client_id]['last_sync_timestamp']:
                client_aggregates[client_id]['cap_balance'] = float(sync.cap_compensation_balance) if sync.cap_compensation_balance else 0.0
                client_aggregates[client_id]['last_sync_timestamp'] = sync.sync_timestamp
        
        # Calculate balance for each client
        for client_id, data in client_aggregates.items():
            data['balance'] = data['total_payin'] - data['total_payout']
        
        # Convert to list and sort
        clients_list = list(client_aggregates.values())
        
        # Apply sorting
        if sort_by == 'total_payin':
            clients_list.sort(key=lambda x: x['total_payin'], reverse=(sort_order == 'desc'))
        elif sort_by == 'total_payout':
            clients_list.sort(key=lambda x: x['total_payout'], reverse=(sort_order == 'desc'))
        elif sort_by == 'net_profit':
            clients_list.sort(key=lambda x: x['balance'], reverse=(sort_order == 'desc'))
        elif sort_by == 'total_bets':
            clients_list.sort(key=lambda x: x['total_bets'], reverse=(sort_order == 'desc'))
        elif sort_by == 'total_matches':
            clients_list.sort(key=lambda x: x['total_matches'], reverse=(sort_order == 'desc'))
        else:
            clients_list.sort(key=lambda x: x['last_sync_timestamp'], reverse=(sort_order == 'desc'))
        
        # Calculate overall totals for summary blocks
        total_payin = sum(c['total_payin'] for c in clients_list)
        total_payout = sum(c['total_payout'] for c in clients_list)
        total_balance = total_payin - total_payout
        cap_balance = clients_list[0]['cap_balance'] if clients_list else 0.0
        
        # Pagination
        total_clients = len(clients_list)
        start_idx = (page - 1) * per_page
        end_idx = start_idx + per_page
        clients_page = clients_list[start_idx:end_idx]
        
        # Create pagination object
        class SimplePagination:
            def __init__(self, page, per_page, total, items):
                self.page = page
                self.per_page = per_page
                self.total = total
                self.items = items
                self.pages = (total + per_page - 1) // per_page
                self.has_prev = page > 1
                self.has_next = page < self.pages
                self.prev_num = page - 1 if self.has_prev else None
                self.next_num = page + 1 if self.has_next else None
        
        pagination = SimplePagination(page, per_page, total_clients, clients_page)
        
        # Get unique client IDs for filter dropdown with token names
        if current_user.is_admin:
            # Get all clients with their token names
            clients_query = db.session.query(
                ReportSync.client_id,
                APIToken.name.label('token_name')
            ).join(
                ClientActivity, ReportSync.client_id == ClientActivity.rustdesk_id
            ).join(
                APIToken, ClientActivity.api_token_id == APIToken.id
            ).filter(
                APIToken.is_active == True
            ).distinct().all()
            
            client_data = [{'client_id': c.client_id, 'token_name': c.token_name} for c in clients_query]
        else:
            if user_token_ids:
                clients_query = db.session.query(
                    ClientActivity.rustdesk_id.label('client_id'),
                    APIToken.name.label('token_name')
                ).join(
                    APIToken, ClientActivity.api_token_id == APIToken.id
                ).filter(
                    ClientActivity.api_token_id.in_(user_token_ids),
                    APIToken.is_active == True
                ).distinct().all()
                
                client_data = [{'client_id': c.client_id, 'token_name': c.token_name} for c in clients_query]
            else:
                client_data = []
        
        return render_template('main/reports.html',
                             clients=clients_page,
                             pagination=pagination,
                             client_data=client_data,
                             totals={
                                 'total_payin': total_payin,
                                 'total_payout': total_payout,
                                 'total_balance': total_balance,
                                 'cap_balance': cap_balance
                             },
                             filters={
                                 'client_id': client_id_filter,
                                 'date_range': date_range_filter,
                                 'start_date': start_date_filter,
                                 'end_date': end_date_filter,
                                 'start_time': start_time_filter,
                                 'end_time': end_time_filter,
                                 'sort_by': sort_by,
                                 'sort_order': sort_order
                             })
    
    except Exception as e:
        logger.error(f"Reports page error: {str(e)}")
        flash('Error loading reports', 'error')
        return render_template('main/reports.html', reports=[], pagination=None, client_ids=[])

@csrf.exempt
@bp.route('/reports/<int:sync_id>')
@login_required
@require_active_user
def report_detail(sync_id):
    """Detailed view of a specific report sync"""
    try:
        from app.models import ReportSync, Bet, ExtractionStats
        
        # Get report sync
        if current_user.is_admin:
            report = ReportSync.query.get_or_404(sync_id)
        else:
            # Check if user has access to this client
            from app.models import APIToken, ClientActivity
            user_token_ids = [t.id for t in APIToken.query.filter_by(user_id=current_user.id).all()]
            if user_token_ids:
                client_ids = [c.rustdesk_id for c in ClientActivity.query.filter(
                    ClientActivity.api_token_id.in_(user_token_ids)
                ).all()]
                report = ReportSync.query.filter(
                    and_(ReportSync.id == sync_id, ReportSync.client_id.in_(client_ids))
                ).first_or_404()
            else:
                report = ReportSync.query.filter_by(id=sync_id, client_id='none').first_or_404()
        
        # Get bets for this sync
        bets = Bet.query.filter_by(sync_id=sync_id).all()
        
        # Get extraction stats for this sync
        stats = ExtractionStats.query.filter_by(sync_id=sync_id).all()
        
        return render_template('main/report_detail.html',
                             report=report,
                             bets=bets,
                             stats=stats)
    
    except Exception as e:
        logger.error(f"Report detail error: {str(e)}")
        flash('Error loading report details', 'error')
        return redirect(url_for('main.reports'))

def export_reports(query, export_format):
    """Export reports to various formats"""
    try:
        from app.models import ReportSync, Bet, ExtractionStats
        
        # Get all reports matching query
        reports = query.all()
        
        if not reports:
            flash('No reports to export', 'warning')
            return redirect(url_for('main.reports'))
        
        # Prepare data for export
        export_data = []
        for report in reports:
            # Get bets and stats for this report
            bets = Bet.query.filter_by(sync_id=report.id).all()
            stats = ExtractionStats.query.filter_by(sync_id=report.id).all()
            
            # Add summary row
            export_data.append({
                'Type': 'Summary',
                'Sync ID': report.sync_id,
                'Client ID': report.client_id,
                'Sync Timestamp': report.sync_timestamp.strftime('%Y-%m-%d %H:%M:%S') if report.sync_timestamp else '',
                'Date Range': report.date_range,
                'Start Date': report.start_date.strftime('%Y-%m-%d') if report.start_date else '',
                'End Date': report.end_date.strftime('%Y-%m-%d') if report.end_date else '',
                'Total Payin': float(report.total_payin) if report.total_payin else 0.0,
                'Total Payout': float(report.total_payout) if report.total_payout else 0.0,
                'Net Profit': float(report.net_profit) if report.net_profit else 0.0,
                'Total Bets': report.total_bets,
                'Total Matches': report.total_matches,
                'Bet UUID': '',
                'Match ID': '',
                'Outcome': '',
                'Amount': '',
                'Result': ''
            })
            
            # Add bet details
            for bet in bets:
                for detail in bet.details:
                    export_data.append({
                        'Type': 'Bet Detail',
                        'Sync ID': report.sync_id,
                        'Client ID': report.client_id,
                        'Sync Timestamp': report.sync_timestamp.strftime('%Y-%m-%d %H:%M:%S') if report.sync_timestamp else '',
                        'Date Range': report.date_range,
                        'Start Date': report.start_date.strftime('%Y-%m-%d') if report.start_date else '',
                        'End Date': report.end_date.strftime('%Y-%m-%d') if report.end_date else '',
                        'Total Payin': '',
                        'Total Payout': '',
                        'Net Profit': '',
                        'Total Bets': '',
                        'Total Matches': '',
                        'Bet UUID': bet.uuid,
                        'Match ID': detail.match_id,
                        'Outcome': detail.outcome,
                        'Amount': float(detail.amount) if detail.amount else 0.0,
                        'Result': detail.result
                    })
            
            # Add extraction stats
            for stat in stats:
                export_data.append({
                    'Type': 'Extraction Stats',
                    'Sync ID': report.sync_id,
                    'Client ID': report.client_id,
                    'Sync Timestamp': report.sync_timestamp.strftime('%Y-%m-%d %H:%M:%S') if report.sync_timestamp else '',
                    'Date Range': report.date_range,
                    'Start Date': report.start_date.strftime('%Y-%m-%d') if report.start_date else '',
                    'End Date': report.end_date.strftime('%Y-%m-%d') if report.end_date else '',
                    'Total Payin': float(stat.total_amount_collected) if stat.total_amount_collected else 0.0,
                    'Total Payout': float(stat.total_redistributed) if stat.total_redistributed else 0.0,
                    'Net Profit': '',
                    'Total Bets': stat.total_bets,
                    'Total Matches': '',
                    'Bet UUID': '',
                    'Match ID': stat.match_id,
                    'Outcome': stat.actual_result,
                    'Amount': '',
                    'Result': stat.extraction_result
                })
        
        # Create DataFrame
        df = pd.DataFrame(export_data)
        
        # Export based on format
        if export_format == 'csv':
            output = StringIO()
            df.to_csv(output, index=False)
            output.seek(0)
            return Response(
                output.getvalue(),
                mimetype='text/csv',
                headers={'Content-Disposition': 'attachment; filename=reports_export.csv'}
            )
        
        elif export_format == 'xlsx':
            output = BytesIO()
            with pd.ExcelWriter(output, engine='openpyxl') as writer:
                df.to_excel(writer, index=False, sheet_name='Reports')
            output.seek(0)
            return Response(
                output.getvalue(),
                mimetype='application/vnd.openxmlformats-officedocument.spreadsheetml.sheet',
                headers={'Content-Disposition': 'attachment; filename=reports_export.xlsx'}
            )
        
        elif export_format == 'pdf':
            # For PDF, we'll create a simple HTML table and convert it
            from weasyprint import HTML
            import tempfile
            
            # Create HTML table
            html_table = df.to_html(index=False, classes='table table-striped table-bordered')
            html_content = f"""
            <!DOCTYPE html>
            <html>
            <head>
                <style>
                    body {{ font-family: Arial, sans-serif; margin: 20px; }}
                    table {{ border-collapse: collapse; width: 100%; margin-bottom: 20px; }}
                    th, td {{ border: 1px solid #ddd; padding: 8px; text-align: left; }}
                    th {{ background-color: #4CAF50; color: white; }}
                    tr:nth-child(even) {{ background-color: #f2f2f2; }}
                </style>
            </head>
            <body>
                <h1>Reports Export</h1>
                <p>Generated on: {datetime.utcnow().strftime('%Y-%m-%d %H:%M:%S')}</p>
                {html_table}
            </body>
            </html>
            """
            
            # Generate PDF
            pdf_bytes = HTML(string=html_content).write_pdf()
            return Response(
                pdf_bytes,
                mimetype='application/pdf',
                headers={'Content-Disposition': 'attachment; filename=reports_export.pdf'}
            )
        
        else:
            flash('Invalid export format', 'error')
            return redirect(url_for('main.reports'))
    
    except Exception as e:
        logger.error(f"Export reports error: {str(e)}")
        flash('Error exporting reports', 'error')
        return redirect(url_for('main.reports'))

@csrf.exempt
@bp.route('/sync-logs')
@login_required
@require_active_user
def sync_logs():
    """Sync logs page with filtering and search"""
    try:
        from app.models import ReportSyncLog
        from sqlalchemy import and_, or_
        
        # Check for export request
        export_format = request.args.get('export')
        if export_format:
            return export_sync_logs(export_format)
        
        # Get filter parameters
        client_id_filter = request.args.get('client_id', '').strip()
        operation_type_filter = request.args.get('operation_type', '').strip()
        status_filter = request.args.get('status', '').strip()
        search_query = request.args.get('search', '').strip()
        start_date_filter = request.args.get('start_date', '').strip()
        end_date_filter = request.args.get('end_date', '').strip()
        sort_by = request.args.get('sort_by', 'created_at')
        sort_order = request.args.get('sort_order', 'desc')
        
        # Pagination
        page = request.args.get('page', 1, type=int)
        per_page = min(request.args.get('per_page', 50, type=int), 200)
        
        # Base query
        if current_user.is_admin:
            query = ReportSyncLog.query
        else:
            # Non-admin users can only see logs from their own clients
            from app.models import APIToken, ClientActivity
            user_token_ids = [t.id for t in APIToken.query.filter_by(user_id=current_user.id).all()]
            if user_token_ids:
                client_ids = [c.rustdesk_id for c in ClientActivity.query.filter(
                    ClientActivity.api_token_id.in_(user_token_ids)
                ).all()]
                if client_ids:
                    query = ReportSyncLog.query.filter(ReportSyncLog.client_id.in_(client_ids))
                else:
                    query = ReportSyncLog.query.filter(ReportSyncLog.client_id == 'none')
            else:
                query = ReportSyncLog.query.filter(ReportSyncLog.client_id == 'none')
        
        # Apply filters
        if client_id_filter:
            query = query.filter(ReportSyncLog.client_id == client_id_filter)
        
        if operation_type_filter:
            query = query.filter(ReportSyncLog.operation_type == operation_type_filter)
        
        if status_filter:
            query = query.filter(ReportSyncLog.status == status_filter)
        
        if start_date_filter:
            try:
                start_date = datetime.strptime(start_date_filter, '%Y-%m-%d')
                query = query.filter(ReportSyncLog.created_at >= start_date)
            except ValueError:
                pass
        
        if end_date_filter:
            try:
                end_date = datetime.strptime(end_date_filter, '%Y-%m-%d')
                # Include the entire end date
                end_date = end_date.replace(hour=23, minute=59, second=59)
                query = query.filter(ReportSyncLog.created_at <= end_date)
            except ValueError:
                pass
        
        # Search functionality
        if search_query:
            search_pattern = f"%{search_query}%"
            query = query.filter(
                or_(
                    ReportSyncLog.sync_id.ilike(search_pattern),
                    ReportSyncLog.client_id.ilike(search_pattern),
                    ReportSyncLog.error_message.ilike(search_pattern)
                )
            )
        
        # Sorting
        if hasattr(ReportSyncLog, sort_by):
            sort_column = getattr(ReportSyncLog, sort_by)
            if sort_order == 'asc':
                query = query.order_by(sort_column.asc())
            else:
                query = query.order_by(sort_column.desc())
        else:
            query = query.order_by(ReportSyncLog.created_at.desc())
        
        # Pagination
        logs_pagination = query.paginate(page=page, per_page=per_page, error_out=False)
        
        # Get unique client IDs for filter dropdown
        if current_user.is_admin:
            all_client_ids = db.session.query(ReportSyncLog.client_id).distinct().all()
        else:
            if user_token_ids:
                all_client_ids = db.session.query(ClientActivity.rustdesk_id).filter(
                    ClientActivity.api_token_id.in_(user_token_ids)
                ).distinct().all()
            else:
                all_client_ids = []
        
        client_ids = [cid[0] for cid in all_client_ids if cid[0]]
        
        return render_template('main/sync_logs.html',
                             logs=logs_pagination.items,
                             pagination=logs_pagination,
                             client_ids=client_ids,
                             filters={
                                 'client_id': client_id_filter,
                                 'operation_type': operation_type_filter,
                                 'status': status_filter,
                                 'search': search_query,
                                 'start_date': start_date_filter,
                                 'end_date': end_date_filter,
                                 'sort_by': sort_by,
                                 'sort_order': sort_order
                             })
    
    except Exception as e:
        logger.error(f"Sync logs page error: {str(e)}")
        flash('Error loading sync logs', 'error')
        return render_template('main/sync_logs.html', logs=[], pagination=None, client_ids=[])

def export_sync_logs(export_format):
    """Export sync logs to CSV, XLSX, or PDF"""
    try:
        from app.models import ReportSyncLog
        from sqlalchemy import and_, or_
        
        # Get filter parameters
        client_id_filter = request.args.get('client_id', '').strip()
        operation_type_filter = request.args.get('operation_type', '').strip()
        status_filter = request.args.get('status', '').strip()
        search_query = request.args.get('search', '').strip()
        start_date_filter = request.args.get('start_date', '').strip()
        end_date_filter = request.args.get('end_date', '').strip()
        
        # Base query
        if current_user.is_admin:
            query = ReportSyncLog.query
        else:
            # Non-admin users can only see logs from their own clients
            from app.models import APIToken, ClientActivity
            user_token_ids = [t.id for t in APIToken.query.filter_by(user_id=current_user.id).all()]
            if user_token_ids:
                client_ids = [c.rustdesk_id for c in ClientActivity.query.filter(
                    ClientActivity.api_token_id.in_(user_token_ids)
                ).all()]
                if client_ids:
                    query = ReportSyncLog.query.filter(ReportSyncLog.client_id.in_(client_ids))
                else:
                    query = ReportSyncLog.query.filter(ReportSyncLog.client_id == 'none')
            else:
                query = ReportSyncLog.query.filter(ReportSyncLog.client_id == 'none')
        
        # Apply filters
        if client_id_filter:
            query = query.filter(ReportSyncLog.client_id == client_id_filter)
        
        if operation_type_filter:
            query = query.filter(ReportSyncLog.operation_type == operation_type_filter)
        
        if status_filter:
            query = query.filter(ReportSyncLog.status == status_filter)
        
        if start_date_filter:
            try:
                start_date = datetime.strptime(start_date_filter, '%Y-%m-%d')
                query = query.filter(ReportSyncLog.created_at >= start_date)
            except ValueError:
                pass
        
        if end_date_filter:
            try:
                end_date = datetime.strptime(end_date_filter, '%Y-%m-%d')
                end_date = end_date.replace(hour=23, minute=59, second=59)
                query = query.filter(ReportSyncLog.created_at <= end_date)
            except ValueError:
                pass
        
        # Search functionality
        if search_query:
            search_pattern = f"%{search_query}%"
            query = query.filter(
                or_(
                    ReportSyncLog.sync_id.ilike(search_pattern),
                    ReportSyncLog.client_id.ilike(search_pattern),
                    ReportSyncLog.error_message.ilike(search_pattern)
                )
            )
        
        # Get all logs (no pagination for export)
        logs = query.order_by(ReportSyncLog.created_at.desc()).all()
        
        if export_format == 'csv':
            # Export to CSV
            output = BytesIO()
            data = []
            for log in logs:
                data.append({
                    'Sync ID': log.sync_id,
                    'Client ID': log.client_id,
                    'Sync Timestamp': log.sync_timestamp.strftime('%Y-%m-%d %H:%M:%S') if log.sync_timestamp else '',
                    'Operation Type': log.operation_type,
                    'Status': log.status,
                    'Bets Processed': log.bets_processed,
                    'Bets New': log.bets_new,
                    'Bets Duplicate': log.bets_duplicate,
                    'Stats Processed': log.stats_processed,
                    'Stats New': log.stats_new,
                    'Stats Updated': log.stats_updated,
                    'Total Payin': float(log.total_payin) if log.total_payin else 0.0,
                    'Total Payout': float(log.total_payout) if log.total_payout else 0.0,
                    'Net Profit': float(log.net_profit) if log.net_profit else 0.0,
                    'Total Bets': log.total_bets,
                    'Total Matches': log.total_matches,
                    'Error Message': log.error_message or '',
                    'IP Address': log.ip_address or '',
                    'User Agent': log.user_agent or '',
                    'Request Size (bytes)': log.request_size or 0,
                    'Processing Time (ms)': log.processing_time_ms or 0,
                    'Created At': log.created_at.strftime('%Y-%m-%d %H:%M:%S') if log.created_at else ''
                })
            
            df = pd.DataFrame(data)
            df.to_csv(output, index=False)
            output.seek(0)
            
            response = make_response(output.read())
            response.headers['Content-Type'] = 'text/csv'
            response.headers['Content-Disposition'] = 'attachment; filename=sync_logs_export.csv'
            return response
        
        elif export_format == 'xlsx':
            # Export to Excel
            output = BytesIO()
            data = []
            for log in logs:
                data.append({
                    'Sync ID': log.sync_id,
                    'Client ID': log.client_id,
                    'Sync Timestamp': log.sync_timestamp.strftime('%Y-%m-%d %H:%M:%S') if log.sync_timestamp else '',
                    'Operation Type': log.operation_type,
                    'Status': log.status,
                    'Bets Processed': log.bets_processed,
                    'Bets New': log.bets_new,
                    'Bets Duplicate': log.bets_duplicate,
                    'Stats Processed': log.stats_processed,
                    'Stats New': log.stats_new,
                    'Stats Updated': log.stats_updated,
                    'Total Payin': float(log.total_payin) if log.total_payin else 0.0,
                    'Total Payout': float(log.total_payout) if log.total_payout else 0.0,
                    'Net Profit': float(log.net_profit) if log.net_profit else 0.0,
                    'Total Bets': log.total_bets,
                    'Total Matches': log.total_matches,
                    'Error Message': log.error_message or '',
                    'IP Address': log.ip_address or '',
                    'User Agent': log.user_agent or '',
                    'Request Size (bytes)': log.request_size or 0,
                    'Processing Time (ms)': log.processing_time_ms or 0,
                    'Created At': log.created_at.strftime('%Y-%m-%d %H:%M:%S') if log.created_at else ''
                })
            
            df = pd.DataFrame(data)
            with pd.ExcelWriter(output, engine='openpyxl') as writer:
                df.to_excel(writer, index=False, sheet_name='Sync Logs')
            
            output.seek(0)
            
            response = make_response(output.read())
            response.headers['Content-Type'] = 'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet'
            response.headers['Content-Disposition'] = 'attachment; filename=sync_logs_export.xlsx'
            return response
        
        elif export_format == 'pdf':
            # Export to PDF
            from weasyprint import HTML, CSS
            import os
            
            # Create HTML table
            html_content = f"""
            <!DOCTYPE html>
            <html>
            <head>
                <meta charset="UTF-8">
                <style>
                    body {{ font-family: Arial, sans-serif; margin: 20px; }}
                    h1 {{ color: #333; border-bottom: 2px solid #007bff; padding-bottom: 10px; }}
                    table {{ width: 100%; border-collapse: collapse; margin-top: 20px; }}
                    th {{ background-color: #007bff; color: white; padding: 10px; text-align: left; font-weight: bold; }}
                    td {{ padding: 8px; border: 1px solid #ddd; }}
                    tr:nth-child(even) {{ background-color: #f9f9f9; }}
                    .success {{ color: #28a745; }}
                    .failed {{ color: #dc3545; }}
                    .warning {{ color: #ffc107; }}
                    .info {{ color: #17a2b8; }}
                </style>
            </head>
            <body>
                <h1>Sync Logs Export</h1>
                <p><strong>Generated:</strong> {datetime.utcnow().strftime('%Y-%m-%d %H:%M:%S')}</p>
                <p><strong>Total Records:</strong> {len(logs)}</p>
                <table>
                    <thead>
                        <tr>
                            <th>Sync ID</th>
                            <th>Client ID</th>
                            <th>Sync Timestamp</th>
                            <th>Operation Type</th>
                            <th>Status</th>
                            <th>Bets Processed</th>
                            <th>Stats Processed</th>
                            <th>Total Payin</th>
                            <th>Total Payout</th>
                            <th>Net Profit</th>
                            <th>Processing Time</th>
                            <th>Created At</th>
                        </tr>
                    </thead>
                    <tbody>
            """
            
            for log in logs:
                status_class = 'success' if log.status == 'success' else 'failed'
                operation_class = {
                    'new_sync': 'info',
                    'duplicate_sync': 'info',
                    'update_stats': 'warning',
                    'error': 'danger'
                }.get(log.operation_type, 'info')
                
                html_content += f"""
                        <tr>
                            <td>{log.sync_id}</td>
                            <td>{log.client_id}</td>
                            <td>{log.sync_timestamp.strftime('%Y-%m-%d %H:%M:%S') if log.sync_timestamp else ''}</td>
                            <td class="{operation_class}">{log.operation_type.replace('_', ' ').title()}</td>
                            <td class="{status_class}">{log.status.title()}</td>
                            <td>{log.bets_processed}</td>
                            <td>{log.stats_processed}</td>
                            <td>{float(log.total_payin) if log.total_payin else 0.0:.2f}</td>
                            <td>{float(log.total_payout) if log.total_payout else 0.0:.2f}</td>
                            <td>{float(log.net_profit) if log.net_profit else 0.0:.2f}</td>
                            <td>{log.processing_time_ms}ms</td>
                            <td>{log.created_at.strftime('%Y-%m-%d %H:%M:%S') if log.created_at else ''}</td>
                        </tr>
                """
            
            html_content += """
                    </tbody>
                </table>
            </body>
            </html>
            """
            
            # Generate PDF
            pdf_bytes = HTML(string=html_content).write_pdf()
            
            response = make_response(pdf_bytes)
            response.headers['Content-Type'] = 'application/pdf'
            response.headers['Content-Disposition'] = 'attachment; filename=sync_logs_export.pdf'
            return response
        
        else:
            flash('Invalid export format', 'error')
            return redirect(url_for('main.sync_logs'))
    
    except Exception as e:
        logger.error(f"Export sync logs error: {str(e)}")
        flash('Error exporting sync logs', 'error')

@csrf.exempt
@bp.route('/client-report/<client_id>')
@login_required
@require_active_user
def client_report_detail(client_id):
    """Detailed view of a specific client for selected period"""
    try:
        from app.models import ReportSync, Bet, ExtractionStats, APIToken, ClientActivity
        from datetime import timedelta
        
        # Get filter parameters
        date_range_filter = request.args.get('date_range', 'today').strip()
        start_date_filter = request.args.get('start_date', '').strip()
        end_date_filter = request.args.get('end_date', '').strip()
        start_time_filter = request.args.get('start_time', '').strip()
        end_time_filter = request.args.get('end_time', '').strip()
        
        # Calculate date range
        now = datetime.utcnow()
        start_date = None
        end_date = None
        
        if date_range_filter == 'today':
            start_date = now.replace(hour=0, minute=0, second=0, microsecond=0)
            end_date = now
        elif date_range_filter == 'yesterday':
            yesterday = now - timedelta(days=1)
            start_date = yesterday.replace(hour=0, minute=0, second=0, microsecond=0)
            end_date = yesterday.replace(hour=23, minute=59, second=59, microsecond=999999)
        elif date_range_filter == 'this_week':
            start_date = now - timedelta(days=now.weekday())
            start_date = start_date.replace(hour=0, minute=0, second=0, microsecond=0)
            end_date = now
        elif date_range_filter == 'last_week':
            last_week_end = now - timedelta(days=now.weekday() + 1)
            last_week_start = last_week_end - timedelta(days=6)
            start_date = last_week_start.replace(hour=0, minute=0, second=0, microsecond=0)
            end_date = last_week_end.replace(hour=23, minute=59, second=59, microsecond=999999)
        elif date_range_filter == 'this_month':
            start_date = now.replace(day=1, hour=0, minute=0, second=0, microsecond=0)
            end_date = now
        elif date_range_filter == 'all':
            start_date = None
            end_date = None
        elif date_range_filter == 'custom':
            if start_date_filter:
                try:
                    start_date = datetime.strptime(start_date_filter, '%Y-%m-%d')
                    if start_time_filter:
                        hour, minute = map(int, start_time_filter.split(':'))
                        start_date = start_date.replace(hour=hour, minute=minute, second=0, microsecond=0)
                    else:
                        start_date = start_date.replace(hour=0, minute=0, second=0, microsecond=0)
                except ValueError:
                    pass
            if end_date_filter:
                try:
                    end_date = datetime.strptime(end_date_filter, '%Y-%m-%d')
                    if end_time_filter:
                        hour, minute = map(int, end_time_filter.split(':'))
                        end_date = end_date.replace(hour=hour, minute=minute, second=59, microsecond=999999)
                    else:
                        end_date = end_date.replace(hour=23, minute=59, second=59, microsecond=999999)
                except ValueError:
                    pass
        
        # Check if user has access to this client
        if current_user.is_admin:
            query = ReportSync.query.filter_by(client_id=client_id)
        else:
            user_token_ids = [t.id for t in APIToken.query.filter_by(user_id=current_user.id).all()]
            if user_token_ids:
                client_ids = [c.rustdesk_id for c in ClientActivity.query.filter(
                    ClientActivity.api_token_id.in_(user_token_ids)
                ).all()]
                if client_id not in client_ids:
                    flash('Access denied to this client', 'error')
                    return redirect(url_for('main.reports'))
                query = ReportSync.query.filter_by(client_id=client_id)
            else:
                flash('Access denied to this client', 'error')
                return redirect(url_for('main.reports'))
        
        # Apply date filters
        if start_date:
            query = query.filter(ReportSync.start_date >= start_date)
        if end_date:
            query = query.filter(ReportSync.end_date <= end_date)
        
        # Get all matching syncs for this client
        syncs = query.order_by(ReportSync.sync_timestamp.desc()).all()
        
        # Get all bets and extraction stats for these syncs
        sync_ids = [s.id for s in syncs]
        bets = Bet.query.filter(Bet.sync_id.in_(sync_ids)).all()
        stats = ExtractionStats.query.filter(ExtractionStats.sync_id.in_(sync_ids)).all()
        
        # Calculate totals
        total_payin = sum(float(s.total_payin) for s in syncs if s.total_payin)
        total_payout = sum(float(s.total_payout) for s in syncs if s.total_payout)
        total_balance = total_payin - total_payout
        total_bets = sum(s.total_bets for s in syncs)
        total_matches = sum(s.total_matches for s in syncs)
        cap_balance = float(syncs[0].cap_compensation_balance) if syncs and syncs[0].cap_compensation_balance else 0.0
        
        # Get client token name
        client_activity = ClientActivity.query.filter_by(rustdesk_id=client_id).first()
        token_name = client_activity.api_token.name if client_activity and client_activity.api_token else 'Unknown'
        
        return render_template('main/client_report_detail.html',
                             client_id=client_id,
                             token_name=token_name,
                             syncs=syncs,
                             bets=bets,
                             stats=stats,
                             totals={
                                 'total_payin': total_payin,
                                 'total_payout': total_payout,
                                 'total_balance': total_balance,
                                 'total_bets': total_bets,
                                 'total_matches': total_matches,
                                 'cap_balance': cap_balance
                             },
                             filters={
                                 'date_range': date_range_filter,
                                 'start_date': start_date_filter,
                                 'end_date': end_date_filter,
                                 'start_time': start_time_filter,
                                 'end_time': end_time_filter
                             })
    
    except Exception as e:
        logger.error(f"Client report detail error: {str(e)}")
        flash('Error loading client report details', 'error')
        return redirect(url_for('main.reports'))
        return redirect(url_for('main.sync_logs'))