"""
Database migration system for Fixture Manager
Handles automatic schema updates and versioning
"""

import logging
import os
from datetime import datetime
from typing import List, Dict, Any
from app import db
from sqlalchemy import text, inspect
from sqlalchemy.exc import SQLAlchemyError

logger = logging.getLogger(__name__)

class DatabaseVersion(db.Model):
    """Track database schema versions"""
    __tablename__ = 'database_versions'
    
    id = db.Column(db.Integer, primary_key=True)
    version = db.Column(db.String(50), unique=True, nullable=False)
    description = db.Column(db.String(255), nullable=False)
    applied_at = db.Column(db.DateTime, default=datetime.utcnow)
    
    def __repr__(self):
        return f'<DatabaseVersion {self.version}: {self.description}>'

class Migration:
    """Base migration class"""
    
    def __init__(self, version: str, description: str):
        self.version = version
        self.description = description
    
    def up(self):
        """Apply the migration"""
        raise NotImplementedError("Subclasses must implement up() method")
    
    def down(self):
        """Rollback the migration (optional)"""
        pass
    
    def can_rollback(self) -> bool:
        """Check if migration can be rolled back"""
        return False

class Migration_001_RemoveFixtureIdUnique(Migration):
    """Remove unique constraint from fixture_id in matches table"""
    
    def __init__(self):
        super().__init__("001", "Remove unique constraint from fixture_id in matches table")
    
    def up(self):
        """Remove unique constraint from fixture_id"""
        try:
            # Check if we're using MySQL or SQLite
            inspector = inspect(db.engine)
            
            # Get current constraints
            constraints = inspector.get_unique_constraints('matches')
            indexes = inspector.get_indexes('matches')
            
            # Find fixture_id unique constraint/index
            fixture_id_constraint = None
            fixture_id_index = None
            
            for constraint in constraints:
                if 'fixture_id' in constraint['column_names']:
                    fixture_id_constraint = constraint['name']
                    break
            
            for index in indexes:
                if 'fixture_id' in index['column_names'] and index['unique']:
                    fixture_id_index = index['name']
                    break
            
            # Drop unique constraint if it exists
            if fixture_id_constraint:
                try:
                    with db.engine.connect() as conn:
                        conn.execute(text(f"ALTER TABLE matches DROP CONSTRAINT {fixture_id_constraint}"))
                        conn.commit()
                    logger.info(f"Dropped unique constraint {fixture_id_constraint} from matches.fixture_id")
                except Exception as e:
                    logger.warning(f"Could not drop constraint {fixture_id_constraint}: {str(e)}")
            
            # Drop unique index if it exists
            if fixture_id_index:
                try:
                    with db.engine.connect() as conn:
                        conn.execute(text(f"DROP INDEX {fixture_id_index}"))
                        conn.commit()
                    logger.info(f"Dropped unique index {fixture_id_index} from matches.fixture_id")
                except Exception as e:
                    logger.warning(f"Could not drop index {fixture_id_index}: {str(e)}")
            
            # Create regular index for performance (non-unique)
            try:
                with db.engine.connect() as conn:
                    conn.execute(text("CREATE INDEX IF NOT EXISTS idx_matches_fixture_id ON matches(fixture_id)"))
                    conn.commit()
                logger.info("Created non-unique index on matches.fixture_id")
            except Exception as e:
                logger.warning(f"Could not create index on fixture_id: {str(e)}")
            
            return True
            
        except Exception as e:
            logger.error(f"Migration 001 failed: {str(e)}")
            raise
    
    def can_rollback(self) -> bool:
        return False  # Cannot safely rollback unique constraint removal

class Migration_002_AddDatabaseVersionTable(Migration):
    """Ensure database version tracking table exists"""
    
    def __init__(self):
        super().__init__("002", "Create database version tracking table")
    
    def up(self):
        """Create database version table if it doesn't exist"""
        try:
            # This migration is handled by the migration system itself
            # when it creates the DatabaseVersion table
            return True
        except Exception as e:
            logger.error(f"Migration 002 failed: {str(e)}")
            raise

class Migration_003_CreateAPITokensTable(Migration):
    """Create API tokens table for external authentication"""
    
    def __init__(self):
        super().__init__("003", "Create API tokens table for external authentication")
    
    def up(self):
        """Create api_tokens table"""
        try:
            # Check if table already exists
            inspector = inspect(db.engine)
            if 'api_tokens' in inspector.get_table_names():
                logger.info("api_tokens table already exists, skipping creation")
                return True
            
            # Create the table using raw SQL to ensure compatibility
            create_table_sql = '''
                CREATE TABLE api_tokens (
                    id INT AUTO_INCREMENT PRIMARY KEY,
                    user_id INT NOT NULL,
                    name VARCHAR(255) NOT NULL,
                    token_hash VARCHAR(255) NOT NULL UNIQUE,
                    is_active BOOLEAN DEFAULT TRUE,
                    expires_at DATETIME NOT NULL,
                    last_used_at DATETIME NULL,
                    last_used_ip VARCHAR(45) NULL,
                    created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
                    updated_at DATETIME DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
                    INDEX idx_api_tokens_user_id (user_id),
                    INDEX idx_api_tokens_token_hash (token_hash),
                    INDEX idx_api_tokens_is_active (is_active),
                    INDEX idx_api_tokens_expires_at (expires_at),
                    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE
                ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;
            '''
            
            with db.engine.connect() as conn:
                conn.execute(text(create_table_sql))
                conn.commit()
            
            logger.info("Created api_tokens table successfully")
            return True
            
        except Exception as e:
            logger.error(f"Migration 003 failed: {str(e)}")
            raise
    
    def down(self):
        """Drop api_tokens table"""
        try:
            with db.engine.connect() as conn:
                conn.execute(text("DROP TABLE IF EXISTS api_tokens"))
                conn.commit()
            logger.info("Dropped api_tokens table")
            return True
        except Exception as e:
            logger.error(f"Rollback of migration 003 failed: {str(e)}")
            raise
    
    def can_rollback(self) -> bool:
        return True

class Migration_004_CreateSystemSettingsTable(Migration):
    """Create system settings table for persistent configuration"""
    
    def __init__(self):
        super().__init__("004", "Create system settings table for persistent configuration")
    
    def up(self):
        """Create system_settings table"""
        try:
            # Check if table already exists
            inspector = inspect(db.engine)
            if 'system_settings' in inspector.get_table_names():
                logger.info("system_settings table already exists, skipping creation")
                return True
            
            # Create the table using raw SQL to ensure compatibility
            create_table_sql = '''
                CREATE TABLE system_settings (
                    id INT AUTO_INCREMENT PRIMARY KEY,
                    setting_key VARCHAR(255) NOT NULL UNIQUE,
                    setting_value TEXT,
                    setting_type VARCHAR(50) DEFAULT 'string',
                    description TEXT,
                    created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
                    updated_at DATETIME DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
                    INDEX idx_system_settings_key (setting_key),
                    INDEX idx_system_settings_type (setting_type)
                ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;
            '''
            
            with db.engine.connect() as conn:
                conn.execute(text(create_table_sql))
                conn.commit()
            
            # Insert default settings
            default_settings_sql = '''
                INSERT INTO system_settings (setting_key, setting_value, setting_type, description) VALUES
                ('registration_enabled', 'false', 'boolean', 'Enable or disable user registration'),
                ('app_name', 'Fixture Manager', 'string', 'Application name'),
                ('maintenance_mode', 'false', 'boolean', 'Enable maintenance mode')
            '''
            
            with db.engine.connect() as conn:
                conn.execute(text(default_settings_sql))
                conn.commit()
            
            logger.info("Created system_settings table with default settings successfully")
            return True
            
        except Exception as e:
            logger.error(f"Migration 004 failed: {str(e)}")
            raise
    
    def down(self):
        """Drop system_settings table"""
        try:
            with db.engine.connect() as conn:
                conn.execute(text("DROP TABLE IF EXISTS system_settings"))
                conn.commit()
            logger.info("Dropped system_settings table")
            return True
        except Exception as e:
            logger.error(f"Rollback of migration 004 failed: {str(e)}")
            raise
    
    def can_rollback(self) -> bool:
        return True

class Migration_005_AddFixtureActiveTime(Migration):
    """Add fixture active time column to matches table"""
    
    def __init__(self):
        super().__init__("005", "Add fixture active time unix timestamp column to matches table")
    
    def up(self):
        """Add fixture_active_time column"""
        try:
            # Check if column already exists
            inspector = inspect(db.engine)
            columns = [col['name'] for col in inspector.get_columns('matches')]
            
            if 'fixture_active_time' in columns:
                logger.info("fixture_active_time column already exists, skipping creation")
                return True
            
            # Add the column
            alter_table_sql = '''
                ALTER TABLE matches 
                ADD COLUMN fixture_active_time BIGINT NULL COMMENT 'Unix timestamp when all matches in fixture became active'
            '''
            
            with db.engine.connect() as conn:
                conn.execute(text(alter_table_sql))
                conn.commit()
            
            # Create index for performance
            create_index_sql = '''
                CREATE INDEX idx_matches_fixture_active_time ON matches(fixture_active_time)
            '''
            
            with db.engine.connect() as conn:
                conn.execute(text(create_index_sql))
                conn.commit()
            
            logger.info("Added fixture_active_time column and index successfully")
            return True
            
        except Exception as e:
            logger.error(f"Migration 005 failed: {str(e)}")
            raise
    
    def down(self):
        """Drop fixture_active_time column"""
        try:
            with db.engine.connect() as conn:
                conn.execute(text("DROP INDEX IF EXISTS idx_matches_fixture_active_time"))
                conn.execute(text("ALTER TABLE matches DROP COLUMN IF EXISTS fixture_active_time"))
                conn.commit()
            logger.info("Dropped fixture_active_time column and index")
            return True
        except Exception as e:
            logger.error(f"Rollback of migration 005 failed: {str(e)}")
            raise
    
    def can_rollback(self) -> bool:
        return True

class Migration_006_AddStatusColumn(Migration):
    """Add status column to matches table"""
    
    def __init__(self):
        super().__init__("006", "Add status column to matches table")
    
    def up(self):
        """Add status column"""
        try:
            # Check if column already exists
            inspector = inspect(db.engine)
            columns = [col['name'] for col in inspector.get_columns('matches')]
            
            if 'status' in columns:
                logger.info("status column already exists, skipping creation")
                return True
            
            # Add the column
            alter_table_sql = '''
                ALTER TABLE matches
                ADD COLUMN status ENUM('pending', 'scheduled', 'bet', 'ingame', 'cancelled', 'failed', 'paused', 'done') DEFAULT 'pending'
            '''
            
            with db.engine.connect() as conn:
                conn.execute(text(alter_table_sql))
                conn.commit()
            
            # Create index for performance
            create_index_sql = '''
                CREATE INDEX idx_matches_status ON matches(status)
            '''
            
            with db.engine.connect() as conn:
                conn.execute(text(create_index_sql))
                conn.commit()
            
            logger.info("Added status column and index successfully")
            return True
            
        except Exception as e:
            logger.error(f"Migration 006 failed: {str(e)}")
            raise
    
    def down(self):
        """Drop status column"""
        try:
            with db.engine.connect() as conn:
                conn.execute(text("DROP INDEX IF EXISTS idx_matches_status"))
                conn.execute(text("ALTER TABLE matches DROP COLUMN IF EXISTS status"))
                conn.commit()
            logger.info("Dropped status column and index")
            return True
        except Exception as e:
            logger.error(f"Rollback of migration 006 failed: {str(e)}")
            raise
    
    def can_rollback(self) -> bool:
        return True

class Migration_007_AddDoneToStatusEnum(Migration):
    """Add 'done' value to existing status ENUM column"""
    
    def __init__(self):
        super().__init__("007", "Add 'done' value to existing status ENUM column if not present")
    
    def up(self):
        """Add 'done' to status ENUM if not already present"""
        try:
            # Check if status column exists
            inspector = inspect(db.engine)
            columns = inspector.get_columns('matches')
            
            status_column = None
            for col in columns:
                if col['name'] == 'status':
                    status_column = col
                    break
            
            if not status_column:
                logger.info("status column does not exist, skipping ENUM modification")
                return True
            
            # Check current ENUM values by trying to insert 'done' and seeing if it fails
            try:
                with db.engine.connect() as conn:
                    # Try a test query to see if 'done' is already valid
                    result = conn.execute(text("SELECT 1 FROM matches WHERE status = 'done' LIMIT 1"))
                    logger.info("'done' value already exists in status ENUM, skipping modification")
                    return True
            except Exception:
                # 'done' is not a valid ENUM value, so we need to add it
                pass
            
            # Add 'done' to the ENUM
            alter_enum_sql = '''
                ALTER TABLE matches
                MODIFY COLUMN status ENUM('pending', 'scheduled', 'bet', 'ingame', 'cancelled', 'failed', 'paused', 'done') DEFAULT 'pending'
            '''
            
            with db.engine.connect() as conn:
                conn.execute(text(alter_enum_sql))
                conn.commit()
            
            logger.info("Added 'done' value to status ENUM successfully")
            return True
            
        except Exception as e:
            logger.error(f"Migration 007 failed: {str(e)}")
            raise
    
    def down(self):
        """Remove 'done' from status ENUM (not recommended as it may cause data loss)"""
        try:
            # Check if any records use 'done' status
            with db.engine.connect() as conn:
                result = conn.execute(text("SELECT COUNT(*) as count FROM matches WHERE status = 'done'"))
                row = result.fetchone()
                if row and row.count > 0:
                    logger.warning(f"Cannot rollback migration 007: {row.count} records use 'done' status")
                    return False
            
            # Remove 'done' from ENUM
            alter_enum_sql = '''
                ALTER TABLE matches
                MODIFY COLUMN status ENUM('pending', 'scheduled', 'bet', 'ingame', 'cancelled', 'failed', 'paused') DEFAULT 'pending'
            '''
            
            with db.engine.connect() as conn:
                conn.execute(text(alter_enum_sql))
                conn.commit()
            
            logger.info("Removed 'done' value from status ENUM")
            return True
        except Exception as e:
            logger.error(f"Rollback of migration 007 failed: {str(e)}")
            raise
    
    def can_rollback(self) -> bool:
        return True

class Migration_008_AddRemoteDomainSetting(Migration):
    """Add remote_domain setting to system_settings table"""
    
    def __init__(self):
        super().__init__("008", "Add remote_domain setting for client remote connections")
    
    def up(self):
        """Add remote_domain setting"""
        try:
            # Check if setting already exists
            from app.models import SystemSettings
            existing = SystemSettings.query.filter_by(key='remote_domain').first()
            
            if existing:
                logger.info("remote_domain setting already exists, skipping creation")
                return True
            
            # Add the setting
            setting = SystemSettings(
                key='remote_domain',
                value='townshipscombatleague.com',
                value_type='string',
                description='Domain for remote client connections'
            )
            db.session.add(setting)
            db.session.commit()
            
            logger.info("Added remote_domain setting successfully")
            return True
            
        except Exception as e:
            logger.error(f"Migration 008 failed: {str(e)}")
            raise
    
    def down(self):
        """Remove remote_domain setting"""
        try:
            from app.models import SystemSettings
            setting = SystemSettings.query.filter_by(key='remote_domain').first()
            if setting:
                db.session.delete(setting)
                db.session.commit()
            logger.info("Removed remote_domain setting")
            return True
        except Exception as e:
            logger.error(f"Rollback of migration 008 failed: {str(e)}")
            raise
    
    def can_rollback(self) -> bool:
        return True

class Migration_009_CreateClientActivityTable(Migration):
    """Create client activity table for tracking online clients"""
    
    def __init__(self):
        super().__init__("009", "Create client activity table for tracking online clients")
    
    def up(self):
        """Create client_activity table"""
        try:
            # Check if table already exists
            inspector = inspect(db.engine)
            if 'client_activity' in inspector.get_table_names():
                logger.info("client_activity table already exists, skipping creation")
                return True
            
            # Create the table using raw SQL to ensure compatibility
            create_table_sql = '''
                CREATE TABLE client_activity (
                    id INT AUTO_INCREMENT PRIMARY KEY,
                    api_token_id INT NOT NULL,
                    rustdesk_id VARCHAR(255) NOT NULL,
                    last_seen DATETIME DEFAULT CURRENT_TIMESTAMP,
                    ip_address VARCHAR(45),
                    user_agent TEXT,
                    INDEX idx_client_activity_api_token_id (api_token_id),
                    INDEX idx_client_activity_rustdesk_id (rustdesk_id),
                    INDEX idx_client_activity_last_seen (last_seen),
                    FOREIGN KEY (api_token_id) REFERENCES api_tokens(id) ON DELETE CASCADE
                ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;
            '''
            
            with db.engine.connect() as conn:
                conn.execute(text(create_table_sql))
                conn.commit()
            
            logger.info("Created client_activity table successfully")
            return True
            
        except Exception as e:
            logger.error(f"Migration 009 failed: {str(e)}")
            raise
    
    def down(self):
        """Drop client_activity table"""
        try:
            with db.engine.connect() as conn:
                conn.execute(text("DROP TABLE IF EXISTS client_activity"))
                conn.commit()
            logger.info("Dropped client_activity table")
            return True
        except Exception as e:
            logger.error(f"Rollback of migration 009 failed: {str(e)}")
            raise
    
    def can_rollback(self) -> bool:
        return True

class Migration_010_CreateReportsTables(Migration):
    """Create reports synchronization tables for client report data"""
    
    def __init__(self):
        super().__init__("010", "Create reports synchronization tables for client report data")
    
    def up(self):
        """Create all reports-related tables"""
        try:
            inspector = inspect(db.engine)
            
            # Create report_syncs table
            if 'report_syncs' not in inspector.get_table_names():
                create_report_syncs_sql = '''
                    CREATE TABLE report_syncs (
                        id INT AUTO_INCREMENT PRIMARY KEY,
                        sync_id VARCHAR(255) NOT NULL UNIQUE,
                        client_id VARCHAR(255) NOT NULL,
                        sync_timestamp DATETIME NOT NULL,
                        date_range VARCHAR(50) NOT NULL,
                        start_date DATETIME NOT NULL,
                        end_date DATETIME NOT NULL,
                        total_payin DECIMAL(15,2) NOT NULL DEFAULT 0.00,
                        total_payout DECIMAL(15,2) NOT NULL DEFAULT 0.00,
                        net_profit DECIMAL(15,2) NOT NULL DEFAULT 0.00,
                        total_bets INT NOT NULL DEFAULT 0,
                        total_matches INT NOT NULL DEFAULT 0,
                        created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
                        updated_at DATETIME DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
                        INDEX idx_report_syncs_sync_id (sync_id),
                        INDEX idx_report_syncs_client_id (client_id),
                        INDEX idx_report_syncs_sync_timestamp (sync_timestamp),
                        INDEX idx_report_syncs_date_range (date_range),
                        INDEX idx_report_syncs_created_at (created_at)
                    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;
                '''
                with db.engine.connect() as conn:
                    conn.execute(text(create_report_syncs_sql))
                    conn.commit()
                logger.info("Created report_syncs table")
            else:
                logger.info("report_syncs table already exists, skipping")
            
            # Create bets table
            if 'bets' not in inspector.get_table_names():
                create_bets_sql = '''
                    CREATE TABLE bets (
                        id INT AUTO_INCREMENT PRIMARY KEY,
                        report_sync_id INT NOT NULL,
                        uuid VARCHAR(36) NOT NULL UNIQUE,
                        fixture_id VARCHAR(255) NOT NULL,
                        bet_datetime DATETIME NOT NULL,
                        paid BOOLEAN NOT NULL DEFAULT FALSE,
                        paid_out BOOLEAN NOT NULL DEFAULT FALSE,
                        total_amount DECIMAL(15,2) NOT NULL DEFAULT 0.00,
                        bet_count INT NOT NULL DEFAULT 0,
                        created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
                        updated_at DATETIME DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
                        INDEX idx_bets_report_sync_id (report_sync_id),
                        INDEX idx_bets_uuid (uuid),
                        INDEX idx_bets_fixture_id (fixture_id),
                        INDEX idx_bets_bet_datetime (bet_datetime),
                        INDEX idx_bets_paid (paid),
                        INDEX idx_bets_paid_out (paid_out),
                        FOREIGN KEY (report_sync_id) REFERENCES report_syncs(id) ON DELETE CASCADE
                    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;
                '''
                with db.engine.connect() as conn:
                    conn.execute(text(create_bets_sql))
                    conn.commit()
                logger.info("Created bets table")
            else:
                logger.info("bets table already exists, skipping")
            
            # Create bet_details table
            if 'bet_details' not in inspector.get_table_names():
                create_bet_details_sql = '''
                    CREATE TABLE bet_details (
                        id INT AUTO_INCREMENT PRIMARY KEY,
                        bet_id INT NOT NULL,
                        match_id INT NOT NULL,
                        match_number INT NOT NULL,
                        outcome VARCHAR(50) NOT NULL,
                        amount DECIMAL(15,2) NOT NULL DEFAULT 0.00,
                        win_amount DECIMAL(15,2) NOT NULL DEFAULT 0.00,
                        result VARCHAR(50) NOT NULL,
                        created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
                        updated_at DATETIME DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
                        INDEX idx_bet_details_bet_id (bet_id),
                        INDEX idx_bet_details_match_id (match_id),
                        INDEX idx_bet_details_outcome (outcome),
                        INDEX idx_bet_details_result (result),
                        FOREIGN KEY (bet_id) REFERENCES bets(id) ON DELETE CASCADE
                    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;
                '''
                with db.engine.connect() as conn:
                    conn.execute(text(create_bet_details_sql))
                    conn.commit()
                logger.info("Created bet_details table")
            else:
                logger.info("bet_details table already exists, skipping")
            
            # Create extraction_stats table
            if 'extraction_stats' not in inspector.get_table_names():
                create_extraction_stats_sql = '''
                    CREATE TABLE extraction_stats (
                        id INT AUTO_INCREMENT PRIMARY KEY,
                        report_sync_id INT NOT NULL,
                        match_id INT NOT NULL,
                        fixture_id VARCHAR(255) NOT NULL,
                        match_datetime DATETIME NOT NULL,
                        total_bets INT NOT NULL DEFAULT 0,
                        total_amount_collected DECIMAL(15,2) NOT NULL DEFAULT 0.00,
                        total_redistributed DECIMAL(15,2) NOT NULL DEFAULT 0.00,
                        actual_result VARCHAR(50) NOT NULL,
                        extraction_result VARCHAR(50) NOT NULL,
                        cap_applied BOOLEAN NOT NULL DEFAULT FALSE,
                        cap_percentage DECIMAL(5,2),
                        under_bets INT NOT NULL DEFAULT 0,
                        under_amount DECIMAL(15,2) NOT NULL DEFAULT 0.00,
                        over_bets INT NOT NULL DEFAULT 0,
                        over_amount DECIMAL(15,2) NOT NULL DEFAULT 0.00,
                        result_breakdown JSON,
                        created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
                        updated_at DATETIME DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
                        INDEX idx_extraction_stats_report_sync_id (report_sync_id),
                        INDEX idx_extraction_stats_match_id (match_id),
                        INDEX idx_extraction_stats_fixture_id (fixture_id),
                        INDEX idx_extraction_stats_match_datetime (match_datetime),
                        INDEX idx_extraction_stats_actual_result (actual_result),
                        FOREIGN KEY (report_sync_id) REFERENCES report_syncs(id) ON DELETE CASCADE
                    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;
                '''
                with db.engine.connect() as conn:
                    conn.execute(text(create_extraction_stats_sql))
                    conn.commit()
                logger.info("Created extraction_stats table")
            else:
                logger.info("extraction_stats table already exists, skipping")
            
            # Create report_sync_logs table
            if 'report_sync_logs' not in inspector.get_table_names():
                create_report_sync_logs_sql = '''
                    CREATE TABLE report_sync_logs (
                        id INT AUTO_INCREMENT PRIMARY KEY,
                        sync_id VARCHAR(255) NOT NULL,
                        client_id VARCHAR(255) NOT NULL,
                        sync_timestamp DATETIME NOT NULL,
                        operation_type ENUM('new_sync', 'duplicate_sync', 'update_stats', 'error') NOT NULL,
                        status ENUM('success', 'failed') NOT NULL,
                        bets_processed INT NOT NULL DEFAULT 0,
                        bets_new INT NOT NULL DEFAULT 0,
                        bets_duplicate INT NOT NULL DEFAULT 0,
                        stats_processed INT NOT NULL DEFAULT 0,
                        stats_new INT NOT NULL DEFAULT 0,
                        stats_updated INT NOT NULL DEFAULT 0,
                        total_payin DECIMAL(15,2) NOT NULL DEFAULT 0.00,
                        total_payout DECIMAL(15,2) NOT NULL DEFAULT 0.00,
                        net_profit DECIMAL(15,2) NOT NULL DEFAULT 0.00,
                        total_bets INT NOT NULL DEFAULT 0,
                        total_matches INT NOT NULL DEFAULT 0,
                        error_message TEXT,
                        error_details JSON,
                        ip_address VARCHAR(45),
                        user_agent TEXT,
                        request_size INT,
                        processing_time_ms INT,
                        created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
                        INDEX idx_report_sync_logs_sync_id (sync_id),
                        INDEX idx_report_sync_logs_client_id (client_id),
                        INDEX idx_report_sync_logs_sync_timestamp (sync_timestamp),
                        INDEX idx_report_sync_logs_operation_type (operation_type),
                        INDEX idx_report_sync_logs_status (status),
                        INDEX idx_report_sync_logs_created_at (created_at)
                    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;
                '''
                with db.engine.connect() as conn:
                    conn.execute(text(create_report_sync_logs_sql))
                    conn.commit()
                logger.info("Created report_sync_logs table")
            else:
                logger.info("report_sync_logs table already exists, skipping")
            
            logger.info("Migration 010: All reports tables created successfully")
            return True
            
        except Exception as e:
            logger.error(f"Migration 010 failed: {str(e)}")
            raise
    
    def down(self):
        """Drop all reports-related tables"""
        try:
            with db.engine.connect() as conn:
                # Drop in reverse order of creation (due to foreign keys)
                conn.execute(text("DROP TABLE IF EXISTS report_sync_logs"))
                conn.execute(text("DROP TABLE IF EXISTS extraction_stats"))
                conn.execute(text("DROP TABLE IF EXISTS bet_details"))
                conn.execute(text("DROP TABLE IF EXISTS bets"))
                conn.execute(text("DROP TABLE IF EXISTS report_syncs"))
                conn.commit()
            logger.info("Dropped all reports tables")
            return True
        except Exception as e:
            logger.error(f"Rollback of migration 010 failed: {str(e)}")
            raise
    
    def can_rollback(self) -> bool:
        return True

class Migration_011_AddCapCompensationBalance(Migration):
    """Add cap_compensation_balance column to report_syncs table"""
    
    def __init__(self):
        super().__init__("011", "Add cap_compensation_balance column to report_syncs table")
    
    def up(self):
        """Add cap_compensation_balance column"""
        try:
            # Check if column already exists
            inspector = inspect(db.engine)
            
            # Check if report_syncs table exists
            if 'report_syncs' not in inspector.get_table_names():
                logger.info("report_syncs table does not exist yet, skipping migration")
                return True
            
            columns = [col['name'] for col in inspector.get_columns('report_syncs')]
            
            if 'cap_compensation_balance' in columns:
                logger.info("cap_compensation_balance column already exists, skipping creation")
                return True
            
            # Add column
            alter_table_sql = '''
                ALTER TABLE report_syncs
                ADD COLUMN cap_compensation_balance DECIMAL(15,2) NOT NULL DEFAULT 0.00
            '''
            
            with db.engine.connect() as conn:
                conn.execute(text(alter_table_sql))
                conn.commit()
            
            logger.info("Added cap_compensation_balance column successfully")
            return True
            
        except Exception as e:
            logger.error(f"Migration 011 failed: {str(e)}")
            raise
    
    def down(self):
        """Drop cap_compensation_balance column"""
        try:
            with db.engine.connect() as conn:
                conn.execute(text("ALTER TABLE report_syncs DROP COLUMN IF EXISTS cap_compensation_balance"))
                conn.commit()
            logger.info("Dropped cap_compensation_balance column")
            return True
        except Exception as e:
            logger.error(f"Rollback of migration 011 failed: {str(e)}")
            raise
    
    def can_rollback(self) -> bool:
        return True

class MigrationManager:
    """Manages database migrations and versioning"""
    
    def __init__(self):
        self.migrations = [
            Migration_002_AddDatabaseVersionTable(),
            Migration_001_RemoveFixtureIdUnique(),
            Migration_003_CreateAPITokensTable(),
            Migration_004_CreateSystemSettingsTable(),
            Migration_005_AddFixtureActiveTime(),
            Migration_006_AddStatusColumn(),
            Migration_007_AddDoneToStatusEnum(),
            Migration_008_AddRemoteDomainSetting(),
            Migration_009_CreateClientActivityTable(),
            Migration_010_CreateReportsTables(),
            Migration_011_AddCapCompensationBalance(),
        ]
    
    def ensure_version_table(self):
        """Ensure the database version table exists"""
        try:
            # Create all tables (this will create DatabaseVersion if it doesn't exist)
            db.create_all()
            logger.info("Database version table ensured")
        except Exception as e:
            logger.error(f"Failed to create version table: {str(e)}")
            raise
    
    def get_current_version(self) -> str:
        """Get the current database version"""
        try:
            latest_version = DatabaseVersion.query.order_by(DatabaseVersion.applied_at.desc()).first()
            return latest_version.version if latest_version else "000"
        except Exception as e:
            logger.warning(f"Could not get current version: {str(e)}")
            return "000"
    
    def get_applied_versions(self) -> List[str]:
        """Get list of applied migration versions"""
        try:
            versions = DatabaseVersion.query.order_by(DatabaseVersion.applied_at.asc()).all()
            return [v.version for v in versions]
        except Exception as e:
            logger.warning(f"Could not get applied versions: {str(e)}")
            return []
    
    def is_migration_applied(self, version: str) -> bool:
        """Check if a migration version has been applied"""
        try:
            return DatabaseVersion.query.filter_by(version=version).first() is not None
        except Exception as e:
            logger.warning(f"Could not check migration status for {version}: {str(e)}")
            return False
    
    def apply_migration(self, migration: Migration) -> bool:
        """Apply a single migration"""
        try:
            if self.is_migration_applied(migration.version):
                logger.info(f"Migration {migration.version} already applied, skipping")
                return True
            
            logger.info(f"Applying migration {migration.version}: {migration.description}")
            
            # Apply the migration
            migration.up()
            
            # Record the migration
            version_record = DatabaseVersion(
                version=migration.version,
                description=migration.description
            )
            db.session.add(version_record)
            db.session.commit()
            
            logger.info(f"Migration {migration.version} applied successfully")
            return True
            
        except Exception as e:
            db.session.rollback()
            logger.error(f"Migration {migration.version} failed: {str(e)}")
            raise
    
    def run_migrations(self) -> Dict[str, Any]:
        """Run all pending migrations"""
        try:
            # Ensure version table exists
            self.ensure_version_table()
            
            applied_versions = self.get_applied_versions()
            pending_migrations = []
            failed_migrations = []
            
            # Find pending migrations
            for migration in self.migrations:
                if migration.version not in applied_versions:
                    pending_migrations.append(migration)
            
            if not pending_migrations:
                logger.info("No pending migrations")
                return {
                    'status': 'success',
                    'message': 'Database is up to date',
                    'applied_count': 0,
                    'failed_count': 0
                }
            
            # Apply pending migrations
            applied_count = 0
            for migration in pending_migrations:
                try:
                    if self.apply_migration(migration):
                        applied_count += 1
                    else:
                        failed_migrations.append(migration.version)
                except Exception as e:
                    failed_migrations.append(migration.version)
                    logger.error(f"Failed to apply migration {migration.version}: {str(e)}")
            
            # Return results
            if failed_migrations:
                return {
                    'status': 'partial',
                    'message': f'Applied {applied_count} migrations, {len(failed_migrations)} failed',
                    'applied_count': applied_count,
                    'failed_count': len(failed_migrations),
                    'failed_migrations': failed_migrations
                }
            else:
                return {
                    'status': 'success',
                    'message': f'Successfully applied {applied_count} migrations',
                    'applied_count': applied_count,
                    'failed_count': 0
                }
                
        except Exception as e:
            logger.error(f"Migration system failed: {str(e)}")
            return {
                'status': 'error',
                'message': f'Migration system failed: {str(e)}',
                'applied_count': 0,
                'failed_count': 0
            }
    
    def get_migration_status(self) -> Dict[str, Any]:
        """Get current migration status"""
        try:
            self.ensure_version_table()
            
            current_version = self.get_current_version()
            applied_versions = self.get_applied_versions()
            
            all_versions = [m.version for m in self.migrations]
            pending_versions = [v for v in all_versions if v not in applied_versions]
            
            return {
                'current_version': current_version,
                'applied_versions': applied_versions,
                'pending_versions': pending_versions,
                'total_migrations': len(self.migrations),
                'applied_count': len(applied_versions),
                'pending_count': len(pending_versions)
            }
            
        except Exception as e:
            logger.error(f"Could not get migration status: {str(e)}")
            return {
                'error': str(e),
                'current_version': 'unknown',
                'applied_versions': [],
                'pending_versions': [],
                'total_migrations': 0,
                'applied_count': 0,
                'pending_count': 0
            }

# Global migration manager instance
migration_manager = MigrationManager()

def get_migration_manager():
    """Get migration manager instance"""
    return migration_manager

def run_migrations():
    """Run all pending migrations"""
    return migration_manager.run_migrations()

def get_migration_status():
    """Get migration status"""
    return migration_manager.get_migration_status()