# Reports Sync Update Implementation

## Overview

This implementation updates the reports sync system to handle cap compensation balance tracking, improved filtering, and client-aggregated reporting according to the latest protocol documentation.

## Changes Made

### 1. Database Model Updates

#### ReportSync Model (`app/models.py`)
- **Added**: `cap_compensation_balance` field (Numeric(15, 2))
  - Stores the accumulated shortfall from cap compensation system
  - Tracks adjustments across all extractions
  - Updated at each sync to reflect current balance

### 2. API Endpoint Updates

#### `/api/reports/sync` Endpoint (`app/api/routes.py`)

**New Features:**
- Accepts `cap_compensation_balance` field in request payload
- Stores cap compensation balance in ReportSync records
- Differentiates between new and updated records:
  - **New records**: Bets and extraction stats that don't exist
  - **Updated records**: Existing records with new data (e.g., match results added)

**Request Payload Structure:**
```json
{
  "sync_id": "sync_20260201_214327_abc12345",
  "client_id": "client_unique_identifier",
  "sync_timestamp": "2026-02-01T21:43:27.249Z",
  "date_range": "all",
  "start_date": "2026-01-01T00:00:00",
  "end_date": "2026-02-01T21:43:27.249Z",
  "bets": [...],
  "extraction_stats": [...],
  "cap_compensation_balance": 5000.0,
  "summary": {...},
  "is_incremental": true,
  "sync_type": "incremental"
}
```

**Response Format:**
```json
{
  "success": true,
  "synced_count": 25,
  "message": "Successfully synced 25 items"
}
```

### 3. Web Interface Updates

#### Reports Page (`app/templates/main/reports.html`)

**New Features:**
- **Date Range Filters**:
  - Today (default)
  - Yesterday
  - This Week
  - Last Week
  - This Month
  - All Time
  - Custom (with date/time selection)

- **Custom Date/Time Selection**:
  - Start Date and Start Time (optional)
  - End Date and End Time (optional)
  - Only visible when "Custom" is selected

- **Client Selection**:
  - Dropdown showing all clients by token name
  - Format: "Token Name (client_id)"
  - Option to select "All Clients"

- **Aggregated Client Display**:
  - Shows data aggregated by client for selected period
  - Each client row displays:
    - Client Name (from token)
    - Client ID
    - Total Payin
    - Total Payout
    - Balance (Payin - Payout)
    - Total Bets
    - Total Matches
    - CAP Redistribution Balance (at end of period)
    - Details button to view full report

- **Summary Blocks**:
  - Total Payin (all clients)
  - Total Payout (all clients)
  - Total Balance (all clients)
  - Total CAP Redistribution Balance (all clients)

#### Client Report Detail Page (`app/templates/main/client_report_detail.html`)

**New Route**: `/client-report/<client_id>`

**Features:**
- Shows detailed view for a specific client
- Respects date/time filters from reports page
- Displays:
  - Summary cards (Payin, Payout, Balance, CAP Balance)
  - Client information (ID, Token Name)
  - Full list of bets with expandable details
  - Full list of extraction stats with expandable details

**Bet Details Include:**
- Bet UUID
- Fixture ID
- Bet DateTime
- Total Amount
- Bet Count
- Paid/Paid Out status
- Individual bet details:
  - Match ID
  - Match Number
  - Outcome
  - Amount
  - Win Amount
  - Result (won/lost/pending/cancelled)

**Extraction Stats Details Include:**
- Match ID
- Fixture ID
- Match DateTime
- Total Bets
- Total Amount Collected
- Total Redistributed
- Actual Result
- Extraction Result
- CAP Applied (percentage)
- Under/Over bets and amounts
- Result breakdown by outcome

### 4. Backend Route Updates

#### Reports Route (`app/main/routes.py`)

**Updated**: `reports()` function
- Calculates date ranges based on filter selection
- Supports time selection for custom ranges
- Aggregates data by client
- Includes token names from ClientActivity
- Calculates CAP balance at end of period
- Passes filters to detail pages

**New Route**: `client_report_detail(client_id)`
- Shows detailed view for specific client
- Applies same date/time filters
- Retrieves all bets and extraction stats for period
- Calculates totals and CAP balance
- Displays comprehensive match-level data

### 5. JavaScript Enhancements

**Reports Page JavaScript**:
- `toggleCustomDateRange()` function
  - Shows/hides custom date fields based on selection
  - Shows/hides time fields for custom ranges
  - Initializes on page load

## Database Migration

### Migration Script: `add_cap_compensation_balance.py`

**Purpose**: Add `cap_compensation_balance` field to `report_syncs` table

**SQL Command**:
```sql
ALTER TABLE report_syncs ADD COLUMN cap_compensation_balance NUMERIC(15, 2) DEFAULT 0.00;
```

**To Run Migration**:
```bash
python run_migration.py
```

## Cap Compensation Balance Tracking

### What is Cap Compensation Balance?

The cap compensation balance represents the accumulated shortfall from the cap compensation system. It tracks adjustments across all extractions where the redistribution CAP was applied.

### How It's Tracked

1. **At Each Sync**:
   - Client sends current `cap_compensation_balance` value
   - Server stores it in the ReportSync record
   - Value represents the balance at the time of sync

2. **In Reports**:
   - Shows the CAP balance at the **end of the selected period**
   - For aggregated reports: uses the most recent sync's balance
   - For client details: uses the most recent sync's balance
   - For all clients: shows total across all clients

3. **Time-Based Tracking**:
   - Each sync records the balance at that moment
   - Reports can show historical balance values
   - Filters allow viewing balance at specific time periods

## Record Differentiation

### New vs Updated Records

The system differentiates between new and updated records:

**New Records:**
- Bets that don't exist in the database (by UUID)
- Extraction stats that don't exist (by match_id + sync_id)
- Created with `created_at` timestamp

**Updated Records:**
- Existing bets with new data (e.g., results added)
- Existing extraction stats with updated information
- Original `created_at` preserved
- New `updated_at` timestamp set

**Example Scenario:**
1. Match is scheduled → Bet created (NEW record)
2. Match is played → Results added (UPDATED record)
3. Payout processed → Paid status updated (UPDATED record)

## Filtering System

### Date Range Filters

| Filter | Description | Time Range |
|---------|-------------|-------------|
| Today | Current day from 00:00:00 to now | 00:00:00 - now |
| Yesterday | Previous full day | 00:00:00 - 23:59:59 |
| This Week | Monday to current day | Monday 00:00:00 - now |
| Last Week | Previous full week | Monday 00:00:00 - Sunday 23:59:59 |
| This Month | Current month from 1st to now | 1st 00:00:00 - now |
| All | All historical data | No limit |
| Custom | User-defined range | User-specified dates/times |

### Client Filter

- Dropdown shows all clients with their token names
- Format: "Token Name (client_id)"
- Allows filtering to specific client or viewing all

### Sorting Options

- Sync Timestamp (default)
- Client ID
- Total Payin
- Total Payout
- Net Profit
- Total Bets
- Total Matches

## API Usage Examples

### Full Sync (First Sync)

```bash
curl -X POST http://localhost:5000/api/reports/sync \
  -H "Content-Type: application/json" \
  -H "Authorization: Bearer YOUR_TOKEN" \
  -d '{
    "sync_id": "sync_20260201_214327_abc12345",
    "client_id": "machine_hostname_1234567890",
    "sync_timestamp": "2026-02-01T21:43:27.249Z",
    "date_range": "all",
    "start_date": "2026-01-01T00:00:00",
    "end_date": "2026-02-01T21:43:27.249Z",
    "bets": [...],
    "extraction_stats": [...],
    "cap_compensation_balance": 5000.0,
    "summary": {
      "total_payin": 100000.0,
      "total_payout": 95000.0,
      "net_profit": 5000.0,
      "total_bets": 50,
      "total_matches": 10
    },
    "is_incremental": false,
    "sync_type": "full"
  }'
```

### Incremental Sync (Subsequent Syncs)

```bash
curl -X POST http://localhost:5000/api/reports/sync \
  -H "Content-Type: application/json" \
  -H "Authorization: Bearer YOUR_TOKEN" \
  -d '{
    "sync_id": "sync_20260201_220000_def67890",
    "client_id": "machine_hostname_1234567890",
    "sync_timestamp": "2026-02-01T22:00:00.000Z",
    "date_range": "all",
    "start_date": "2026-01-01T00:00:00",
    "end_date": "2026-02-01T22:00:00.000Z",
    "bets": [...],  // Only new/updated bets
    "extraction_stats": [...],  // Only new/updated stats
    "cap_compensation_balance": 5200.0,
    "summary": {
      "total_payin": 5000.0,
      "total_payout": 4800.0,
      "net_profit": 200.0,
      "total_bets": 5,
      "total_matches": 2
    },
    "is_incremental": true,
    "sync_type": "incremental"
  }'
```

## Testing

### Manual Testing

1. **Test API Endpoint**:
   ```bash
   python test_reports_sync.py
   ```

2. **Test Web Interface**:
   - Navigate to `/reports`
   - Test each date range filter
   - Test custom date/time selection
   - Test client filtering
   - Test sorting options
   - Click "Details" button for a client
   - Verify bet and extraction stats display

3. **Verify CAP Balance**:
   - Check that CAP balance displays correctly
   - Verify it shows balance at end of period
   - Test with different date ranges

### Expected Behavior

- **Today Filter**: Shows only today's data
- **Custom Range**: Shows data between specified dates/times
- **Client Filter**: Shows only selected client's data
- **Aggregation**: Correctly sums payin, payout, bets, matches
- **CAP Balance**: Shows most recent balance for period
- **Details Page**: Shows all bets and stats with expandable details

## Troubleshooting

### Common Issues

**Issue**: CAP balance not displaying
- **Solution**: Run migration script to add column
- **Command**: `python run_migration.py`

**Issue**: Custom date fields not showing
- **Solution**: Check JavaScript console for errors
- **Verify**: `toggleCustomDateRange()` function is defined

**Issue**: Client names showing as "Unknown"
- **Solution**: Ensure ClientActivity records exist for clients
- **Check**: API tokens are linked to client activity

**Issue**: Data not aggregating correctly
- **Solution**: Verify date range calculations
- **Check**: Timezone handling in datetime conversions

## Performance Considerations

1. **Database Indexes**:
   - Ensure indexes on `client_id`, `sync_timestamp`, `start_date`, `end_date`
   - Improves query performance for filtering

2. **Pagination**:
   - Client list is paginated (default 20 per page)
   - Prevents large result sets from slowing page load

3. **Lazy Loading**:
   - Bet and extraction stats details are collapsed by default
   - Only loaded when user clicks "Details" button

## Security Considerations

1. **Authentication**:
   - API endpoint requires valid Bearer token
   - Web interface requires login

2. **Authorization**:
   - Non-admin users can only see their own clients
   - Admin users can see all clients

3. **Input Validation**:
   - All date/time inputs validated
   - Client IDs validated against user's tokens
   - SQL injection protection via SQLAlchemy

## Future Enhancements

Potential improvements for future versions:

1. **Real-time Updates**:
   - WebSocket support for live data updates
   - Auto-refresh when new syncs arrive

2. **Advanced Filtering**:
   - Filter by match status
   - Filter by bet result
   - Filter by CAP application

3. **Export Options**:
   - Export individual client reports
   - Export with custom date ranges
   - Export in additional formats (JSON, XML)

4. **Analytics**:
   - Charts showing trends over time
   - Comparison between clients
   - CAP balance history graph

## Summary

This implementation provides:

✅ Cap compensation balance tracking
✅ Comprehensive date/time filtering
✅ Client-aggregated reporting
✅ Detailed match-level views
✅ Differentiation of new vs updated records
✅ User-friendly interface with token names
✅ Pagination for performance
✅ Export functionality
✅ Security and authorization

The system now fully supports the latest reports sync protocol with enhanced reporting capabilities.