
# Reports Sync Implementation - Final Documentation

## Overview
This document describes the complete implementation of the reports sync system according to the REPORTS_SYNC_PROTOCOL_DOCUMENTATION.md specification.

## Changes Made

### 1. Database Models (`app/models.py`)

#### Added MatchReport Model
A new model to store comprehensive match-level data for detailed reporting:

```python
class MatchReport(db.Model):
    """Comprehensive match-level reports with detailed statistics"""
    __tablename__ = 'match_reports'
    
    id = db.Column(db.Integer, primary_key=True)
    sync_id = db.Column(db.Integer, db.ForeignKey('report_syncs.id'), nullable=False, index=True)
    client_id = db.Column(db.String(255), nullable=False, index=True)
    client_token_name = db.Column(db.String(255), nullable=False)
    match_id = db.Column(db.Integer, nullable=False, index=True)
    match_number = db.Column(db.Integer, nullable=False)
    fixture_id = db.Column(db.String(255), nullable=False, index=True)
    match_datetime = db.Column(db.DateTime, nullable=False, index=True)
    total_bets = db.Column(db.Integer, nullable=False)
    winning_bets = db.Column(db.Integer, default=0)
    losing_bets = db.Column(db.Integer, default=0)
    pending_bets = db.Column(db.Integer, default=0)
    total_payin = db.Column(db.Numeric(15, 2), nullable=False)
    total_payout = db.Column(db.Numeric(15, 2), nullable=False)
    balance = db.Column(db.Numeric(15, 2), nullable=False)
    actual_result = db.Column(db.String(50), nullable=False)
    extraction_result = db.Column(db.String(50), nullable=False)
    cap_applied = db.Column(db.Boolean, default=False)
    cap_percentage = db.Column(db.Numeric(5, 2))
    cap_compensation_balance = db.Column(db.Numeric(15, 2), nullable=False)
    under_bets = db.Column(db.Integer, default=0)
    under_amount = db.Column(db.Numeric(15, 2), default=0.00)
    over_bets = db.Column(db.Integer, default=0)
    over_amount = db.Column(db.Numeric(15, 2), default=0.00)
    result_breakdown = db.Column(db.JSON)
    created_at = db.Column(db.DateTime, default=datetime.utcnow)
```

#### Updated ReportSync Model
Added `cap_compensation_balance` field to track cap compensation over time:

```python
cap_compensation_balance = db.Column(db.Numeric(15, 2), default=0.00)
```

### 2. API Endpoint (`app/api/routes.py`)

#### Updated `/api/reports/sync` Endpoint

**Key Changes:**

1. **Cap Compensation Balance Handling**
   - Extracts `cap_compensation_balance` from request payload
   - Stores it in ReportSync record
   - Tracks it over time for historical analysis

2. **MatchReport Creation**
   - Creates comprehensive match-level records for each extraction stat
   - Calculates winning/losing/pending bets from bet details
   - Stores complete match information including:
     - Match ID and number
     - Fixture ID
     - Match datetime
     - Total bets and breakdown (winning/losing/pending)
     - Payin, payout, and balance
     - Actual and extraction results
     - CAP application details
     - CAP compensation balance at match time
     - Under/over bet breakdowns
     - Result breakdown by outcome

3. **Record Differentiation**
   - New records: Created when UUID doesn't exist
   - Updated records: Updated when UUID exists but data changed
   - Tracked via ReportSyncLog with operation types:
     - `new_sync`: First sync from client
     - `duplicate_sync`: Sync already processed
     - `update_stats`: Existing records updated

### 3. Web Interface (`app/main/routes.py`)

#### Reports Page (`/reports`)

**Features:**

1. **Date Range Filters**
   - Today (default)
   - Yesterday
   - This Week
   - Last Week
   - This Month
   - All
   - Custom (with date and time selection)

2. **Client Selection**
   - Dropdown with all clients by token name
   - Shows client ID for identification
   - Filters reports by selected client

3. **Aggregated Data Display**
   - Shows summary cards at top:
     - Total Payin
     - Total Payout
     - Total Balance
     - CAP Redistribution Balance
   - Client table with:
     - Client Name (token name)
     - Client ID
     - Number of Matches
     - Number of Bets
     - Payin
     - Payout
     - Balance
     - CAP Redistribution Balance
     - Actions (View Details button)

4. **Sorting Options**
   - Sort by: Client Name, Matches, Bets, Payin, Payout, Balance, CAP Balance
   - Sort direction: Ascending/Descending

#### Client Report Detail Page (`/reports/client/<client_id>`)

**Features:**

1. **Same Summary Cards**
   - Total Payin
   - Total Payout
   - Total Balance
   - CAP Redistribution Balance

2. **Match-Level Table**
   - Match ID
   - Match Number
   - Fixture ID
   - Match DateTime
   - Total Bets
   - Winning Bets
   - Losing Bets
   - Pending Bets
   - Payin
   - Payout
   - Balance
   - Results (Actual/Extraction)
   - CAP Applied (Yes/No)
   - CAP Percentage
   - CAP Compensation Balance

3. **Same Filters**
   - Date range filters
   - Date/time selection for custom range

### 4. Templates (`app/templates/main/`)

#### Updated `reports.html`
- Complete rewrite with client aggregation
- Summary cards at top
- Client table with all required fields
- Filter controls for date range and client selection
- Sorting functionality
- Responsive design

#### Updated `client_report_detail.html`
- Complete rewrite with match-level data
- Summary cards at top
- Match reports table with comprehensive information
- Same filter controls as reports page
- Detailed match information display

### 5. Database Migrations

#### Migration Script: `add_match_reports_table.py`
Creates the `match_reports` table with all required fields and indexes.

## Implementation Details

### Cap Compensation Balance Tracking

The cap compensation balance is tracked at two levels:

1. **Sync Level** (`ReportSync.cap_compensation_balance`)
   - Stores the cap compensation balance at the time of sync
   - Represents the accumulated shortfall across all extractions
   - Used for historical tracking and reconciliation

2. **Match Level** (`MatchReport.cap_compensation_balance`)
   - Stores the cap compensation balance at the time of each match
   - Allows tracking how the balance changes over time
   - Used for detailed match-level reporting

### Record Differentiation Logic

The system differentiates between new and updated records using:

1. **Sync ID Tracking**
   - Each sync has a unique `sync_id`
   - Prevents duplicate processing of the same sync

2. **UUID-Based Deduplication**
   - Bets are tracked by UUID
   - Existing bets are updated instead of creating duplicates
   - New bets are created when UUID doesn't exist

3. **Match ID Tracking**
   - Match reports are tracked by match_id and client_id
   - Existing match reports are updated with new data
   - New match reports are created for new matches

4. **Operation Logging**
   - All operations are logged in `ReportSyncLog`
   - Operation types: `new_sync`, `duplicate_sync`, `update_stats`
   - Provides audit trail for all sync operations

### Date Range Filtering

The system supports multiple date range filters:

1. **Today**
   - Start: Today 00:00:00
   - End: Today 23:59:59

2. **Yesterday**
   - Start: Yesterday 00:00:00
   - End: Yesterday 23:59:59

3. **This Week**
   - Start: Monday 00:00:00 of current week
   - End: Sunday 23:59:59 of current week

4. **Last Week**
   - Start: Monday 00:00:00 of previous week
   - End: Sunday 23:59:59 of previous week

5. **This Month**
   - Start: 1st day of month 00:00:00
   - End: Last day of month 23:59:59

6. **All**
   - No date filtering
   - Shows all records

7. **Custom**
   - User selects start and end date
   - Optionally selects start and end time
   - Default time: 00:00:00 for start, 23:59:59 for end

### Client Aggregation

The reports page aggregates data by client:

1. **Grouping**
   - Groups all match reports by client_id
   - Calculates totals for each client

2. **Aggregations**
   - Number of matches: Count of match reports
   - Number of bets: Sum of total_bets
   - Payin: Sum of total_payin
   - Payout: Sum of total_payout
   - Balance: Payin - Payout
   - CAP Redistribution Balance: Latest cap_compensation_balance

3. **Display**
   - Shows client token name for easy identification
   - Shows client ID for reference
   - Provides button to view detailed match reports

## Testing

### Manual Testing Steps

1. **Test Sync Endpoint**
   ```bash
   curl -X POST http://localhost:5000/api/reports/sync \
     -H "Content-Type: application/json" \
     -H "Authorization: Bearer YOUR_TOKEN" \
     -d @test_sync_payload.json
   ```

2. **Test Reports Page**
   - Navigate to `/reports`
   - Verify summary cards display correctly
   - Test each date range filter
   - Test custom date range with time selection
   - Test client selection dropdown
   - Verify client table shows correct data
   - Test sorting options

3. **Test Client Detail Page**
   - Click "View Details" for a client
   - Verify summary cards display correctly
   - Verify match reports table shows all matches
   - Test filters on detail page

4. **Verify Database**
   - Check `report_syncs` table for cap_compensation_balance
   - Check `match_reports` table for comprehensive match data
   - Verify indexes are created correctly

## Deployment

### Migration Steps

1. **Run Migration**
   ```bash
   python run_migration.py
   ```

2. **Verify Migration**
   ```bash
   python -c "from app import create_app, db; app = create_app(); app.app_context().push(); print(db.engine.table_names())"
   ```

3. **Restart Application**
   ```bash
   # Stop current instance
   # Start new instance
   ```

### Configuration

No additional configuration is required. The system uses existing configuration from `config.py`.

## Troubleshooting

### Common Issues

1. **Migration Fails**
   - Check database connection
   - Verify user has CREATE TABLE permissions
   - Check for existing table conflicts

2. **Sync Fails**
   - Verify API token is valid
   - Check request payload format
   - Review server logs for errors

3. **Reports Page Shows No Data**
   - Verify sync has completed successfully
   - Check date range filters
   - Verify client selection

4. **CAP Balance Not Showing**
   - Verify sync includes cap_compensation_balance field
   - Check database for cap_compensation_balance values
   - Review sync logs for errors

## Future Enhancements

Potential improvements for future versions:

1. **Export Functionality**
   - Export reports to CSV/Excel
   - Export match details

2. **Advanced Filtering**
   - Filter by match result
   - Filter by CAP application
   - Filter by bet status

3. **Charts and Visualizations**
   - Payin/Payout trends over time
   - CAP balance history
   - Client performance comparison

4. **Real-time Updates**
   - WebSocket support for live updates
   - Auto-refresh reports page

5. **Email Notifications**
   - Daily/weekly report summaries
   - Alert on unusual activity

## Conclusion

The implementation successfully addresses all requirements from the REPORTS_SYNC_PROTOCOL_DOCUMENTATION.md:

✅ Cap compensation balance tracking at sync and match levels
✅ Differentiation between new and updated records
✅ Comprehensive date range filtering with time selection
✅ Client selection dropdown with token names
✅ Aggregated reports by client with all required metrics
✅ Detailed match-level view with complete information
✅ Historical tracking of all data
✅ Robust database schema with proper indexes
✅ Clean, responsive web interface

The system is production-ready and provides a complete solution for reports synchronization and visualization.