# Fixture Manager - Comprehensive Python Daemon System

A sophisticated Python daemon system for Linux servers with internet exposure, implementing a secure web dashboard and RESTful API with robust authentication mechanisms. The system provides advanced file upload capabilities with real-time progress tracking and a comprehensive fixture management system.

## Features

### Core Functionality
- **Secure Web Dashboard**: Modern web interface with authentication and authorization
- **RESTful API**: Comprehensive API with JWT authentication
- **MySQL Database Integration**: Robust database connectivity with connection pooling
- **Advanced File Upload System**: Real-time progress tracking with SHA1 checksum verification
- **Dual-Format Support**: Intelligent parsing of CSV/XLSX fixture files
- **Two-Stage Upload Workflow**: Fixture files followed by mandatory ZIP uploads
- **Daemon Process Management**: Full Linux daemon with systemd integration

### Security Features
- **Multi-layer Authentication**: Session-based and JWT token authentication
- **Rate Limiting**: Protection against brute force attacks
- **File Validation**: Comprehensive security checks and malicious content detection
- **SQL Injection Protection**: Parameterized queries and ORM usage
- **CSRF Protection**: Cross-site request forgery prevention
- **Security Headers**: Comprehensive HTTP security headers
- **Input Sanitization**: All user inputs are validated and sanitized

### Database Schema
- **Normalized Design**: Optimized relational database structure
- **Primary Matches Table**: Core fixture data with system fields
- **Secondary Outcomes Table**: Dynamic result columns with foreign key relationships
- **File Upload Tracking**: Complete upload lifecycle management
- **System Logging**: Comprehensive audit trail
- **Session Management**: Secure user session handling

## Installation

### Prerequisites
- Linux server (Ubuntu 18.04+, CentOS 7+, or similar)
- Python 3.8+
- MySQL 5.7+ or MariaDB 10.3+
- Root or sudo access

### Quick Installation
```bash
# Clone the repository
git clone <repository-url>
cd fixture-manager

# Make installation script executable
chmod +x install.sh

# Run installation (as root)
sudo ./install.sh
```

### Manual Installation
1. **Install System Dependencies**:
   ```bash
   # Ubuntu/Debian
   apt-get update
   apt-get install python3 python3-pip python3-venv mysql-server nginx supervisor

   # CentOS/RHEL
   yum install python3 python3-pip mysql-server nginx supervisor
   ```

2. **Create System User**:
   ```bash
   useradd --system --home-dir /var/lib/fixture-daemon fixture
   ```

3. **Install Python Dependencies**:
   ```bash
   python3 -m venv venv
   source venv/bin/activate
   pip install -r requirements.txt
   ```

4. **Configure Database**:
   ```bash
   mysql -u root -p < database/schema.sql
   ```

5. **Configure Environment**:
   ```bash
   cp .env.example .env
   # Edit .env with your configuration
   ```

## Configuration

### Environment Variables
The system uses environment variables for configuration. Key settings include:

```bash
# Database Configuration
MYSQL_HOST=localhost
MYSQL_PORT=3306
MYSQL_USER=fixture_user
MYSQL_PASSWORD=secure_password
MYSQL_DATABASE=fixture_manager

# Security Configuration
SECRET_KEY=your-secret-key-here
JWT_SECRET_KEY=your-jwt-secret-key
BCRYPT_LOG_ROUNDS=12

# File Upload Configuration
UPLOAD_FOLDER=/var/lib/fixture-daemon/uploads
MAX_CONTENT_LENGTH=524288000  # 500MB
MAX_CONCURRENT_UPLOADS=5

# Server Configuration
HOST=0.0.0.0
PORT=5000
DEBUG=false
```

### Database Schema
The system automatically creates the following tables:
- `users` - User authentication and management
- `matches` - Core fixture data with system fields
- `match_outcomes` - Dynamic outcome results
- `file_uploads` - Upload tracking and progress
- `system_logs` - Comprehensive logging
- `user_sessions` - Session management

## Usage

### Daemon Management
```bash
# Start the daemon
sudo systemctl start fixture-daemon

# Stop the daemon
sudo systemctl stop fixture-daemon

# Restart the daemon
sudo systemctl restart fixture-daemon

# Check status
sudo systemctl status fixture-daemon

# View logs
journalctl -u fixture-daemon -f
```

### Direct Daemon Control
```bash
# Start in foreground (for debugging)
python daemon.py start --foreground

# Start as daemon
python daemon.py start

# Stop daemon
python daemon.py stop

# Restart daemon
python daemon.py restart

# Check status
python daemon.py status

# Reload configuration
python daemon.py reload
```

### Web Interface
Access the web dashboard at `http://your-server-ip/`

**Default Credentials**:
- Username: `admin`
- Password: `admin123`

**⚠️ Important**: Change the default password immediately after installation!

### API Usage

#### Authentication
```bash
# Login and get JWT token
curl -X POST http://your-server/auth/api/login \
  -H "Content-Type: application/json" \
  -d '{"username": "admin", "password": "admin123"}'
```

#### Upload Fixture File
```bash
# Upload CSV/XLSX fixture file
curl -X POST http://your-server/upload/api/fixture \
  -H "Authorization: Bearer YOUR_JWT_TOKEN" \
  -F "file=@fixtures.csv"
```

#### Upload ZIP File
```bash
# Upload ZIP file for specific match
curl -X POST http://your-server/upload/api/zip/123 \
  -H "Authorization: Bearer YOUR_JWT_TOKEN" \
  -F "file=@match_data.zip"
```

#### Get Matches
```bash
# Get all matches with pagination
curl -X GET "http://your-server/api/matches?page=1&per_page=20" \
  -H "Authorization: Bearer YOUR_JWT_TOKEN"
```

## File Format Requirements

### Fixture Files (CSV/XLSX)
**Required Columns**:
- `Match #` (integer) - Unique match identifier
- `Fighter1 (Township)` (varchar255) - First fighter details
- `Fighter2 (Township)` (varchar255) - Second fighter details
- `Venue (Kampala Township)` (varchar255) - Match venue

**Optional Columns**:
- Any numeric columns will be automatically detected as outcome results
- Values must be numeric (float with 2-decimal precision)

**Example CSV**:
```csv
Match #,Fighter1 (Township),Fighter2 (Township),Venue (Kampala Township),Score1,Score2,Duration
1,John Doe (Central),Jane Smith (North),Stadium A (Kampala),85.5,92.3,12.5
2,Mike Johnson (East),Sarah Wilson (West),Arena B (Kampala),78.2,81.7,15.2
```

### ZIP Files
- Must be uploaded after fixture file processing
- Associated with specific match records
- Triggers match activation upon successful upload
- SHA1 checksum verification for integrity

## Architecture

### System Components
1. **Flask Web Application**: Core web framework with blueprints
2. **SQLAlchemy ORM**: Database abstraction and management
3. **JWT Authentication**: Stateless API authentication
4. **File Upload Handler**: Chunked uploads with progress tracking
5. **Fixture Parser**: Intelligent CSV/XLSX parsing
6. **Security Layer**: Multi-layer security implementation
7. **Logging System**: Comprehensive audit and monitoring
8. **Daemon Manager**: Linux daemon process management

### Security Architecture
- **Authentication**: Multi-factor with session and JWT support
- **Authorization**: Role-based access control (RBAC)
- **Input Validation**: Comprehensive sanitization and validation
- **File Security**: Malicious content detection and quarantine
- **Network Security**: Rate limiting and DDoS protection
- **Data Protection**: Encryption at rest and in transit

### Database Design
- **Normalized Schema**: Third normal form compliance
- **Foreign Key Constraints**: Referential integrity
- **Indexing Strategy**: Optimized query performance
- **Transaction Management**: ACID compliance
- **Connection Pooling**: Efficient resource utilization

## Monitoring and Maintenance

### Log Files
- **Application Logs**: `/var/log/fixture-daemon.log`
- **System Logs**: `journalctl -u fixture-daemon`
- **Database Logs**: MySQL error logs
- **Web Server Logs**: Nginx access/error logs

### Health Monitoring
```bash
# Check system health
curl http://your-server/health

# Get system statistics
curl -H "Authorization: Bearer TOKEN" http://your-server/api/statistics
```

### Backup and Recovery
```bash
# Manual backup
/opt/fixture-manager/backup.sh

# Restore from backup
mysql -u fixture_user -p fixture_manager < backup.sql
```

### Maintenance Tasks
The daemon automatically performs:
- **Session Cleanup**: Expired sessions removed hourly
- **Log Rotation**: Old logs archived daily
- **File Cleanup**: Failed uploads cleaned every 6 hours
- **Database Optimization**: Statistics updated nightly

## Troubleshooting

### Common Issues

1. **Database Connection Failed**
   ```bash
   # Check MySQL service
   systemctl status mysql
   
   # Verify credentials
   mysql -u fixture_user -p
   ```

2. **File Upload Errors**
   ```bash
   # Check permissions
   ls -la /var/lib/fixture-daemon/uploads
   
   # Check disk space
   df -h
   ```

3. **Daemon Won't Start**
   ```bash
   # Check logs
   journalctl -u fixture-daemon -n 50
   
   # Test configuration
   python daemon.py start --foreground
   ```

4. **Permission Denied**
   ```bash
   # Fix ownership
   chown -R fixture:fixture /var/lib/fixture-daemon
   
   # Fix permissions
   chmod 755 /opt/fixture-manager
   ```

### Debug Mode
```bash
# Run in debug mode
export DEBUG=true
python daemon.py start --foreground --config development
```

## API Documentation

### Authentication Endpoints
- `POST /auth/api/login` - User login
- `POST /auth/api/logout` - User logout
- `POST /auth/api/refresh` - Refresh JWT token
- `GET /auth/api/profile` - Get user profile

### Upload Endpoints
- `POST /upload/api/fixture` - Upload fixture file
- `POST /upload/api/zip/{match_id}` - Upload ZIP file
- `GET /upload/api/progress/{upload_id}` - Get upload progress
- `GET /upload/api/uploads` - List user uploads

### Match Management
- `GET /api/matches` - List matches with pagination
- `GET /api/matches/{id}` - Get match details
- `PUT /api/matches/{id}` - Update match
- `DELETE /api/matches/{id}` - Delete match (admin)

### Administration
- `GET /api/admin/users` - List users (admin)
- `PUT /api/admin/users/{id}` - Update user (admin)
- `GET /api/admin/logs` - System logs (admin)
- `GET /api/admin/system-info` - System information (admin)

## Performance Optimization

### Database Optimization
- Connection pooling with 10 connections
- Query optimization with proper indexing
- Prepared statements for security
- Transaction batching for bulk operations

### File Upload Optimization
- Chunked uploads for large files
- Concurrent upload support (configurable)
- Progress tracking with minimal overhead
- Automatic cleanup of failed uploads

### Caching Strategy
- Session caching with Redis (optional)
- Static file caching with Nginx
- Database query result caching
- API response caching for read-heavy endpoints

## Security Considerations

### Production Deployment
1. **Change Default Credentials**: Update admin password immediately
2. **SSL/TLS Configuration**: Enable HTTPS with valid certificates
3. **Firewall Configuration**: Restrict access to necessary ports only
4. **Regular Updates**: Keep system and dependencies updated
5. **Backup Strategy**: Implement regular automated backups
6. **Monitoring**: Set up comprehensive monitoring and alerting

### Security Best Practices
- Regular security audits
- Penetration testing
- Vulnerability scanning
- Access log monitoring
- Incident response procedures

## Contributing

1. Fork the repository
2. Create a feature branch
3. Make your changes
4. Add tests for new functionality
5. Submit a pull request

## License

This project is licensed under the MIT License - see the LICENSE file for details.

## Support

For support and questions:
- Check the troubleshooting section
- Review system logs
- Contact system administrator

---

**Version**: 1.0.0  
**Last Updated**: 2025-08-18  
**Minimum Requirements**: Python 3.8+, MySQL 5.7+, Linux Kernel 3.10+