# Client-Side Reports Sync Implementation Guide

## Overview

This guide provides instructions for implementing client-side reports synchronization with the server, including how to query the server for the last sync information to verify and recover from tracking corruption.

## Server Endpoints

### 1. Sync Reports Endpoint

**Endpoint**: `POST /api/reports/sync`

**Authentication**: Bearer token (API token)

**Request Format**:
```json
{
  "sync_id": "sync_20260201_214327_abc12345",
  "client_id": "client_unique_identifier",
  "sync_timestamp": "2026-02-01T21:43:27.249Z",
  "date_range": "all",
  "start_date": "2026-01-01T00:00:00",
  "end_date": "2026-02-01T21:43:27.249Z",
  "bets": [...],
  "extraction_stats": [...],
  "cap_compensation_balance": 5000.0,
  "summary": {...},
  "is_incremental": true,
  "sync_type": "incremental"
}
```

**Response Format**:
```json
{
  "success": true,
  "synced_count": 25,
  "message": "Report data synchronized successfully",
  "server_timestamp": "2026-02-01T21:43:27.249Z"
}
```

### 2. Get Last Sync Endpoint (NEW)

**Endpoint**: `GET /api/reports/last-sync?client_id=<client_id>`

**Authentication**: Bearer token (API token)

**Query Parameters**:
- `client_id` (required): The unique client identifier

**Response Format**:
```json
{
  "success": true,
  "client_id": "client_unique_identifier",
  "last_sync_id": "sync_20260201_214327_abc12345",
  "last_sync_timestamp": "2026-02-01T21:43:27.249Z",
  "last_sync_type": "incremental",
  "last_date_range": "all",
  "last_start_date": "2026-01-01T00:00:00",
  "last_end_date": "2026-02-01T21:43:27.249Z",
  "total_syncs": 25,
  "last_sync_summary": {
    "total_payin": 100000.0,
    "total_payout": 95000.0,
    "net_profit": 5000.0,
    "total_bets": 50,
    "total_matches": 10,
    "cap_compensation_balance": 5000.0
  },
  "last_sync_log": {
    "operation_type": "new_sync",
    "status": "success",
    "bets_processed": 50,
    "bets_new": 50,
    "bets_duplicate": 0,
    "stats_processed": 10,
    "stats_new": 10,
    "stats_updated": 0,
    "created_at": "2026-02-01T21:43:27.249Z"
  },
  "server_timestamp": "2026-02-01T21:43:27.249Z"
}
```

**Response When No Syncs Exist**:
```json
{
  "success": true,
  "message": "No sync records found for this client",
  "client_id": "client_unique_identifier",
  "last_sync_id": null,
  "last_sync_timestamp": null,
  "last_sync_type": null,
  "total_syncs": 0,
  "server_timestamp": "2026-02-01T21:43:27.249Z"
}
```

## Client-Side Implementation

### Step 1: Initialize Client Tracking

Create a local tracking system to manage sync state:

```python
class ReportsSyncTracking:
    def __init__(self, db_session):
        self.db = db_session
        self.client_id = self.get_client_id()
    
    def get_client_id(self):
        """Get unique client identifier (machine ID or rustdesk_id)"""
        # Implement your client ID generation logic
        return "client_unique_identifier"
    
    def get_last_sync_info(self):
        """Get last sync information from local tracking"""
        sync_record = self.db.query(ReportsSyncTrackingModel)\
            .filter_by(entity_type='sync', entity_id='latest')\
            .first()
        
        if sync_record:
            return {
                'last_synced_at': sync_record.last_synced_at,
                'sync_id': sync_record.sync_id
            }
        return None
    
    def update_last_sync(self, sync_id, timestamp):
        """Update last sync information in local tracking"""
        sync_record = self.db.query(ReportsSyncTrackingModel)\
            .filter_by(entity_type='sync', entity_id='latest')\
            .first()
        
        if sync_record:
            sync_record.last_synced_at = timestamp
            sync_record.sync_id = sync_id
        else:
            sync_record = ReportsSyncTrackingModel(
                entity_type='sync',
                entity_id='latest',
                last_synced_at=timestamp,
                sync_id=sync_id
            )
            self.db.add(sync_record)
        
        self.db.commit()
    
    def track_bet(self, bet_uuid, updated_at):
        """Track individual bet sync status"""
        bet_record = self.db.query(ReportsSyncTrackingModel)\
            .filter_by(entity_type='bet', entity_id=bet_uuid)\
            .first()
        
        if bet_record:
            bet_record.last_synced_at = updated_at
        else:
            bet_record = ReportsSyncTrackingModel(
                entity_type='bet',
                entity_id=bet_uuid,
                last_synced_at=updated_at
            )
            self.db.add(bet_record)
        
        self.db.commit()
    
    def track_extraction_stat(self, match_id, updated_at):
        """Track individual extraction stat sync status"""
        stat_record = self.db.query(ReportsSyncTrackingModel)\
            .filter_by(entity_type='extraction_stat', entity_id=match_id)\
            .first()
        
        if stat_record:
            stat_record.last_synced_at = updated_at
        else:
            stat_record = ReportsSyncTrackingModel(
                entity_type='extraction_stat',
                entity_id=match_id,
                last_synced_at=updated_at
            )
            self.db.add(stat_record)
        
        self.db.commit()
```

### Step 2: Query Server for Last Sync (NEW)

Before performing a sync, query the server to verify your local tracking:

```python
def verify_server_sync_state(api_token, client_id):
    """Query server for last sync information"""
    import requests
    
    url = "https://your-server.com/api/reports/last-sync"
    headers = {
        "Authorization": f"Bearer {api_token}",
        "Content-Type": "application/json"
    }
    params = {"client_id": client_id}
    
    try:
        response = requests.get(url, headers=headers, params=params)
        response.raise_for_status()
        data = response.json()
        
        if data.get('success'):
            return data
        else:
            print(f"Error: {data.get('error')}")
            return None
    except requests.RequestException as e:
        print(f"Request failed: {str(e)}")
        return None
```

### Step 3: Compare Local and Server State

Compare your local tracking with server state to detect discrepancies:

```python
def sync_state_verification(tracking, api_token, client_id):
    """Verify local tracking matches server state"""
    
    # Get local tracking
    local_info = tracking.get_last_sync_info()
    
    # Get server state
    server_info = verify_server_sync_state(api_token, client_id)
    
    if not server_info:
        print("No sync records on server - this is a first sync")
        return 'first_sync'
    
    if not local_info:
        print("No local tracking - need to recover from server")
        return 'recover_from_server'
    
    # Compare sync IDs
    if local_info['sync_id'] != server_info['last_sync_id']:
        print(f"Sync ID mismatch!")
        print(f"  Local: {local_info['sync_id']}")
        print(f"  Server: {server_info['last_sync_id']}")
        return 'sync_id_mismatch'
    
    # Compare timestamps
    local_time = local_info['last_synced_at']
    server_time = datetime.fromisoformat(server_info['last_sync_timestamp'])
    
    if abs((local_time - server_time).total_seconds()) > 60:
        print(f"Timestamp mismatch!")
        print(f"  Local: {local_time}")
        print(f"  Server: {server_time}")
        return 'timestamp_mismatch'
    
    print("Local tracking matches server state")
    return 'verified'
```

### Step 4: Handle Recovery Scenarios

Implement recovery logic for different scenarios:

```python
def handle_recovery(state, tracking, api_token, client_id):
    """Handle different recovery scenarios"""
    
    if state == 'first_sync':
        # First sync - send all data
        print("Performing first sync - sending all historical data")
        return perform_full_sync(tracking, api_token, client_id)
    
    elif state == 'recover_from_server':
        # Local tracking lost - recover from server
        print("Recovering from server state")
        server_info = verify_server_sync_state(api_token, client_id)
        
        if server_info and server_info['last_sync_id']:
            # Update local tracking with server state
            server_time = datetime.fromisoformat(server_info['last_sync_timestamp'])
            tracking.update_last_sync(
                server_info['last_sync_id'],
                server_time
            )
            print(f"Recovered: sync_id={server_info['last_sync_id']}")
            
            # Perform incremental sync from server's last sync
            return perform_incremental_sync(tracking, api_token, client_id)
        else:
            # No server data either - perform full sync
            return perform_full_sync(tracking, api_token, client_id)
    
    elif state in ['sync_id_mismatch', 'timestamp_mismatch']:
        # Tracking corruption detected
        print("Tracking corruption detected - performing full sync")
        return perform_full_sync(tracking, api_token, client_id)
    
    else:
        # Verified - perform normal incremental sync
        return perform_incremental_sync(tracking, api_token, client_id)
```

### Step 5: Perform Sync Operations

Implement full and incremental sync operations:

```python
def perform_full_sync(tracking, api_token, client_id):
    """Perform full sync - send all historical data"""
    import requests
    from datetime import datetime, timedelta
    
    # Generate unique sync ID
    sync_id = f"sync_{datetime.now().strftime('%Y%m%d_%H%M%S')}_{uuid.uuid4().hex[:8]}"
    
    # Collect all bets and extraction stats
    all_bets = collect_all_bets()
    all_stats = collect_all_extraction_stats()
    
    # Build sync payload
    payload = {
        "sync_id": sync_id,
        "client_id": client_id,
        "sync_timestamp": datetime.utcnow().isoformat(),
        "date_range": "all",
        "start_date": (datetime.utcnow() - timedelta(days=365)).isoformat(),
        "end_date": datetime.utcnow().isoformat(),
        "bets": all_bets,
        "extraction_stats": all_stats,
        "cap_compensation_balance": get_current_cap_balance(),
        "summary": calculate_summary(all_bets, all_stats),
        "is_incremental": False,
        "sync_type": "full"
    }
    
    # Send to server
    return send_sync_request(api_token, payload, tracking)

def perform_incremental_sync(tracking, api_token, client_id):
    """Perform incremental sync - send only new/changed data"""
    import requests
    from datetime import datetime
    
    # Get last sync time from local tracking
    last_sync_info = tracking.get_last_sync_info()
    last_synced_at = last_sync_info['last_synced_at'] if last_sync_info else None
    
    # Generate unique sync ID
    sync_id = f"sync_{datetime.now().strftime('%Y%m%d_%H%M%S')}_{uuid.uuid4().hex[:8]}"
    
    # Collect only new/changed bets and stats
    new_bets = collect_new_bets(last_synced_at)
    new_stats = collect_new_extraction_stats(last_synced_at)
    
    # Build sync payload
    payload = {
        "sync_id": sync_id,
        "client_id": client_id,
        "sync_timestamp": datetime.utcnow().isoformat(),
        "date_range": "all",
        "start_date": last_synced_at.isoformat() if last_synced_at else None,
        "end_date": datetime.utcnow().isoformat(),
        "bets": new_bets,
        "extraction_stats": new_stats,
        "cap_compensation_balance": get_current_cap_balance(),
        "summary": calculate_summary(new_bets, new_stats),
        "is_incremental": True,
        "sync_type": "incremental"
    }
    
    # Send to server
    return send_sync_request(api_token, payload, tracking)

def send_sync_request(api_token, payload, tracking):
    """Send sync request to server and update local tracking"""
    import requests
    from datetime import datetime
    
    url = "https://your-server.com/api/reports/sync"
    headers = {
        "Authorization": f"Bearer {api_token}",
        "Content-Type": "application/json"
    }
    
    try:
        response = requests.post(url, headers=headers, json=payload)
        response.raise_for_status()
        data = response.json()
        
        if data.get('success'):
            print(f"Sync successful: {data.get('synced_count')} items synced")
            
            # Update local tracking
            sync_timestamp = datetime.fromisoformat(payload['sync_timestamp'])
            tracking.update_last_sync(payload['sync_id'], sync_timestamp)
            
            # Track individual bets and stats
            for bet in payload['bets']:
                tracking.track_bet(bet['uuid'], sync_timestamp)
            
            for stat in payload['extraction_stats']:
                tracking.track_extraction_stat(stat['match_id'], sync_timestamp)
            
            return True
        else:
            print(f"Sync failed: {data.get('error')}")
            return False
            
    except requests.RequestException as e:
        print(f"Sync request failed: {str(e)}")
        return False
```

### Step 6: Implement Periodic Sync

Set up periodic sync with verification:

```python
import time
from datetime import datetime

def periodic_sync(tracking, api_token, client_id, interval_minutes=10):
    """Perform periodic sync with server state verification"""
    
    while True:
        try:
            print(f"\n[{datetime.utcnow()}] Starting sync cycle...")
            
            # Step 1: Verify server state
            state = sync_state_verification(tracking, api_token, client_id)
            print(f"Verification state: {state}")
            
            # Step 2: Handle recovery if needed
            result = handle_recovery(state, tracking, api_token, client_id)
            
            if result:
                print(f"Sync completed successfully")
            else:
                print(f"Sync failed - will retry in {interval_minutes} minutes")
            
        except Exception as e:
            print(f"Sync error: {str(e)}")
        
        # Wait for next sync cycle
        print(f"Waiting {interval_minutes} minutes until next sync...")
        time.sleep(interval_minutes * 60)
```

## Best Practices

### 1. Always Verify Before Sync

Before every sync, query the server for last sync information:

```python
# Before sync
server_info = verify_server_sync_state(api_token, client_id)
if server_info:
    print(f"Server last sync: {server_info['last_sync_id']}")
    print(f"Server last sync time: {server_info['last_sync_timestamp']}")
```

### 2. Handle Tracking Corruption

If local tracking is corrupted or lost, recover from server:

```python
# Detect corruption
local_info = tracking.get_last_sync_info()
server_info = verify_server_sync_state(api_token, client_id)

if not local_info and server_info:
    # Local tracking lost - recover from server
    server_time = datetime.fromisoformat(server_info['last_sync_timestamp'])
    tracking.update_last_sync(
        server_info['last_sync_id'],
        server_time
    )
```

### 3. Use Unique Sync IDs

Always generate unique sync IDs for each sync operation:

```python
import uuid
from datetime import datetime

sync_id = f"sync_{datetime.now().strftime('%Y%m%d_%H%M%S')}_{uuid.uuid4().hex[:8]}"
```

### 4. Track Individual Records

Track sync status for individual bets and extraction stats:

```python
# After successful sync
for bet in payload['bets']:
    tracking.track_bet(bet['uuid'], sync_timestamp)

for stat in payload['extraction_stats']:
    tracking.track_extraction_stat(stat['match_id'], sync_timestamp)
```

### 5. Implement Retry Logic

Implement exponential backoff for failed syncs:

```python
import time

def sync_with_retry(tracking, api_token, client_id, max_retries=5):
    """Sync with exponential backoff retry"""
    
    for attempt in range(max_retries):
        try:
            result = perform_incremental_sync(tracking, api_token, client_id)
            if result:
                return True
        except Exception as e:
            print(f"Sync attempt {attempt + 1} failed: {str(e)}")
            
            if attempt < max_retries - 1:
                # Exponential backoff: 60s * 2^attempt
                wait_time = 60 * (2 ** attempt)
                print(f"Retrying in {wait_time} seconds...")
                time.sleep(wait_time)
    
    return False
```

## Error Handling

### Common Errors and Solutions

**Error 401: Authentication required**
- Solution: Ensure API token is valid and included in Authorization header

**Error 403: Access denied**
- Solution: Verify client_id belongs to your API token

**Error 400: Invalid request**
- Solution: Check request format and required fields

**Error 500: Internal server error**
- Solution: Log error details and retry with exponential backoff

## Testing

### Test 1: Query Last Sync

```python
# Test querying server for last sync
server_info = verify_server_sync_state(api_token, client_id)
print(f"Last sync ID: {server_info['last_sync_id']}")
print(f"Last sync time: {server_info['last_sync_timestamp']}")
```

### Test 2: Full Sync

```python
# Test full sync
result = perform_full_sync(tracking, api_token, client_id)
print(f"Full sync result: {result}")
```

### Test 3: Incremental Sync

```python
# Test incremental sync
result = perform_incremental_sync(tracking, api_token, client_id)
print(f"Incremental sync result: {result}")
```

### Test 4: Recovery Scenario

```python
# Simulate tracking corruption
# Delete local tracking records
# Then verify and recover
state = sync_state_verification(tracking, api_token, client_id)
result = handle_recovery(state, tracking, api_token, client_id)
print(f"Recovery result: {result}")
```

## Summary

The new `/api/reports/last-sync` endpoint allows clients to:

1. **Verify Server State**: Query server for last sync information
2. **Detect Tracking Corruption**: Compare local tracking with server state
3. **Recover from Server**: Restore local tracking from server state
4. **Prevent Data Loss**: Ensure no syncs are missed due to tracking issues

By implementing this verification step before each sync, clients can maintain data integrity and recover from tracking corruption automatically.