# Building Fixture Manager as a Single Executable

This document explains how to build the Fixture Manager daemon as a single executable file using PyInstaller.

## Prerequisites

- Python 3.13 or higher (for current build configuration)
- pip (Python package installer)
- At least 2GB of free disk space for the build process
- Internet connection for downloading dependencies
- Compatible GLIBC version (see GLIBC Compatibility section below)

## Quick Build

The easiest way to build the executable is using the automated build script:

```bash
# Make the build script executable (Linux/macOS)
chmod +x build.py

# Run the build script
python build.py
```

This will:
1. Install all required dependencies
2. Clean previous build artifacts
3. Create the single executable file
4. Test the executable
5. Create a distribution package

## Manual Build Process

If you prefer to build manually or need more control:

### 1. Install Build Dependencies

```bash
# Install PyInstaller and build tools
pip install -r requirements-build.txt

# Install runtime dependencies
pip install -r requirements.txt
```

### 2. Clean Previous Builds

```bash
# Remove build artifacts
rm -rf build/ dist/ __pycache__/
find . -name "*.pyc" -delete
```

### 3. Build with PyInstaller

```bash
# Build using the spec file
pyinstaller --clean fixture-manager.spec
```

### 4. Test the Executable

```bash
# Test the built executable
./dist/fixture-manager --help
```

## Build Output

After a successful build, you'll find:

- **`dist/fixture-manager`** - The main executable file (Linux/macOS)
- **`dist/fixture-manager.exe`** - The main executable file (Windows)
- **`distribution/`** - Complete distribution package with:
  - The executable file
  - README.md
  - .env.example (configuration template)
  - database/schema.sql
  - install.sh
  - run.sh (Linux/macOS launcher script)
  - run.bat (Windows launcher script)

## Distribution

The single executable file contains:
- All Python dependencies
- The complete Flask application
- Database models and utilities
- Authentication system
- File upload handlers
- Web dashboard
- RESTful API
- All necessary libraries

### File Size
The executable will be approximately 45-50MB depending on your platform and included dependencies.

### Platform Compatibility
- **Linux**: Build on the target distribution for best compatibility
- **Windows**: Build on Windows 10/11 for maximum compatibility
- **macOS**: Build on macOS 10.15+ for modern compatibility

## Running the Executable

### Basic Usage
```bash
# Start the daemon
./fixture-manager start

# Stop the daemon
./fixture-manager stop

# Check status
./fixture-manager status

# Run in foreground (for testing)
./fixture-manager start --foreground
```

### Configuration
The executable looks for configuration in these locations (in order):
1. `.env` file in the same directory as the executable
2. Environment variables
3. Default configuration values

### First Run Setup
1. Copy `.env.example` to `.env`
2. Edit `.env` with your database and security settings
3. Ensure MySQL is installed and running
4. Run the executable to initialize the database

## GLIBC Compatibility

⚠️ **Important**: Executables built with Python 3.13 require GLIBC 2.38 or higher on the target system.

### Checking GLIBC Version
```bash
# Check GLIBC version on target system
ldd --version
```

### Compatibility Solutions

**If you encounter GLIBC errors:**
```
[PYI-16812:ERROR] Failed to load Python shared library: version `GLIBC_2.38' not found
```

**Solution 1: Build on Target System (Recommended)**
```bash
# Build directly on the deployment system
git clone <repository>
cd MBetter
python3 -m venv venv
source venv/bin/activate
pip install -r requirements-build.txt
python build.py
```

**Solution 2: Use Virtual Environment Deployment**
Instead of a single executable, deploy with a virtual environment for maximum compatibility:
```bash
# Create deployment package
python3 -m venv deployment_venv
source deployment_venv/bin/activate
pip install -r requirements.txt
tar -czf fixture-manager-deployment.tar.gz deployment_venv/ fixture_daemon.py app/ database/ config.py .env.example
```

**Solution 3: Use Docker for Consistent Builds**
```dockerfile
FROM ubuntu:20.04
RUN apt-get update && apt-get install -y python3 python3-pip python3-venv git
WORKDIR /app
COPY . .
RUN python3 -m venv venv && . venv/bin/activate && pip install -r requirements-build.txt
RUN . venv/bin/activate && python build.py
```

See `GLIBC_COMPATIBILITY.md` for detailed solutions.

## Troubleshooting

### Common Build Issues

**1. Missing Dependencies**
```bash
# Install missing system packages (Ubuntu/Debian)
sudo apt-get install python3-dev libmysqlclient-dev build-essential

# Install missing system packages (CentOS/RHEL)
sudo yum install python3-devel mysql-devel gcc gcc-c++
```

**2. PyInstaller Import Errors**
If you get import errors during build, add the missing modules to the `hiddenimports` list in `fixture-manager.spec`.

**3. Large Executable Size**
The executable includes all dependencies. To reduce size:
- Remove unused dependencies from requirements.txt
- Use `--exclude-module` in the spec file for unnecessary modules

**4. Runtime Errors**
If the executable fails at runtime:
- Check that all data files are included in the `datas` section of the spec file
- Verify that hidden imports are correctly specified
- Test with `--debug` flag for more verbose output

### Platform-Specific Issues

**Linux:**
- Ensure all shared libraries are available on the target system
- Consider building on the oldest supported distribution

**Windows:**
- Install Microsoft Visual C++ Redistributable on target systems
- Some antivirus software may flag the executable as suspicious

**macOS:**
- Code signing may be required for distribution
- Gatekeeper may prevent execution of unsigned binaries

## Advanced Configuration

### Custom Build Options

You can modify `fixture-manager.spec` to customize the build:

```python
# Add additional data files
datas = [
    ('custom_templates/', 'templates/'),
    ('static_files/', 'static/'),
]

# Exclude unnecessary modules
excludes = [
    'tkinter',
    'matplotlib',
    'test',
]

# Add custom hidden imports
hiddenimports = [
    'your.custom.module',
]
```

### Build Optimization

For smaller executables:
1. Use `--onefile` for single file distribution
2. Use `--strip` to remove debug symbols (Linux/macOS)
3. Use `--upx` to compress the executable (if UPX is installed)

### Cross-Platform Building

To build for different platforms:
1. Use virtual machines or containers for each target platform
2. Consider using GitHub Actions or similar CI/CD for automated builds
3. Docker can be used for consistent Linux builds

## Security Considerations

### Executable Security
- The executable contains all source code (though compiled)
- Sensitive configuration should still use environment variables
- Consider code obfuscation for proprietary deployments

### Distribution Security
- Sign executables for production distribution
- Provide checksums for integrity verification
- Use secure channels for distribution

## Deployment

### Single Server Deployment
1. Copy the executable to the target server
2. Copy the configuration files
3. Set up the database
4. Run the executable

### Multiple Server Deployment
1. Build once on a compatible system
2. Distribute the same executable to all servers
3. Use different configuration files per server
4. Consider using configuration management tools

## Support

For build issues:
1. Check the build logs for specific error messages
2. Verify all dependencies are correctly installed
3. Test the build on a clean system
4. Check PyInstaller documentation for platform-specific issues

The single executable provides a convenient way to distribute the Fixture Manager daemon without requiring Python installation on target systems.