#!/usr/bin/env python3
"""
Minimal video test - NO overlays at all to test pure QVideoWidget rendering
"""

import sys
import logging
from pathlib import Path
from PyQt6.QtWidgets import QApplication, QMainWindow, QVBoxLayout, QWidget
from PyQt6.QtCore import QUrl
from PyQt6.QtMultimedia import QMediaPlayer, QAudioOutput
from PyQt6.QtMultimediaWidgets import QVideoWidget

# Add project path for imports
project_path = Path(__file__).parent
sys.path.insert(0, str(project_path))


def setup_logging():
    """Setup basic logging"""
    logging.basicConfig(
        level=logging.DEBUG,
        format='%(asctime)s - %(levelname)s - %(message)s',
        handlers=[logging.StreamHandler(sys.stdout)]
    )


class MinimalVideoWindow(QMainWindow):
    """Absolute minimal video player - NO overlays, just pure video"""
    
    def __init__(self):
        super().__init__()
        self.setup_ui()
        self.setup_media_player()
        
    def setup_ui(self):
        """Setup minimal UI - just video widget"""
        self.setWindowTitle("MINIMAL Video Test - NO Overlays")
        self.setGeometry(100, 100, 800, 600)
        
        # PURE BLACK BACKGROUND - no transparency anywhere
        self.setStyleSheet("QMainWindow { background-color: black; }")
        
        # Central widget - completely opaque
        central_widget = QWidget()
        central_widget.setStyleSheet("background-color: black;")
        self.setCentralWidget(central_widget)
        
        # Layout
        layout = QVBoxLayout(central_widget)
        layout.setContentsMargins(0, 0, 0, 0)
        
        # ONLY QVideoWidget - no overlays at all
        self.video_widget = QVideoWidget()
        self.video_widget.setStyleSheet("QVideoWidget { background-color: black; }")
        layout.addWidget(self.video_widget)
        
        print("Minimal video window created - PURE QVideoWidget only")
    
    def setup_media_player(self):
        """Setup media player"""
        self.media_player = QMediaPlayer()
        self.audio_output = QAudioOutput()
        
        self.media_player.setAudioOutput(self.audio_output)
        self.media_player.setVideoOutput(self.video_widget)
        
        # Connect signals for debugging
        self.media_player.playbackStateChanged.connect(self.on_state_changed)
        self.media_player.mediaStatusChanged.connect(self.on_status_changed)
        self.media_player.errorOccurred.connect(self.on_error)
        
        print("Media player setup completed")
    
    def play_video(self, file_path):
        """Play video file"""
        path_obj = Path(file_path)
        
        if not path_obj.exists():
            print(f"ERROR: File not found: {file_path}")
            return
            
        print(f"Loading video: {file_path}")
        print(f"File size: {path_obj.stat().st_size} bytes")
        
        url = QUrl.fromLocalFile(str(path_obj.absolute()))
        print(f"QUrl: {url.toString()}")
        
        self.media_player.setSource(url)
        self.media_player.play()
        
        print("Video play command sent")
    
    def on_state_changed(self, state):
        """Debug state changes"""
        print(f"MEDIA STATE: {state}")
    
    def on_status_changed(self, status):
        """Debug status changes"""
        print(f"MEDIA STATUS: {status}")
    
    def on_error(self, error):
        """Debug errors"""
        print(f"MEDIA ERROR: {error}")
    
    def keyPressEvent(self, event):
        """Handle keys"""
        if event.key() == 32:  # Space
            if self.media_player.playbackState() == QMediaPlayer.PlaybackState.PlayingState:
                self.media_player.pause()
                print("PAUSED")
            else:
                self.media_player.play()
                print("PLAYING")


def main():
    """Test minimal video rendering"""
    setup_logging()
    
    print("MINIMAL VIDEO TEST - NO OVERLAYS")
    print("=" * 40)
    print("This test uses ONLY QVideoWidget with NO overlays")
    print("If video is not visible here, the issue is with QVideoWidget itself")
    print("")
    
    app = QApplication(sys.argv)
    
    window = MinimalVideoWindow()
    window.show()
    
    # Play test video after delay
    from PyQt6.QtCore import QTimer
    QTimer.singleShot(1000, lambda: window.play_video("test_video.mp4"))
    
    print("Window shown. Video should start playing in 1 second.")
    print("Expected: You should see moving test pattern (countdown)")
    print("Controls: Space = Play/Pause, Escape = Exit")
    
    return app.exec()


if __name__ == "__main__":
    sys.exit(main())