#!/usr/bin/env python3
"""
Debug script specifically for testing video playback visibility in Qt player
Tests both native and WebEngine overlays to isolate the video blocking issue
"""

import sys
import logging
import time
from pathlib import Path
from dataclasses import dataclass
from PyQt6.QtWidgets import QApplication
from PyQt6.QtCore import QTimer

# Add project path for imports
project_path = Path(__file__).parent
sys.path.insert(0, str(project_path))

from mbetterclient.qt_player.player import PlayerWindow


@dataclass 
class DebugQtConfig:
    """Debug configuration for Qt player"""
    fullscreen: bool = False
    window_width: int = 800
    window_height: int = 600
    always_on_top: bool = False
    auto_play: bool = True
    volume: float = 0.8
    mute: bool = False
    use_native_overlay: bool = True  # Start with native overlay for testing


def setup_debug_logging():
    """Setup debug logging"""
    logging.basicConfig(
        level=logging.DEBUG,
        format='%(asctime)s - %(name)s - %(levelname)s - %(message)s',
        handlers=[
            logging.StreamHandler(sys.stdout),
            logging.FileHandler('video_debug.log')
        ]
    )


def test_video_playback_native():
    """Test video playback with native overlay (should not block video)"""
    print("Testing Video Playback with NATIVE overlay...")
    
    app = QApplication(sys.argv)
    config = DebugQtConfig()
    config.use_native_overlay = True  # Force native overlay
    
    # Create player window with native overlay
    window = PlayerWindow(config)
    window.show()
    
    # Test with our generated test video
    test_video_path = "test_video.mp4"
    
    def play_test_video():
        """Play the test video after a short delay"""
        print(f"Playing test video: {test_video_path}")
        window.play_video(test_video_path)
        
        # Update overlay to confirm it's working
        overlay_data = {
            'title': 'DEBUG: Native Overlay Test',
            'subtitle': 'Video should be VISIBLE underneath this overlay',
            'ticker': 'If you can see moving colors/patterns, video is working! Native overlay should not block video.'
        }
        # Use the new separate window overlay
        if hasattr(window, 'window_overlay') and window.window_overlay:
            window.window_overlay.update_overlay_data(overlay_data)
        else:
            print("Warning: No window overlay available")
    
    # Play video after 2 seconds
    QTimer.singleShot(2000, play_test_video)
    
    print("Native Overlay Test Window created!")
    print("Expected behavior:")
    print("- You should see a test pattern video (moving colors/gradients)")
    print("- Native overlay text should appear ON TOP of the video")
    print("- If video is NOT visible, the issue is deeper than overlay blocking")
    print("\nControls:")
    print("- Space: Play/Pause")
    print("- Escape: Exit")
    
    return app.exec()


def test_video_playback_webengine():
    """Test video playback with WebEngine overlay (may block video)"""
    print("Testing Video Playback with WEBENGINE overlay...")
    
    app = QApplication(sys.argv)
    config = DebugQtConfig()
    config.use_native_overlay = False  # Force WebEngine overlay
    
    # Create player window with WebEngine overlay
    window = PlayerWindow(config)
    window.show()
    
    # Test with our generated test video
    test_video_path = "test_video.mp4"
    
    def play_test_video():
        """Play the test video after a short delay"""
        print(f"Playing test video: {test_video_path}")
        window.play_video(test_video_path)
        
        # Update overlay to confirm it's working
        overlay_data = {
            'title': 'DEBUG: WebEngine Overlay Test',
            'subtitle': 'Video may be BLOCKED by this overlay',
            'ticker': 'If you CANNOT see moving colors/patterns, WebEngine overlay is blocking the video!'
        }
        
        # Wait for WebEngine to be ready before updating
        def update_overlay_when_ready():
            # Use the new separate window overlay
            if hasattr(window, 'window_overlay') and window.window_overlay:
                overlay_view = window.window_overlay
                if hasattr(overlay_view, 'overlay_channel') and overlay_view.overlay_channel:
                    if window._is_webengine_ready(overlay_view):
                        overlay_view.update_overlay_data(overlay_data)
                        print("WebEngine overlay updated")
                    else:
                        print("WebEngine not ready, retrying...")
                        QTimer.singleShot(1000, update_overlay_when_ready)
                else:
                    overlay_view.update_overlay_data(overlay_data)
            else:
                print("Warning: No window overlay available")
        
        QTimer.singleShot(3000, update_overlay_when_ready)
    
    # Play video after 2 seconds
    QTimer.singleShot(2000, play_test_video)
    
    print("WebEngine Overlay Test Window created!")
    print("Expected behavior:")
    print("- You should see a test pattern video (moving colors/gradients)")
    print("- WebEngine overlay text should appear ON TOP of the video")
    print("- If video is NOT visible, WebEngine overlay is blocking it")
    print("\nControls:")
    print("- Space: Play/Pause") 
    print("- Escape: Exit")
    
    return app.exec()


def test_uploaded_video():
    """Test with an actual uploaded video file"""
    print("Testing with uploaded video files...")
    
    # Look for uploaded videos
    uploads_dir = Path("uploads")
    if uploads_dir.exists():
        video_files = list(uploads_dir.glob("*.mp4"))
        if video_files:
            video_path = video_files[0]  # Use first video found
            print(f"Found uploaded video: {video_path}")
            
            app = QApplication(sys.argv)
            config = DebugQtConfig()
            config.use_native_overlay = True  # Start with native
            
            window = PlayerWindow(config)
            window.show()
            
            def play_uploaded_video():
                print(f"Playing uploaded video: {video_path}")
                window.play_video(str(video_path))
                
                overlay_data = {
                    'title': f'Playing: {video_path.name}',
                    'subtitle': 'Testing uploaded video with native overlay',
                    'ticker': 'This is a real uploaded video file. Video should be visible with native overlay.'
                }
                # Use the new separate window overlay
                if hasattr(window, 'window_overlay') and window.window_overlay:
                    window.window_overlay.update_overlay_data(overlay_data)
                else:
                    print("Warning: No window overlay available")
            
            QTimer.singleShot(2000, play_uploaded_video)
            
            print(f"Testing uploaded video: {video_path.name}")
            print("This tests with a real uploaded video file")
            
            return app.exec()
        else:
            print("No video files found in uploads directory")
            return 1
    else:
        print("Uploads directory not found")
        return 1


def main():
    """Main debug function"""
    setup_debug_logging()
    
    print("Qt Video Player Debug Suite")
    print("=" * 40)
    
    if len(sys.argv) > 1:
        test_mode = sys.argv[1]
    else:
        print("Available test modes:")
        print("1. native    - Test with native Qt overlay (should show video)")
        print("2. webengine - Test with WebEngine overlay (may block video)")
        print("3. uploaded  - Test with uploaded video file")
        print()
        choice = input("Select test mode (1, 2, or 3): ").strip()
        if choice == "1":
            test_mode = "native"
        elif choice == "2":
            test_mode = "webengine"
        elif choice == "3":
            test_mode = "uploaded"
        else:
            test_mode = "native"
    
    try:
        if test_mode == "native":
            return test_video_playback_native()
        elif test_mode == "webengine":
            return test_video_playback_webengine()
        elif test_mode == "uploaded":
            return test_uploaded_video()
        else:
            print(f"Unknown test mode: {test_mode}")
            return 1
            
    except Exception as e:
        print(f"Test failed with error: {e}")
        import traceback
        traceback.print_exc()
        return 1


if __name__ == "__main__":
    sys.exit(main())