#!/usr/bin/env python3
"""
Test script for Qt player functionality
"""

import sys
import os
import logging
import time
import threading
from pathlib import Path

# Add the project root to Python path
project_root = Path(__file__).parent
sys.path.insert(0, str(project_root))

from mbetterclient.config.settings import AppSettings
from mbetterclient.core.message_bus import MessageBus, MessageBuilder, MessageType
from mbetterclient.qt_player.player import QtVideoPlayer

def setup_logging():
    """Setup logging for the test"""
    logging.basicConfig(
        level=logging.DEBUG,
        format='%(asctime)s - %(name)s - %(levelname)s - %(message)s'
    )
    return logging.getLogger(__name__)

def test_qt_player_standalone():
    """Test Qt player in standalone mode"""
    logger = setup_logging()
    logger.info("Starting Qt player standalone test")
    
    # Create settings
    settings = AppSettings()
    settings.qt.fullscreen = False
    settings.qt.window_width = 800
    settings.qt.window_height = 600
    
    # Create message bus
    message_bus = MessageBus()
    
    # Create Qt player
    qt_player = QtVideoPlayer(message_bus, settings.qt)
    
    # Initialize Qt player
    if not qt_player.initialize():
        logger.error("Failed to initialize Qt player")
        return 1
    
    logger.info("Qt player initialized successfully")
    
    # Start message processing in a separate thread
    qt_player.start_message_processing()
    
    # Send a test message to display default overlay
    test_message = MessageBuilder.template_change(
        sender="test",
        template_data={
            "title": "Qt Player Test",
            "subtitle": "Standalone Mode Test",
            "ticker": "This is a test of the Qt player in standalone mode"
        }
    )
    message_bus.publish(test_message)
    
    # Run Qt event loop (this will block until window is closed)
    logger.info("Running Qt event loop - close the window to exit")
    exit_code = qt_player.run()
    
    # Cleanup
    qt_player.shutdown()
    
    logger.info("Qt player test completed")
    return exit_code

def test_qt_player_with_message_bus():
    """Test Qt player with message bus communication"""
    logger = setup_logging()
    logger.info("Starting Qt player message bus test")
    
    # Create settings
    settings = AppSettings()
    settings.qt.fullscreen = False
    settings.qt.window_width = 800
    settings.qt.window_height = 600
    
    # Create message bus
    message_bus = MessageBus()
    
    # Create Qt player
    qt_player = QtVideoPlayer(message_bus, settings.qt)
    
    # Initialize Qt player
    if not qt_player.initialize():
        logger.error("Failed to initialize Qt player")
        return 1
    
    logger.info("Qt player initialized successfully")
    
    # Start message processing in a separate thread
    qt_player.start_message_processing()
    
    # Send test messages
    def send_test_messages():
        time.sleep(2)  # Wait for window to be ready
        
        # Send overlay update
        overlay_message = MessageBuilder.overlay_update(
            sender="test",
            overlay_data={
                "title": "Message Bus Test",
                "subtitle": "Testing message bus communication",
                "showStats": True
            }
        )
        message_bus.publish(overlay_message)
        
        time.sleep(3)
        
        # Send another overlay update
        overlay_message2 = MessageBuilder.overlay_update(
            sender="test",
            overlay_data={
                "title": "Message Bus Test Continued",
                "subtitle": "Second message bus test",
                "ticker": "Testing continuous updates through message bus"
            }
        )
        message_bus.publish(overlay_message2)
        
        logger.info("Test messages sent")
    
    # Start message sending in a separate thread
    message_thread = threading.Thread(target=send_test_messages)
    message_thread.start()
    
    # Run Qt event loop (this will block until window is closed)
    logger.info("Running Qt event loop with message bus test - close the window to exit")
    exit_code = qt_player.run()
    
    # Wait for message thread to finish
    message_thread.join()
    
    # Cleanup
    qt_player.shutdown()
    
    logger.info("Qt player message bus test completed")
    return exit_code

if __name__ == "__main__":
    if len(sys.argv) > 1 and sys.argv[1] == "standalone":
        exit_code = test_qt_player_standalone()
    elif len(sys.argv) > 1 and sys.argv[1] == "message_bus":
        exit_code = test_qt_player_with_message_bus()
    else:
        print("Usage: python test_qt_player.py [standalone|message_bus]")
        print("  standalone: Test Qt player in standalone mode")
        print("  message_bus: Test Qt player with message bus communication")
        sys.exit(1)
    
    sys.exit(exit_code)