#!/usr/bin/env python3
"""
Standalone test application for PyQt6 Video Player with QWebEngineView overlay
"""

import sys
import logging
import time
from pathlib import Path
from dataclasses import dataclass
from PyQt6.QtWidgets import QApplication
from PyQt6.QtCore import QTimer

# Add project path for imports
project_path = Path(__file__).parent
sys.path.insert(0, str(project_path))

from mbetterclient.qt_player.qt6_player import Qt6VideoPlayer, PlayerWindow
from mbetterclient.core.message_bus import MessageBus, MessageBuilder
from mbetterclient.config.settings import QtConfig


@dataclass
class TestQtConfig:
    """Test configuration for Qt player"""
    fullscreen: bool = False
    window_width: int = 1280
    window_height: int = 720
    always_on_top: bool = False
    auto_play: bool = True
    volume: float = 0.8
    mute: bool = False


def setup_logging():
    """Setup logging for the test application"""
    logging.basicConfig(
        level=logging.INFO,
        format='%(asctime)s - %(name)s - %(levelname)s - %(message)s',
        handlers=[
            logging.StreamHandler(sys.stdout),
            logging.FileHandler('qt6_player_test.log')
        ]
    )


def test_standalone_player():
    """Test the standalone PyQt6 player window"""
    print("Testing Standalone PyQt6 Player...")
    
    app = QApplication(sys.argv)
    config = TestQtConfig()
    
    # Create player window directly
    window = PlayerWindow(config)
    
    # Show window
    window.show()
    
    # Test overlay updates
    overlay_view = window.video_widget.get_overlay_view()
    
    def update_overlay_demo():
        """Demo function to update overlay periodically"""
        current_time = time.strftime("%H:%M:%S")
        overlay_data = {
            'title': f'PyQt6 Demo - {current_time}',
            'subtitle': 'Multi-threaded Video Player with WebEngine Overlay',
            'ticker': 'Real-time JavaScript ↔ Python Communication • Hardware Accelerated Video • Professional Animations'
        }
        overlay_view.update_overlay_data(overlay_data)
        print(f"Updated overlay at {current_time}")
    
    # Setup periodic overlay updates
    timer = QTimer()
    timer.timeout.connect(update_overlay_demo)
    timer.start(2000)  # Update every 2 seconds
    
    # Initial overlay update
    update_overlay_demo()
    
    print("PyQt6 Player Window created successfully!")
    print("Features demonstrated:")
    print("- QMediaPlayer + QVideoWidget for hardware-accelerated video")
    print("- QWebEngineView overlay with transparent background")
    print("- QWebChannel bidirectional Python ↔ JavaScript communication")
    print("- CSS3 animations with GSAP integration")
    print("- Thread-safe signal/slot mechanisms")
    print("- QTimer integration for real-time updates")
    print("- Professional UI with responsive design")
    print("- Cross-platform compatibility")
    print("\nControls:")
    print("- Space: Play/Pause")
    print("- F11: Toggle Fullscreen")
    print("- S: Toggle Stats Panel")
    print("- M: Toggle Mute")
    print("- Escape: Exit")
    print("\nClose the window to exit the test.")
    
    return app.exec()


def test_threaded_player():
    """Test the full threaded PyQt6 player component"""
    print("Testing Threaded PyQt6 Player Component...")
    
    # Create message bus
    message_bus = MessageBus()
    
    # Create Qt config
    config = TestQtConfig()
    
    # Create Qt6 player component
    player = Qt6VideoPlayer(message_bus, config)
    
    # Initialize player
    if not player.initialize():
        print("Failed to initialize Qt6VideoPlayer!")
        return 1
    
    # Start player in separate thread (simulation)
    print("Qt6VideoPlayer initialized successfully!")
    
    # Test sending messages
    def send_test_messages():
        """Send test messages to player"""
        time.sleep(2)
        
        # Test overlay update
        overlay_message = MessageBuilder.template_change(
            sender="test_app",
            template_name="demo_template",
            template_data={
                'title': 'Threaded Player Demo',
                'subtitle': 'Message Bus Communication Test',
                'ticker': 'Successfully communicating via MessageBus • Multi-threaded Architecture • Real-time Updates'
            }
        )
        overlay_message.recipient = "qt6_player"
        message_bus.publish(overlay_message)
        print("Sent overlay update message")
        
        # Test video info update
        time.sleep(2)
        video_info_message = MessageBuilder.system_status(
            sender="test_app",
            status="demo",
            details={
                'videoInfo': {
                    'resolution': '1920x1080',
                    'bitrate': '8.5 Mbps',
                    'codec': 'H.265/HEVC',
                    'fps': '60.0'
                }
            }
        )
        video_info_message.recipient = "qt6_player"
        message_bus.publish(video_info_message)
        print("Sent video info update")
    
    # Setup test message timer
    timer = QTimer()
    timer.timeout.connect(send_test_messages)
    timer.setSingleShot(True)
    timer.start(1000)  # Start after 1 second
    
    print("Threaded player test started. Close the player window to exit.")
    
    # Run the player (this would normally be in a separate thread)
    try:
        player.run()
    except KeyboardInterrupt:
        print("Stopping player...")
        player.shutdown()
    
    return 0


def main():
    """Main test function"""
    setup_logging()
    
    print("PyQt6 Multi-threaded Video Player Test Suite")
    print("=" * 50)
    
    if len(sys.argv) > 1:
        test_mode = sys.argv[1]
    else:
        print("Available test modes:")
        print("1. standalone - Test standalone player window")
        print("2. threaded   - Test full threaded player component")
        print()
        test_mode = input("Select test mode (1 or 2): ").strip()
        if test_mode == "1":
            test_mode = "standalone"
        elif test_mode == "2":
            test_mode = "threaded"
        else:
            test_mode = "standalone"
    
    try:
        if test_mode == "standalone":
            return test_standalone_player()
        elif test_mode == "threaded":
            return test_threaded_player()
        else:
            print(f"Unknown test mode: {test_mode}")
            return 1
            
    except Exception as e:
        print(f"Test failed with error: {e}")
        import traceback
        traceback.print_exc()
        return 1


if __name__ == "__main__":
    sys.exit(main())