"""
PyQt6 Multi-threaded Video Player with QWebEngineView Overlay System
Replaces the previous PyQt5 implementation with enhanced functionality
"""

import sys
import time
import logging
import json
import threading
import signal
from pathlib import Path
from typing import Optional, Dict, Any
from PyQt6.QtWidgets import (
    QApplication, QMainWindow, QWidget, QVBoxLayout, QHBoxLayout,
    QLabel, QPushButton, QSlider, QFrame, QStackedWidget
)
from PyQt6.QtCore import (
    Qt, QTimer, QThread, pyqtSignal, QUrl, QRect, QPropertyAnimation,
    QEasingCurve, QSequentialAnimationGroup, QObject, QMutex, QMutexLocker,
    QThreadPool, QRunnable, pyqtSlot
)
from PyQt6.QtGui import (
    QFont, QPainter, QPen, QBrush, QColor, QPixmap, QMovie, 
    QLinearGradient, QFontMetrics, QAction
)
from PyQt6.QtMultimedia import QMediaPlayer, QAudioOutput
from PyQt6.QtMultimediaWidgets import QVideoWidget
from PyQt6.QtWebEngineWidgets import QWebEngineView
from PyQt6.QtWebChannel import QWebChannel

from ..core.thread_manager import ThreadedComponent
from ..core.message_bus import MessageBus, Message, MessageType, MessageBuilder
from ..config.settings import QtConfig

logger = logging.getLogger(__name__)


class OverlayWebChannel(QObject):
    """QObject for WebChannel communication with overlay HTML/JS"""
    
    # Signals to send data to JavaScript
    dataUpdated = pyqtSignal(dict)
    positionChanged = pyqtSignal(float, float)  # position, duration in seconds
    videoInfoChanged = pyqtSignal(dict)
    
    def __init__(self):
        super().__init__()
        self.mutex = QMutex()
        self.overlay_data = {}
        logger.info("OverlayWebChannel initialized")
    
    @pyqtSlot(str)
    def updateTitle(self, title: str):
        """Update main title from JavaScript"""
        with QMutexLocker(self.mutex):
            self.overlay_data['title'] = title
            self.dataUpdated.emit({'title': title})
            logger.debug(f"Title updated: {title}")
    
    @pyqtSlot(str)
    def updateSubtitle(self, subtitle: str):
        """Update subtitle from JavaScript"""
        with QMutexLocker(self.mutex):
            self.overlay_data['subtitle'] = subtitle
            self.dataUpdated.emit({'subtitle': subtitle})
            logger.debug(f"Subtitle updated: {subtitle}")
    
    @pyqtSlot(bool)
    def toggleStats(self, show: bool):
        """Toggle stats panel visibility"""
        with QMutexLocker(self.mutex):
            self.overlay_data['showStats'] = show
            self.dataUpdated.emit({'showStats': show})
            logger.debug(f"Stats panel toggled: {show}")
    
    def send_data_update(self, data: Dict[str, Any]):
        """Send data update to JavaScript (thread-safe)"""
        with QMutexLocker(self.mutex):
            self.overlay_data.update(data)
            self.dataUpdated.emit(data)
    
    def send_position_update(self, position: float, duration: float):
        """Send playback position update to JavaScript (thread-safe)"""
        self.positionChanged.emit(position, duration)
    
    def send_video_info(self, info: Dict[str, Any]):
        """Send video information to JavaScript (thread-safe)"""
        self.videoInfoChanged.emit(info)


class VideoProcessingWorker(QRunnable):
    """Background worker for video processing tasks"""
    
    def __init__(self, task_type: str, data: Dict[str, Any], callback=None):
        super().__init__()
        self.task_type = task_type
        self.data = data
        self.callback = callback
        self.setAutoDelete(True)
    
    def run(self):
        """Execute the video processing task"""
        try:
            logger.debug(f"Processing video task: {self.task_type}")
            
            if self.task_type == "metadata_extraction":
                result = self._extract_metadata()
            elif self.task_type == "thumbnail_generation":
                result = self._generate_thumbnail()
            elif self.task_type == "overlay_rendering":
                result = self._render_overlay()
            else:
                result = {"error": f"Unknown task type: {self.task_type}"}
            
            if self.callback:
                self.callback(result)
                
        except Exception as e:
            logger.error(f"Video processing worker error: {e}")
            if self.callback:
                self.callback({"error": str(e)})
    
    def _extract_metadata(self) -> Dict[str, Any]:
        """Extract video metadata"""
        # Placeholder for metadata extraction
        return {
            "resolution": "1920x1080",
            "bitrate": "5.2 Mbps",
            "codec": "H.264",
            "fps": "30.0"
        }
    
    def _generate_thumbnail(self) -> Dict[str, Any]:
        """Generate video thumbnail"""
        # Placeholder for thumbnail generation
        return {"thumbnail_path": "/tmp/thumbnail.jpg"}
    
    def _render_overlay(self) -> Dict[str, Any]:
        """Render overlay elements"""
        # Placeholder for overlay rendering
        return {"rendered": True}


class OverlayWebView(QWebEngineView):
    """Custom QWebEngineView for video overlays with transparent background"""
    
    def __init__(self, parent=None):
        super().__init__(parent)
        self.web_channel = None
        self.overlay_channel = None
        self.setup_web_view()
        logger.info("OverlayWebView initialized")
    
    def setup_web_view(self):
        """Setup web view as embedded overlay with CSS transparency"""
        logger.debug("OverlayWebView.setup_web_view() - Starting embedded setup")
        
        # Use CSS transparency instead of widget transparency to prevent desktop bleed-through
        logger.debug("Using CSS transparency for embedded overlay")
        
        # Configure page settings for proper rendering with transparent background
        page = self.page()
        logger.debug("Setting page background to transparent (0,0,0,0)")
        page.setBackgroundColor(QColor(0, 0, 0, 0))  # Transparent page background
        
        # Set QWebEngineView style for embedded use - no widget transparency
        self.setStyleSheet("""
            QWebEngineView {
                background: transparent;
                border: none;
            }
        """)
        
        # Ensure overlay is positioned correctly as child widget
        self.setAttribute(Qt.WidgetAttribute.WA_OpaquePaintEvent, False)
        
        logger.debug("OverlayWebView embedded setup completed")
        
        # Setup WebChannel
        self.web_channel = QWebChannel()
        self.overlay_channel = OverlayWebChannel()
        self.web_channel.registerObject("overlay", self.overlay_channel)
        page.setWebChannel(self.web_channel)
        
        # Load overlay HTML (handle both development and PyInstaller bundle modes)
        overlay_path = self._get_overlay_path()
        if overlay_path and overlay_path.exists():
            self.load(QUrl.fromLocalFile(str(overlay_path)))
            logger.info(f"Loaded overlay HTML: {overlay_path}")
        else:
            logger.error(f"Overlay HTML not found: {overlay_path}")
            # Load fallback minimal overlay
            self._load_fallback_overlay()
    
    def _get_overlay_path(self) -> Optional[Path]:
        """Get overlay HTML path, handling PyInstaller bundle mode"""
        try:
            # Check if running as PyInstaller bundle
            if hasattr(sys, '_MEIPASS'):
                # Running as bundled executable
                bundle_path = Path(sys._MEIPASS)
                overlay_path = bundle_path / "mbetterclient" / "qt_player" / "overlay.html"
                logger.debug(f"Bundle mode - checking overlay at: {overlay_path}")
                return overlay_path
            else:
                # Running in development mode
                overlay_path = Path(__file__).parent / "overlay.html"
                logger.debug(f"Development mode - checking overlay at: {overlay_path}")
                return overlay_path
                
        except Exception as e:
            logger.error(f"Failed to determine overlay path: {e}")
            return None
    
    def _load_fallback_overlay(self):
        """Load minimal fallback overlay if main overlay file not found"""
        try:
            fallback_html = """
            <!DOCTYPE html>
            <html>
            <head>
                <meta charset="UTF-8">
                <style>
                    body {
                        margin: 0;
                        padding: 20px;
                        background: transparent;
                        color: white;
                        font-family: Arial, sans-serif;
                    }
                    .overlay-content {
                        position: absolute;
                        top: 20px;
                        left: 20px;
                        background: rgba(0,0,0,0.7);
                        padding: 10px;
                        border-radius: 5px;
                    }
                </style>
            </head>
            <body>
                <div class="overlay-content">
                    <h2>MbetterClient PyQt6 Player</h2>
                    <p>Overlay System Active</p>
                </div>
            </body>
            </html>
            """
            
            self.setHtml(fallback_html)
            logger.info("Loaded fallback overlay HTML")
            
        except Exception as e:
            logger.error(f"Failed to load fallback overlay: {e}")
    
    def get_overlay_channel(self) -> OverlayWebChannel:
        """Get the overlay communication channel"""
        return self.overlay_channel
    
    def update_overlay_data(self, data: Dict[str, Any]):
        """Update overlay with new data"""
        if self.overlay_channel:
            self.overlay_channel.send_data_update(data)
    
    def update_position(self, position: float, duration: float):
        """Update playback position"""
        if self.overlay_channel:
            self.overlay_channel.send_position_update(position, duration)
    
    def update_video_info(self, info: Dict[str, Any]):
        """Update video information"""
        if self.overlay_channel:
            self.overlay_channel.send_video_info(info)


class NativeOverlayWidget(QWidget):
    """Native Qt overlay widget - no WebEngine to prevent freezing"""
    
    def __init__(self, parent=None):
        super().__init__(parent)
        # Initialize overlay_data BEFORE setup_ui() to prevent attribute errors
        self.overlay_data = {
            'title': 'MbetterClient PyQt6 Player',
            'subtitle': 'Ready for Content',
            'ticker': 'Welcome to MbetterClient • Professional Video Overlay System • Real-time Updates',
            'currentTime': '00:00:00'
        }
        # Add overlay_channel as None for compatibility with WebEngine interface
        self.overlay_channel = None
        self.setup_ui()
        logger.info("NativeOverlayWidget initialized")
    
    def setup_ui(self):
        """Setup native Qt overlay widgets"""
        # Make overlay transparent to mouse events but NOT to paint events
        self.setAttribute(Qt.WidgetAttribute.WA_TransparentForMouseEvents, True)
        # Use transparent background but maintain proper widget layering
        self.setStyleSheet("background: rgba(0,0,0,0);")
        # Ensure overlay doesn't interfere with window opacity
        self.setAttribute(Qt.WidgetAttribute.WA_NoSystemBackground, False)
        
        # Main title label
        self.title_label = QLabel()
        self.title_label.setStyleSheet("""
            QLabel {
                color: white;
                font-size: 32px;
                font-weight: bold;
                background: transparent;
                padding: 10px;
                border-radius: 5px;
                text-shadow: 2px 2px 4px rgba(0,0,0,0.8);
            }
        """)
        self.title_label.setAlignment(Qt.AlignmentFlag.AlignCenter)
        self.title_label.setWordWrap(True)
        
        # Subtitle label
        self.subtitle_label = QLabel()
        self.subtitle_label.setStyleSheet("""
            QLabel {
                color: #cccccc;
                font-size: 18px;
                background: transparent;
                padding: 5px;
                text-shadow: 1px 1px 2px rgba(0,0,0,0.8);
            }
        """)
        self.subtitle_label.setAlignment(Qt.AlignmentFlag.AlignCenter)
        
        # Time display
        self.time_label = QLabel()
        self.time_label.setStyleSheet("""
            QLabel {
                color: white;
                font-size: 16px;
                background: rgba(0,0,0,0.5);
                padding: 8px;
                border-radius: 5px;
            }
        """)
        
        # Ticker label
        self.ticker_label = QLabel()
        self.ticker_label.setStyleSheet("""
            QLabel {
                color: white;
                font-size: 14px;
                background: rgba(220, 53, 69, 0.9);
                padding: 8px;
                border-radius: 5px;
            }
        """)
        self.ticker_label.setWordWrap(True)
        
        # Layout
        layout = QVBoxLayout(self)
        layout.setContentsMargins(20, 20, 20, 20)
        
        # Top section
        layout.addStretch(1)
        layout.addWidget(self.title_label)
        layout.addWidget(self.subtitle_label)
        layout.addStretch(2)
        
        # Bottom section
        layout.addWidget(self.ticker_label)
        layout.addSpacing(10)
        
        # Position time label in top-right corner
        self.time_label.setParent(self)
        
        # Update display with initial data
        self.update_display()
        
        logger.debug("NativeOverlayWidget UI setup completed")
    
    def resizeEvent(self, event):
        """Handle resize events"""
        super().resizeEvent(event)
        # Position time label in top-right corner
        if hasattr(self, 'time_label') and self.time_label:
            self.time_label.adjustSize()
            self.time_label.move(self.width() - self.time_label.width() - 20, 20)
    
    def update_overlay_data(self, data: Dict[str, Any]):
        """Update overlay display with new data"""
        try:
            updated = False
            
            if 'title' in data:
                self.overlay_data['title'] = data['title']
                updated = True
            
            if 'subtitle' in data:
                self.overlay_data['subtitle'] = data['subtitle']
                updated = True
            
            if 'ticker' in data:
                self.overlay_data['ticker'] = data['ticker']
                updated = True
            
            if 'currentTime' in data:
                self.overlay_data['currentTime'] = data['currentTime']
                updated = True
            
            if updated:
                self.update_display()
            
            logger.debug(f"Native overlay updated: {data}")
            
        except Exception as e:
            logger.error(f"Failed to update native overlay: {e}")
    
    def update_display(self):
        """Update all display elements"""
        try:
            self.title_label.setText(self.overlay_data.get('title', ''))
            self.subtitle_label.setText(self.overlay_data.get('subtitle', ''))
            self.ticker_label.setText(self.overlay_data.get('ticker', ''))
            self.time_label.setText(self.overlay_data.get('currentTime', ''))
            
            # Adjust time label position
            self.time_label.adjustSize()
            if self.width() > 0:
                self.time_label.move(self.width() - self.time_label.width() - 20, 20)
            
        except Exception as e:
            logger.error(f"Failed to update native overlay display: {e}")
    
    def update_position(self, position: float, duration: float):
        """Update playback position (compatible with WebEngine interface)"""
        try:
            current_time = self.format_time(position)
            total_time = self.format_time(duration)
            self.update_overlay_data({'currentTime': f"{current_time} / {total_time}"})
        except Exception as e:
            logger.error(f"Failed to update position in native overlay: {e}")
    
    def update_video_info(self, info: Dict[str, Any]):
        """Update video information (compatible with WebEngine interface)"""
        # Native overlay doesn't show detailed video stats by default
        # This is for compatibility with the WebEngine interface
        pass
    
    def format_time(self, seconds: float) -> str:
        """Format time in seconds to MM:SS"""
        mins = int(seconds // 60)
        secs = int(seconds % 60)
        return f"{mins:02d}:{secs:02d}"


class VideoWidget(QWidget):
    """Composite video widget with selectable overlay type"""
    
    def __init__(self, parent=None, use_native_overlay=False):
        super().__init__(parent)
        self.use_native_overlay = use_native_overlay
        self.setup_ui()
        logger.info(f"VideoWidget initialized (native_overlay={use_native_overlay})")
    
    def setup_ui(self):
        """Setup video player with selectable overlay type"""
        # CRITICAL: Container must be completely opaque to prevent desktop bleed-through
        self.setStyleSheet("VideoWidget { background-color: black; }")
        self.setAttribute(Qt.WidgetAttribute.WA_OpaquePaintEvent, True)
        self.setAttribute(Qt.WidgetAttribute.WA_NoSystemBackground, False)
        
        # Create layout for the base video widget
        layout = QVBoxLayout(self)
        layout.setContentsMargins(0, 0, 0, 0)
        layout.setSpacing(0)
        
        # LAYER 1: Video widget as base layer
        # Shows solid black when no video, shows video content when playing
        self.video_widget = QVideoWidget()
        self.video_widget.setStyleSheet("QVideoWidget { background-color: black; }")
        self.video_widget.setAttribute(Qt.WidgetAttribute.WA_OpaquePaintEvent, True)
        layout.addWidget(self.video_widget)
        
        # LAYER 2: Overlay - either native Qt or QWebEngineView
        if self.use_native_overlay:
            self.overlay_view = NativeOverlayWidget(self)
            logger.debug("VideoWidget using native Qt overlay")
        else:
            self.overlay_view = OverlayWebView(self)
            logger.debug("VideoWidget using QWebEngineView overlay")
        
        logger.debug(f"VideoWidget overlay setup completed (native={self.use_native_overlay})")
    
    def resizeEvent(self, event):
        """Handle resize events"""
        super().resizeEvent(event)
        # Position overlay to cover the video widget area
        if self.overlay_view:
            # Position overlay to match the video widget exactly
            video_geometry = self.video_widget.geometry()
            self.overlay_view.setGeometry(video_geometry)
            self.overlay_view.raise_()  # Bring overlay to front
            logger.debug(f"Overlay repositioned to {video_geometry}")
    
    def get_video_widget(self) -> QVideoWidget:
        """Get the video widget for media player"""
        return self.video_widget
    
    def get_overlay_view(self):
        """Get the overlay view (either native or WebEngine)"""
        return self.overlay_view


class PlayerControlsWidget(QWidget):
    """Enhanced video player controls with thread-safe operations"""
    
    play_pause_clicked = pyqtSignal()
    stop_clicked = pyqtSignal()
    seek_requested = pyqtSignal(int)
    volume_changed = pyqtSignal(int)
    
    def __init__(self, parent=None):
        super().__init__(parent)
        self.mutex = QMutex()
        self.setup_ui()
    
    def setup_ui(self):
        """Setup enhanced control UI"""
        layout = QHBoxLayout(self)
        layout.setContentsMargins(10, 5, 10, 5)
        
        # Play/Pause button
        self.play_pause_btn = QPushButton("⏸️")
        self.play_pause_btn.setFixedSize(40, 30)
        self.play_pause_btn.clicked.connect(self.play_pause_clicked.emit)
        
        # Stop button
        self.stop_btn = QPushButton("⏹️")
        self.stop_btn.setFixedSize(40, 30)
        self.stop_btn.clicked.connect(self.stop_clicked.emit)
        
        # Position slider
        self.position_slider = QSlider(Qt.Orientation.Horizontal)
        self.position_slider.setMinimum(0)
        self.position_slider.setMaximum(100)
        self.position_slider.sliderPressed.connect(self._on_slider_pressed)
        self.position_slider.sliderReleased.connect(self._on_slider_released)
        
        # Volume slider
        self.volume_slider = QSlider(Qt.Orientation.Horizontal)
        self.volume_slider.setMinimum(0)
        self.volume_slider.setMaximum(100)
        self.volume_slider.setValue(100)
        self.volume_slider.setMaximumWidth(100)
        self.volume_slider.valueChanged.connect(self.volume_changed.emit)
        
        # Time labels
        self.current_time_label = QLabel("00:00")
        self.duration_label = QLabel("00:00")
        
        # Volume label
        volume_label = QLabel("🔊")
        
        # Layout
        layout.addWidget(self.play_pause_btn)
        layout.addWidget(self.stop_btn)
        layout.addWidget(self.current_time_label)
        layout.addWidget(self.position_slider, 1)  # Stretch
        layout.addWidget(self.duration_label)
        layout.addWidget(volume_label)
        layout.addWidget(self.volume_slider)
        
        # Enhanced styling
        self.setStyleSheet("""
            QWidget {
                background-color: rgba(0, 0, 0, 200);
                color: white;
            }
            QPushButton {
                border: 1px solid #555;
                border-radius: 5px;
                background-color: #333;
                font-weight: bold;
            }
            QPushButton:hover {
                background-color: #555;
            }
            QPushButton:pressed {
                background-color: #777;
            }
            QSlider::groove:horizontal {
                border: 1px solid #555;
                height: 6px;
                background: #222;
                border-radius: 3px;
            }
            QSlider::handle:horizontal {
                background: #fff;
                border: 2px solid #555;
                width: 14px;
                margin: -5px 0;
                border-radius: 7px;
            }
            QSlider::handle:horizontal:hover {
                background: #ddd;
            }
        """)
        
        self.slider_pressed = False
    
    def _on_slider_pressed(self):
        with QMutexLocker(self.mutex):
            self.slider_pressed = True
    
    def _on_slider_released(self):
        with QMutexLocker(self.mutex):
            self.slider_pressed = False
            self.seek_requested.emit(self.position_slider.value())
    
    def update_position(self, position: int, duration: int):
        """Update position display (thread-safe)"""
        with QMutexLocker(self.mutex):
            if not self.slider_pressed and duration > 0:
                self.position_slider.setValue(int(position * 100 / duration))
            
            self.current_time_label.setText(self._format_time(position))
            self.duration_label.setText(self._format_time(duration))
    
    def update_play_pause_button(self, is_playing: bool):
        """Update play/pause button state (thread-safe)"""
        with QMutexLocker(self.mutex):
            self.play_pause_btn.setText("⏸️" if is_playing else "▶️")
    
    def _format_time(self, ms: int) -> str:
        """Format time in milliseconds to MM:SS"""
        seconds = ms // 1000
        minutes = seconds // 60
        seconds = seconds % 60
        return f"{minutes:02d}:{seconds:02d}"


class PlayerWindow(QMainWindow):
    """Enhanced main player window with thread-safe operations"""
    
    # Signals for thread communication
    position_changed = pyqtSignal(int, int)
    video_loaded = pyqtSignal(str)
    
    def __init__(self, settings: QtConfig, message_bus: MessageBus = None):
        super().__init__()
        self.settings = settings
        self.mutex = QMutex()
        self.thread_pool = QThreadPool()
        self.thread_pool.setMaxThreadCount(4)
        self._message_bus = message_bus  # Store message bus reference for shutdown messages
        
        self.setup_ui()
        self.setup_media_player()
        self.setup_timers()
        
        logger.info("PlayerWindow initialized")
    
    def setup_ui(self):
        """Setup enhanced window UI"""
        self.setWindowTitle("MbetterClient - PyQt6 Video Player")
        
        # RADICAL APPROACH: Remove ALL transparency attributes and force complete opacity
        self.setAutoFillBackground(True)  # Critical: Fill background automatically
        
        # Use a simple black background color palette approach
        from PyQt6.QtGui import QPalette
        palette = self.palette()
        palette.setColor(QPalette.ColorRole.Window, QColor(0, 0, 0))  # Black background
        palette.setColor(QPalette.ColorRole.WindowText, QColor(255, 255, 255))  # White text
        palette.setColor(QPalette.ColorRole.Base, QColor(0, 0, 0))  # Black base
        self.setPalette(palette)
        
        # Simple black stylesheet
        self.setStyleSheet("QMainWindow { background-color: black; }")
        
        # Central widget - completely black and opaque
        central_widget = QWidget()
        central_widget.setAutoFillBackground(True)
        central_widget.setPalette(palette)
        central_widget.setStyleSheet("background-color: black;")
        self.setCentralWidget(central_widget)
        
        # Layout
        layout = QVBoxLayout(central_widget)
        layout.setContentsMargins(0, 0, 0, 0)
        layout.setSpacing(0)

        # Video widget with overlay - NO CONTROLS
        use_native = getattr(self.settings, 'use_native_overlay', False)
        logger.debug(f"PlayerWindow: use_native_overlay setting = {use_native}")
        self.video_widget = VideoWidget(parent=self, use_native_overlay=use_native)
        layout.addWidget(self.video_widget, 1)  # Full stretch - no controls
        
        # Controls removed per user request - clean overlay-only interface
        self.controls = None

        # Window settings - FORCE window to display properly
        if self.settings.fullscreen:
            self.showFullScreen()
        else:
            self.resize(self.settings.window_width, self.settings.window_height)
            
        # CRITICAL: Force window display and painting
        self.show()
        self.raise_()  # Bring to front
        self.activateWindow()  # Give focus
        self.repaint()  # Force repaint
        
        # Force central widget to repaint too
        if self.centralWidget():
            self.centralWidget().repaint()
        
        if self.settings.always_on_top:
            self.setWindowFlags(self.windowFlags() | Qt.WindowType.WindowStaysOnTopHint)
            self.show()  # Reshow after flag change
        
        # Setup menu
        self.setup_menu()
    
    
    def setup_menu(self):
        """Setup application menu"""
        menubar = self.menuBar()
        
        # File menu
        file_menu = menubar.addMenu('File')
        
        open_action = QAction('Open Video', self)
        open_action.triggered.connect(self.open_file_dialog)
        file_menu.addAction(open_action)
        
        # View menu
        view_menu = menubar.addMenu('View')
        
        fullscreen_action = QAction('Toggle Fullscreen', self)
        fullscreen_action.setShortcut('F11')
        fullscreen_action.triggered.connect(self.toggle_fullscreen)
        view_menu.addAction(fullscreen_action)
        
        stats_action = QAction('Toggle Stats', self)
        stats_action.setShortcut('S')
        stats_action.triggered.connect(self.toggle_stats)
        view_menu.addAction(stats_action)
    
    def setup_media_player(self):
        """Setup PyQt6 media player with audio output"""
        self.media_player = QMediaPlayer()
        self.audio_output = QAudioOutput()
        
        self.media_player.setAudioOutput(self.audio_output)
        self.media_player.setVideoOutput(self.video_widget.get_video_widget())
        
        # Connect signals
        self.media_player.playbackStateChanged.connect(self.on_state_changed)
        self.media_player.positionChanged.connect(self.on_position_changed)
        self.media_player.durationChanged.connect(self.on_duration_changed)
        self.media_player.errorOccurred.connect(self.on_media_error)
        self.media_player.mediaStatusChanged.connect(self.on_media_status_changed)
        
        # Set volume
        self.audio_output.setVolume(self.settings.volume)
        self.audio_output.setMuted(self.settings.mute)
    
    def setup_timers(self):
        """Setup various timers"""
        # Overlay update timer
        self.overlay_timer = QTimer()
        self.overlay_timer.timeout.connect(self.update_overlay_periodically)
        self.overlay_timer.start(1000)  # Update every second
        
        # Mouse tracking for window interactions (no controls to show/hide)
        self.setMouseTracking(True)
    
    def open_file_dialog(self):
        """Open file dialog to select video"""
        from PyQt6.QtWidgets import QFileDialog
        
        file_path, _ = QFileDialog.getOpenFileName(
            self,
            "Open Video File",
            "",
            "Video Files (*.mp4 *.avi *.mkv *.mov *.wmv *.flv *.webm *.m4v);;All Files (*)"
        )
        
        if file_path:
            self.play_video(file_path)
    
    def play_video(self, file_path: str, template_data: Dict[str, Any] = None):
        """Play video file with optional overlay data"""
        try:
            with QMutexLocker(self.mutex):
                # Handle both absolute and relative file paths
                path_obj = Path(file_path)
                if not path_obj.is_absolute():
                    # For relative paths, resolve them relative to the application directory
                    import os
                    app_dir = Path(os.path.dirname(os.path.dirname(os.path.dirname(__file__))))
                    absolute_path = app_dir / file_path
                    logger.debug(f"Resolving relative path '{file_path}' to '{absolute_path}'")
                else:
                    absolute_path = path_obj
                
                # Verify file exists before trying to play
                if not absolute_path.exists():
                    logger.error(f"Video file not found: {absolute_path}")
                    # Show error in overlay
                    overlay_view = self.video_widget.get_overlay_view()
                    error_data = {
                        'title': 'File Not Found',
                        'subtitle': f'Cannot find: {file_path}',
                        'ticker': 'Please check the file path and try again.'
                    }
                    if hasattr(overlay_view, 'overlay_channel') and overlay_view.overlay_channel:
                        if self._is_webengine_ready(overlay_view):
                            overlay_view.update_overlay_data(error_data)
                    else:
                        overlay_view.update_overlay_data(error_data)
                    return
                
                url = QUrl.fromLocalFile(str(absolute_path))
                logger.info(f"Loading video URL: {url.toString()}")
                self.media_player.setSource(url)
                
                # Update overlay with video info using safe method
                overlay_data = template_data or {}
                overlay_data.update({
                    'title': f'Playing: {Path(file_path).name}',
                    'subtitle': 'MbetterClient PyQt6 Player'
                })
                
                overlay_view = self.video_widget.get_overlay_view()
                if hasattr(overlay_view, 'overlay_channel') and overlay_view.overlay_channel:
                    # QWebEngineView overlay - use safe update
                    QTimer.singleShot(1000, lambda: self._send_safe_overlay_update(overlay_view, overlay_data))
                else:
                    # Native Qt overlay - immediate update is safe
                    overlay_view.update_overlay_data(overlay_data)
                
                if self.settings.auto_play:
                    self.media_player.play()
                
                # Start background metadata extraction
                worker = VideoProcessingWorker(
                    "metadata_extraction",
                    {"file_path": file_path},
                    self.on_metadata_extracted
                )
                self.thread_pool.start(worker)
                
                logger.info(f"Playing video: {file_path}")
                self.video_loaded.emit(file_path)
                
        except Exception as e:
            logger.error(f"Failed to play video: {e}")
    
    def on_metadata_extracted(self, metadata: Dict[str, Any]):
        """Handle extracted metadata"""
        if 'error' not in metadata:
            self.video_widget.get_overlay_view().update_video_info(metadata)
            logger.debug(f"Video metadata: {metadata}")
    
    def toggle_play_pause(self):
        """Toggle play/pause (thread-safe)"""
        with QMutexLocker(self.mutex):
            if self.media_player.playbackState() == QMediaPlayer.PlaybackState.PlayingState:
                self.media_player.pause()
            else:
                self.media_player.play()
    
    def stop_playback(self):
        """Stop playback (thread-safe)"""
        with QMutexLocker(self.mutex):
            self.media_player.stop()
    
    def seek_to_position(self, percentage: int):
        """Seek to position (percentage) (thread-safe)"""
        with QMutexLocker(self.mutex):
            duration = self.media_player.duration()
            if duration > 0:
                position = int(duration * percentage / 100)
                self.media_player.setPosition(position)
    
    def set_volume(self, volume: int):
        """Set volume (0-100) (thread-safe)"""
        with QMutexLocker(self.mutex):
            self.audio_output.setVolume(volume / 100.0)
    
    def toggle_fullscreen(self):
        """Toggle fullscreen mode"""
        if self.isFullScreen():
            self.showNormal()
        else:
            self.showFullScreen()
    
    def toggle_stats(self):
        """Toggle stats panel"""
        overlay_view = self.video_widget.get_overlay_view()
        # Toggle stats via WebChannel
        if overlay_view.overlay_channel:
            current_stats = getattr(self, '_stats_visible', False)
            overlay_view.overlay_channel.toggleStats(not current_stats)
            self._stats_visible = not current_stats
    
    def on_state_changed(self, state):
        """Handle playback state changes (thread-safe)"""
        is_playing = state == QMediaPlayer.PlaybackState.PlayingState
        # Controls removed - no button updates needed
        logger.debug(f"Playback state changed: {'playing' if is_playing else 'paused'}")
    
    def on_position_changed(self, position):
        """Handle position changes (thread-safe)"""
        duration = self.media_player.duration()
        # Controls removed - no slider updates needed
        
        # Update overlay with position info using safe method
        if duration > 0:
            overlay_view = self.video_widget.get_overlay_view()
            
            if hasattr(overlay_view, 'overlay_channel') and overlay_view.overlay_channel:
                # QWebEngineView overlay - check readiness before position update
                if self._is_webengine_ready(overlay_view):
                    overlay_view.update_position(
                        position / 1000.0,  # Convert to seconds
                        duration / 1000.0   # Convert to seconds
                    )
                # Skip position updates if WebEngine not ready
            else:
                # Native Qt overlay - always safe
                overlay_view.update_position(
                    position / 1000.0,  # Convert to seconds
                    duration / 1000.0   # Convert to seconds
                )
        
        # Emit signal for other components
        self.position_changed.emit(position, duration)
    
    def on_duration_changed(self, duration):
        """Handle duration changes"""
        # Controls removed - no updates needed
        logger.debug(f"Duration changed: {duration}ms")
    
    def on_media_error(self, error):
        """Handle media errors"""
        logger.error(f"Media player error: {error}")
        
        # Show error in overlay using safe update
        overlay_view = self.video_widget.get_overlay_view()
        error_data = {
            'title': 'Playback Error',
            'subtitle': f'Error: {error.name if hasattr(error, "name") else str(error)}',
            'ticker': 'Please check the video file and try again.'
        }
        
        if hasattr(overlay_view, 'overlay_channel') and overlay_view.overlay_channel:
            # QWebEngineView overlay - use safe update
            if self._is_webengine_ready(overlay_view):
                overlay_view.update_overlay_data(error_data)
            # Skip if not ready
        else:
            # Native Qt overlay - always safe
            overlay_view.update_overlay_data(error_data)
    
    def on_media_status_changed(self, status):
        """Handle media status changes"""
        logger.debug(f"Media status changed: {status}")
        
        if status == QMediaPlayer.MediaStatus.LoadedMedia:
            # Media loaded successfully - use safe update
            overlay_view = self.video_widget.get_overlay_view()
            status_data = {'subtitle': 'Media loaded successfully'}
            
            if hasattr(overlay_view, 'overlay_channel') and overlay_view.overlay_channel:
                # QWebEngineView overlay - use safe update
                if self._is_webengine_ready(overlay_view):
                    overlay_view.update_overlay_data(status_data)
                # Skip if not ready
            else:
                # Native Qt overlay - always safe
                overlay_view.update_overlay_data(status_data)
    
    def update_overlay_periodically(self):
        """Periodic overlay updates with WebEngine safety checks"""
        try:
            current_time = time.strftime("%H:%M:%S")
            overlay_view = self.video_widget.get_overlay_view()
            
            # Use safe update for WebEngine overlays
            if hasattr(overlay_view, 'overlay_channel') and overlay_view.overlay_channel:
                # QWebEngineView overlay - check readiness
                if self._is_webengine_ready(overlay_view):
                    overlay_view.update_overlay_data({'currentTime': current_time})
                # Skip update if not ready to prevent JavaScript errors
            else:
                # Native Qt overlay - always safe
                overlay_view.update_overlay_data({'currentTime': current_time})
                
        except Exception as e:
            logger.error(f"Periodic overlay update failed: {e}")
    
    def mouseMoveEvent(self, event):
        """Show controls on mouse movement"""
        super().mouseMoveEvent(event)
        self.show_controls()
        
        # Controls removed - no timer handling needed
        pass
    
    def keyPressEvent(self, event):
        """Handle key presses"""
        if event.key() == Qt.Key.Key_Space:
            self.toggle_play_pause()
        elif event.key() == Qt.Key.Key_Escape:
            if self.isFullScreen():
                self.showNormal()
            else:
                self.close()
        elif event.key() == Qt.Key.Key_F11:
            self.toggle_fullscreen()
        elif event.key() == Qt.Key.Key_M:
            current_muted = self.audio_output.isMuted()
            self.audio_output.setMuted(not current_muted)
        elif event.key() == Qt.Key.Key_S:
            self.toggle_stats()
        
        super().keyPressEvent(event)
    
    def show_controls(self):
        """Show controls - REMOVED per user request"""
        # Controls removed - clean overlay-only interface
        self.setCursor(Qt.CursorShape.ArrowCursor)
    
    def hide_controls(self):
        """Hide controls - REMOVED per user request"""
        # Controls removed - clean overlay-only interface
        if self.isFullScreen():
            self.setCursor(Qt.CursorShape.BlankCursor)
    
    def start_controls_timer(self):
        """Start timer to hide controls - REMOVED per user request"""
        # Controls removed - no timer needed
        pass
    
    def closeEvent(self, event):
        """Handle window close (thread-safe)"""
        with QMutexLocker(self.mutex):
            self.media_player.stop()
            self.thread_pool.waitForDone(3000)  # Wait up to 3 seconds for threads
            
        logger.info("Player window closing - Qt will handle application exit")
        event.accept()
    
    def _is_webengine_ready(self, overlay_view):
        """Check if WebEngine overlay is fully ready for updates"""
        try:
            # Check if overlay_view is a QWebEngineView
            if not hasattr(overlay_view, 'page'):
                return False
                
            page = overlay_view.page()
            if not page:
                return False
            
            # Check if WebChannel exists
            if not page.webChannel():
                return False
                
            # Check if URL is loaded (not about:blank)
            current_url = overlay_view.url().toString()
            if current_url == 'about:blank' or not current_url:
                return False
            
            # Check if the overlay_channel object exists and is ready
            if not (hasattr(overlay_view, 'overlay_channel') and overlay_view.overlay_channel):
                return False
                
            return True
            
        except Exception as e:
            logger.debug(f"WebEngine readiness check failed: {e}")
            return False
    
    def _send_safe_overlay_update(self, overlay_view, data):
        """Send overlay update with additional safety checks"""
        try:
            if not self._is_webengine_ready(overlay_view):
                logger.debug("WebEngine lost readiness, skipping update")
                return
                
            overlay_view.update_overlay_data(data)
            logger.debug("Safe overlay update completed")
            
        except Exception as e:
            logger.error(f"Failed to send safe overlay update: {e}")


class QtVideoPlayer:
    """PyQt6 video player component with message bus integration (replaces PyQt5 version)"""
    
    def __init__(self, message_bus: MessageBus, settings: QtConfig):
        self.name = "qt_player"
        self.message_bus = message_bus
        self.settings = settings
        self.app: Optional[QApplication] = None
        self.window: Optional[PlayerWindow] = None
        self.mutex = QMutex()
        
        # Register message queue
        self.message_queue = self.message_bus.register_component(self.name)
        
        # Message processing thread
        self.message_thread: Optional[threading.Thread] = None
        self.message_thread_running = False
        
        logger.info("QtVideoPlayer (PyQt6) initialized")
    
    def initialize(self) -> bool:
        """Initialize PyQt6 application and components"""
        try:
            with QMutexLocker(self.mutex):
                # Linux-specific system configuration
                self._configure_linux_system()
                
                # Create QApplication if it doesn't exist
                if not QApplication.instance():
                    self.app = QApplication(sys.argv)
                    self.app.setApplicationName("MbetterClient PyQt6")
                    self.app.setApplicationVersion("2.0.0")
                    self.app.setQuitOnLastWindowClosed(True)
                    
                    # CRITICAL: Ensure solid background rendering
                    pass  # Remove problematic attribute
                    
                    # Setup signal handling for proper shutdown
                    self._setup_signal_handlers()
                    
                    # Linux-specific application settings
                    self._configure_linux_app_settings()
                else:
                    self.app = QApplication.instance()
                
                # Create player window with message bus reference
                self.window = PlayerWindow(self.settings, self.message_bus)
                
                # CRITICAL: Force window to be completely opaque and visible
                self.window.setWindowOpacity(1.0)
                self.window.setAutoFillBackground(True)
                
                # Additional window display forcing
                QTimer.singleShot(100, lambda: self._force_window_display())
                
                # Connect window signals
                self.window.position_changed.connect(self._on_position_changed)
                self.window.video_loaded.connect(self._on_video_loaded)
                
                # Subscribe to messages
                self.message_bus.subscribe(self.name, MessageType.VIDEO_PLAY, self._handle_video_play)
                self.message_bus.subscribe(self.name, MessageType.VIDEO_PAUSE, self._handle_video_pause)
                self.message_bus.subscribe(self.name, MessageType.VIDEO_STOP, self._handle_video_stop)
                self.message_bus.subscribe(self.name, MessageType.VIDEO_SEEK, self._handle_video_seek)
                self.message_bus.subscribe(self.name, MessageType.VIDEO_VOLUME, self._handle_video_volume)
                self.message_bus.subscribe(self.name, MessageType.VIDEO_FULLSCREEN, self._handle_video_fullscreen)
                self.message_bus.subscribe(self.name, MessageType.TEMPLATE_CHANGE, self._handle_template_change)
                self.message_bus.subscribe(self.name, MessageType.OVERLAY_UPDATE, self._handle_overlay_update)
                self.message_bus.subscribe(self.name, MessageType.STATUS_REQUEST, self._handle_status_request)
                
                # Delay loading default overlay to allow JavaScript initialization
                QTimer.singleShot(2000, self._load_default_overlay)  # Wait 2 seconds
                
                logger.info("QtVideoPlayer (PyQt6) initialized successfully")
                return True
                
        except Exception as e:
            logger.error(f"QtVideoPlayer initialization failed: {e}")
            return False
    
    def _force_window_display(self):
        """Force window to display properly - aggressive approach"""
        try:
            if self.window:
                logger.debug("Forcing window display and repaint")
                self.window.show()
                self.window.raise_()
                self.window.activateWindow()
                self.window.repaint()
                
                # Force video widget to be visible and painted
                video_widget = self.window.video_widget
                if video_widget:
                    video_widget.show()
                    video_widget.repaint()
                    
                    # Force base video widget to be visible
                    base_video = video_widget.get_video_widget()
                    if base_video:
                        base_video.show()
                        base_video.repaint()
                
                logger.debug("Window display forced")
        except Exception as e:
            logger.error(f"Failed to force window display: {e}")
    
    def _configure_linux_system(self):
        """Configure Linux-specific system settings"""
        import platform
        import os
        
        if platform.system() != 'Linux':
            return
            
        try:
            # Set environment variables for better Linux compatibility and rendering stability
            linux_env_vars = {
                'QT_QPA_PLATFORM': 'xcb',
                'QT_AUTO_SCREEN_SCALE_FACTOR': '1',
                'QT_SCALE_FACTOR': '1',
                'QT_LOGGING_RULES': 'qt.qpa.xcb.info=false;qt.qpa.xcb.xcberror.warning=false',
                'QTWEBENGINE_CHROMIUM_FLAGS': '--no-sandbox --disable-gpu-sandbox',
                'QTWEBENGINE_DISABLE_SANDBOX': '1',
                # CRITICAL: Fix OpenGL rendering issues that cause desktop transparency
                'QT_OPENGL': 'software',  # Use software rendering instead of hardware
                'QT_QUICK_BACKEND': 'software',  # Software backend for Qt Quick
                'QT_XCB_GL_INTEGRATION': 'none',  # Disable XCB OpenGL integration
                'LIBGL_ALWAYS_SOFTWARE': '1',  # Force software OpenGL
                'QT_OPENGL_BUGLIST': 'disable',  # Disable OpenGL bug workarounds
            }
            
            for key, value in linux_env_vars.items():
                if key not in os.environ:
                    os.environ[key] = value
                    logger.debug(f"Set Linux env var: {key}={value}")
            
            # Handle headless/server environments
            if not os.environ.get('DISPLAY'):
                logger.warning("No DISPLAY environment variable - may be running headless")
                os.environ['QT_QPA_PLATFORM'] = 'offscreen'
            
        except Exception as e:
            logger.warning(f"Linux system configuration warning: {e}")
    
    def _configure_linux_app_settings(self):
        """Configure Linux-specific QApplication settings"""
        import platform
        
        if platform.system() != 'Linux' or not self.app:
            return
            
        try:
            # Set application attributes for better Linux compatibility
            self.app.setAttribute(Qt.ApplicationAttribute.AA_UseHighDpiPixmaps, True)
            self.app.setAttribute(Qt.ApplicationAttribute.AA_EnableHighDpiScaling, True)
            
            # Handle X11 specific settings
            if hasattr(Qt.ApplicationAttribute, 'AA_X11InitThreads'):
                self.app.setAttribute(Qt.ApplicationAttribute.AA_X11InitThreads, True)
            
            logger.debug("Applied Linux-specific application settings")
            
        except Exception as e:
            logger.warning(f"Linux application configuration warning: {e}")
    
    def _load_default_overlay(self):
        """Load default overlay display with enhanced WebChannel readiness checking"""
        try:
            default_data = {
                'title': 'MbetterClient PyQt6 Player',
                'subtitle': 'Ready for Content',
                'ticker': 'Welcome to MbetterClient • Professional Video Overlay System • Real-time Updates • Hardware Accelerated Playback',
                'showStats': False
            }
            
            if not self.window:
                logger.debug("Window not ready for overlay loading")
                return
                
            overlay_view = self.window.video_widget.get_overlay_view()
            
            # Handle different overlay types
            if hasattr(overlay_view, 'overlay_channel') and overlay_view.overlay_channel:
                # QWebEngineView overlay - enhanced readiness checking
                if not self._is_webengine_ready(overlay_view):
                    logger.debug("WebEngine overlay not ready, retrying in 1 second...")
                    QTimer.singleShot(1000, self._load_default_overlay)
                    return
                
                # Additional delay before sending data to ensure JavaScript is fully initialized
                logger.debug("WebEngine ready, scheduling overlay update...")
                QTimer.singleShot(500, lambda: self._send_safe_overlay_update(overlay_view, default_data))
                
            else:
                # Native Qt overlay - direct update (always works)
                overlay_view.update_overlay_data(default_data)
                logger.info("Default native overlay loaded successfully")
                
        except Exception as e:
            logger.error(f"Failed to load default overlay: {e}")
            # Retry on error with longer delay
            QTimer.singleShot(3000, self._load_default_overlay)
    
    def _is_webengine_ready(self, overlay_view):
        """Check if WebEngine overlay is fully ready for updates"""
        try:
            # Check if overlay_view is a QWebEngineView
            if not hasattr(overlay_view, 'page'):
                return False
                
            page = overlay_view.page()
            if not page:
                return False
            
            # Check if WebChannel exists
            if not page.webChannel():
                return False
                
            # Check if URL is loaded (not about:blank)
            current_url = overlay_view.url().toString()
            if current_url == 'about:blank' or not current_url:
                return False
            
            # Check if the overlay_channel object exists and is ready
            if not (hasattr(overlay_view, 'overlay_channel') and overlay_view.overlay_channel):
                return False
                
            return True
            
        except Exception as e:
            logger.debug(f"WebEngine readiness check failed: {e}")
            return False
    
    def _send_safe_overlay_update(self, overlay_view, data):
        """Send overlay update with additional safety checks"""
        try:
            if not self._is_webengine_ready(overlay_view):
                logger.debug("WebEngine lost readiness, skipping update")
                return
                
            overlay_view.update_overlay_data(data)
            logger.info("Default WebEngine overlay loaded successfully")
            
        except Exception as e:
            logger.error(f"Failed to send safe overlay update: {e}")
    
    def start_message_processing(self):
        """Start message processing in a separate thread"""
        if self.message_thread_running:
            logger.warning("Message processing thread is already running")
            return
            
        self.message_thread_running = True
        self.message_thread = threading.Thread(
            target=self._message_processing_loop,
            name="QtPlayerMessageThread",
            daemon=True
        )
        self.message_thread.start()
        logger.info("QtPlayer message processing thread started")
    
    def stop_message_processing(self):
        """Stop message processing thread"""
        if self.message_thread_running:
            self.message_thread_running = False
            if self.message_thread and self.message_thread.is_alive():
                logger.debug("Waiting for message thread to stop...")
                self.message_thread.join(timeout=2.0)
                
                # If thread is still alive, it's probably stuck
                if self.message_thread.is_alive():
                    logger.warning("Message thread did not stop gracefully within timeout")
                else:
                    logger.info("QtPlayer message processing thread stopped")
            else:
                logger.info("QtPlayer message processing thread already stopped")
    
    def _message_processing_loop(self):
        """Process messages from message bus in a separate thread"""
        try:
            logger.debug("QtPlayer message processing loop started")
            
            while self.message_thread_running:
                try:
                    # Process messages from message bus
                    message = self.message_bus.get_message(self.name, timeout=0.1)
                    if message:
                        self._process_message(message)
                    
                    # Send periodic progress updates if playing
                    if (self.window and
                        self.window.media_player.playbackState() == QMediaPlayer.PlaybackState.PlayingState):
                        self._send_progress_update()
                        
                except Exception as e:
                    logger.error(f"QtPlayer message processing error: {e}")
                    time.sleep(0.1)
                    
        except Exception as e:
            logger.error(f"QtPlayer message processing loop failed: {e}")
        finally:
            logger.debug("QtPlayer message processing loop ended")
    
    def run(self):
        """Run the Qt event loop (this should be called on the main thread)"""
        try:
            logger.info("QtVideoPlayer starting Qt event loop")
            
            # Send ready status
            ready_message = MessageBuilder.system_status(
                sender=self.name,
                status="ready",
                details={"fullscreen": self.settings.fullscreen, "version": "PyQt6-2.0.0"}
            )
            self.message_bus.publish(ready_message)
            
            # Start message processing in separate thread
            self.start_message_processing()
            
            # Run Qt event loop (this blocks until app quits)
            if self.app:
                return self.app.exec()
            else:
                logger.error("No QApplication instance available")
                return 1
                
        except Exception as e:
            logger.error(f"QtVideoPlayer run failed: {e}")
            return 1
    
    def _setup_signal_handlers(self):
        """Setup Qt-specific signal handlers - let main app handle Python signals"""
        try:
            # Only handle Qt application aboutToQuit signal
            # Let the main application handle SIGINT/SIGTERM to avoid conflicts
            if self.app:
                self.app.aboutToQuit.connect(self._handle_app_quit)
            
            logger.debug("Qt signal handlers setup completed")
            
        except Exception as e:
            logger.warning(f"Qt signal handler setup failed: {e}")
    
    def _handle_app_quit(self):
        """Handle Qt application quit signal"""
        logger.info("Qt application about to quit")
        
        # Ensure message processing thread is stopped
        self.stop_message_processing()
    
    def shutdown(self):
        """Shutdown video player"""
        try:
            logger.info("Shutting down QtVideoPlayer...")
            
            # Stop message processing thread with timeout
            self.stop_message_processing()
            
            with QMutexLocker(self.mutex):
                if self.window:
                    self.window.close()
                    self.window = None
                
                if self.app and not self.app.closingDown():
                    self.app.quit()
            
        except Exception as e:
            logger.error(f"QtVideoPlayer shutdown error: {e}")
    
    def _process_message(self, message: Message):
        """Process received message"""
        try:
            # Messages are handled by subscribed handlers automatically
            # This is just for additional processing if needed
            pass
        except Exception as e:
            logger.error(f"Failed to process message: {e}")
    
    def _on_position_changed(self, position: int, duration: int):
        """Handle position changes from player window"""
        try:
            # Send progress update via message bus
            if duration > 0:
                percentage = (position / duration) * 100
                progress_message = MessageBuilder.video_progress(
                    sender=self.name,
                    position=position / 1000.0,  # Convert to seconds
                    duration=duration / 1000.0,  # Convert to seconds
                    percentage=percentage
                )
                self.message_bus.publish(progress_message, broadcast=True)
        except Exception as e:
            logger.error(f"Failed to send progress update: {e}")
    
    def _on_video_loaded(self, file_path: str):
        """Handle video loaded event"""
        try:
            logger.info(f"Video loaded: {file_path}")
            # Send video loaded status message
            loaded_message = MessageBuilder.system_status(
                sender=self.name,
                status="video_loaded",
                details={"file_path": file_path}
            )
            self.message_bus.publish(loaded_message, broadcast=True)
        except Exception as e:
            logger.error(f"Failed to handle video loaded event: {e}")
    
    def _send_progress_update(self):
        """Send video progress update"""
        try:
            if self.window and self.window.media_player.duration() > 0:
                position = self.window.media_player.position()
                duration = self.window.media_player.duration()
                percentage = (position / duration) * 100 if duration > 0 else 0
                
                progress_message = MessageBuilder.video_progress(
                    sender=self.name,
                    position=position / 1000.0,  # Convert to seconds
                    duration=duration / 1000.0,  # Convert to seconds
                    percentage=percentage
                )
                self.message_bus.publish(progress_message, broadcast=True)
                
        except Exception as e:
            logger.error(f"Failed to send progress update: {e}")
    
    # Message handlers for various video control commands
    def _handle_video_play(self, message: Message):
        """Handle video play message"""
        try:
            file_path = message.data.get("file_path")
            template_data = message.data.get("overlay_data", {})
            
            if not file_path:
                logger.error("No file path provided for video play")
                return
            
            logger.info(f"Playing video: {file_path}")
            
            if self.window:
                # Use Qt's signal/slot mechanism to ensure GUI operations happen on main thread
                self.window.play_video(file_path, template_data)
            
        except Exception as e:
            logger.error(f"Failed to handle video play: {e}")
    
    def _handle_video_pause(self, message: Message):
        """Handle video pause message"""
        try:
            if self.window:
                # Use Qt's signal/slot mechanism to ensure GUI operations happen on main thread
                self.window.media_player.pause()
        except Exception as e:
            logger.error(f"Failed to handle video pause: {e}")
    
    def _handle_video_stop(self, message: Message):
        """Handle video stop message"""
        try:
            if self.window:
                # Use Qt's signal/slot mechanism to ensure GUI operations happen on main thread
                self.window.stop_playback()
        except Exception as e:
            logger.error(f"Failed to handle video stop: {e}")
    
    def _handle_video_seek(self, message: Message):
        """Handle video seek message"""
        try:
            position = message.data.get("position", 0)
            if self.window:
                duration = self.window.media_player.duration()
                if duration > 0:
                    percentage = int(position * 100 / duration)
                    # Use Qt's signal/slot mechanism to ensure GUI operations happen on main thread
                    self.window.seek_to_position(percentage)
        except Exception as e:
            logger.error(f"Failed to handle video seek: {e}")
    
    def _handle_video_volume(self, message: Message):
        """Handle video volume message"""
        try:
            volume = message.data.get("volume", 100)
            if self.window:
                # Use Qt's signal/slot mechanism to ensure GUI operations happen on main thread
                self.window.set_volume(volume)
        except Exception as e:
            logger.error(f"Failed to handle video volume: {e}")
    
    def _handle_video_fullscreen(self, message: Message):
        """Handle video fullscreen message"""
        try:
            fullscreen = message.data.get("fullscreen", True)
            if self.window:
                # Use Qt's signal/slot mechanism to ensure GUI operations happen on main thread
                if fullscreen:
                    self.window.showFullScreen()
                else:
                    self.window.showNormal()
        except Exception as e:
            logger.error(f"Failed to handle video fullscreen: {e}")
    
    def _handle_template_change(self, message: Message):
        """Handle template change message"""
        try:
            template_data = message.data.get("template_data", {})
            
            if self.window and template_data:
                # Use Qt's signal/slot mechanism to ensure GUI operations happen on main thread
                overlay_view = self.window.video_widget.get_overlay_view()
                overlay_view.update_overlay_data(template_data)
                
        except Exception as e:
            logger.error(f"Failed to handle template change: {e}")
    
    def _handle_overlay_update(self, message: Message):
        """Handle overlay update message"""
        try:
            overlay_data = message.data.get("overlay_data", {})
            
            if self.window and overlay_data:
                # Use Qt's signal/slot mechanism to ensure GUI operations happen on main thread
                overlay_view = self.window.video_widget.get_overlay_view()
                overlay_view.update_overlay_data(overlay_data)
                
        except Exception as e:
            logger.error(f"Failed to handle overlay update: {e}")
    
    def _handle_status_request(self, message: Message):
        """Handle status request message"""
        try:
            if self.window:
                # Use Qt's signal/slot mechanism to ensure GUI operations happen on main thread
                is_playing = (self.window.media_player.playbackState() ==
                            QMediaPlayer.PlaybackState.PlayingState)
                position = self.window.media_player.position() / 1000.0  # seconds
                duration = self.window.media_player.duration() / 1000.0  # seconds
                volume = self.window.audio_output.volume() * 100
                muted = self.window.audio_output.isMuted()
                fullscreen = self.window.isFullScreen()
                
                status_response = MessageBuilder.system_status(
                    sender=self.name,
                    status="status_response",
                    details={
                        "player_status": "playing" if is_playing else "paused",
                        "position": position,
                        "duration": duration,
                        "volume": volume,
                        "muted": muted,
                        "fullscreen": fullscreen
                    }
                )
                # Send response back to requester
                status_response.recipient = message.sender
                self.message_bus.publish(status_response)
                
        except Exception as e:
            logger.error(f"Failed to handle status request: {e}")
