#!/usr/bin/env python3
"""
Test script for video playback functionality in the web player
"""

import sys
import time
import logging
from pathlib import Path
from PyQt6.QtWidgets import QApplication
from PyQt6.QtCore import QUrl, QTimer
from PyQt6.QtWebEngineWidgets import QWebEngineView
from PyQt6.QtWebEngineCore import QWebEngineProfile

# Setup logging
logging.basicConfig(level=logging.INFO)
logger = logging.getLogger(__name__)

def test_video_playback():
    """Test video playback in a QWebEngineView"""
    try:
        logger.info("Testing video playback in QWebEngineView...")

        # Create QApplication
        app = QApplication(sys.argv)

        # Create a simple web view
        web_view = QWebEngineView()
        web_view.setWindowTitle("Video Playback Test")
        web_view.resize(800, 600)

        # Get the path to the sample video
        sample_video_path = Path(__file__).parent / "assets" / "INTRO.mp4"
        if not sample_video_path.exists():
            # Try to find any video file
            assets_dir = Path(__file__).parent / "assets"
            video_files = list(assets_dir.glob("*.mp4"))
            if video_files:
                sample_video_path = video_files[0]
            else:
                logger.error("No video files found for testing")
                return False

        # Create HTML content with video player
        video_url = QUrl.fromLocalFile(str(sample_video_path)).toString()
        html_content = f"""
        <!DOCTYPE html>
        <html>
        <head>
            <title>Video Playback Test</title>
            <style>
                * {{
                    margin: 0;
                    padding: 0;
                    box-sizing: border-box;
                }}

                body {{
                    margin: 0;
                    padding: 0;
                    background: black;
                    display: flex;
                    justify-content: center;
                    align-items: center;
                    height: 100vh;
                    width: 100vw;
                    overflow: hidden;
                }}

                .video-container {{
                    position: absolute;
                    top: 0;
                    left: 0;
                    width: 100%;
                    height: 100%;
                    background: black;
                    z-index: 100;
                }}

                video {{
                    width: 100%;
                    height: 100%;
                    object-fit: contain;
                    background: black;
                    visibility: visible !important;
                    opacity: 1 !important;
                    display: block !important;
                }}

                .status {{
                    position: absolute;
                    top: 20px;
                    left: 20px;
                    color: white;
                    font-family: Arial, sans-serif;
                    font-size: 18px;
                    z-index: 1000;
                    background: rgba(0,0,0,0.5);
                    padding: 5px 10px;
                    border-radius: 3px;
                }}
            </style>
        </head>
        <body>
            <div class="status" id="status">Loading video...</div>
            <div class="video-container">
                <video id="testVideo" controls autoplay playsinline>
                    <source src="{video_url}" type="video/mp4">
                    Your browser does not support the video tag.
                </video>
            </div>

            <script>
                // Debug video element
                const video = document.getElementById('testVideo');
                const status = document.getElementById('status');

                console.log('Video element:', video);
                console.log('Video style visibility:', window.getComputedStyle(video).visibility);
                console.log('Video style opacity:', window.getComputedStyle(video).opacity);
                console.log('Video style display:', window.getComputedStyle(video).display);

                // Force video to be visible
                video.style.visibility = 'visible';
                video.style.opacity = '1';
                video.style.display = 'block';

                video.addEventListener('loadedmetadata', function() {{
                    status.textContent = 'Video loaded: ' + video.duration + ' seconds';
                    console.log('Video metadata loaded');
                }});

                video.addEventListener('play', function() {{
                    status.textContent = 'Video playing';
                    console.log('Video started playing');
                }});

                video.addEventListener('pause', function() {{
                    status.textContent = 'Video paused';
                    console.log('Video paused');
                }});

                video.addEventListener('ended', function() {{
                    status.textContent = 'Video ended';
                    console.log('Video ended');
                }});

                video.addEventListener('error', function(e) {{
                    status.textContent = 'Video error: ' + e.message;
                    console.error('Video error:', e);
                }});

                // Try to play the video
                video.play().catch(function(e) {{
                    status.textContent = 'Playback failed: ' + e.message;
                    console.error('Playback failed:', e);
                }});
            </script>
        </body>
        </html>
        """

        # Load the HTML content
        web_view.setHtml(html_content)
        web_view.show()

        logger.info(f"Testing video playback with: {sample_video_path}")
        logger.info("Web view should appear with video playback...")

        # Run the application for a short time to test
        app.exec()

        return True

    except Exception as e:
        logger.error(f"Video playback test failed: {e}")
        import traceback
        logger.error(f"Traceback: {traceback.format_exc()}")
        return False

if __name__ == "__main__":
    success = test_video_playback()
    sys.exit(0 if success else 1)