/**
 * Web Player JavaScript API
 * Provides communication between Qt WebChannel and the HTML5 video player
 */

// Web Player API
class WebPlayerAPI {
    constructor() {
        this.videoElement = null;
        this.overlayData = {};
        this.webChannel = null;
        this.currentVideoPath = null;
        this.timerInterval = null;
        this.currentTime = 0;
    }

    // Initialize the web player
    init() {
        console.log('Initializing Web Player API');

        // Get video element
        this.videoElement = document.getElementById('webVideoPlayer');
        if (!this.videoElement) {
            console.error('Video element not found');
            return false;
        }

        // Set up event listeners
        this.setupEventListeners();

        // Initialize WebChannel if available
        this.initWebChannel();

        return true;
    }

    // Set up event listeners
    setupEventListeners() {
        if (!this.videoElement) return;

        this.videoElement.addEventListener('play', () => {
            console.log('Video started playing');
            this.startTimer();
            this.sendPlayerStatus('playing');
        });

        this.videoElement.addEventListener('pause', () => {
            console.log('Video paused');
            this.stopTimer();
            this.sendPlayerStatus('paused');
        });

        this.videoElement.addEventListener('ended', () => {
            console.log('Video ended');
            this.stopTimer();
            this.sendPlayerStatus('ended');
        });

        this.videoElement.addEventListener('timeupdate', () => {
            this.updateTimer();
        });

        this.videoElement.addEventListener('loadedmetadata', () => {
            console.log('Video metadata loaded');
            this.sendVideoInfo();
        });
    }

    // Initialize WebChannel
    initWebChannel() {
        if (typeof QWebChannel !== 'undefined') {
            new QWebChannel(qt.webChannelTransport, (channel) => {
                console.log('WebChannel initialized for web player');
                this.webChannel = channel;

                // Connect to overlay object if available
                if (channel.objects.overlay) {
                    this.overlayChannel = channel.objects.overlay;

                    // Connect signals
                    if (this.overlayChannel.dataUpdated) {
                        this.overlayChannel.dataUpdated.connect((data) => {
                            this.handleOverlayData(data);
                        });
                    }

                    if (this.overlayChannel.playVideo) {
                        this.overlayChannel.playVideo.connect((filePath) => {
                            this.playVideo(filePath);
                        });
                    }

                    if (this.overlayChannel.updateOverlayData) {
                        this.overlayChannel.updateOverlayData.connect((data) => {
                            this.handleOverlayData(data);
                        });
                    }

                    // Get initial data
                    if (this.overlayChannel.getCurrentData) {
                        this.overlayChannel.getCurrentData((data) => {
                            this.handleOverlayData(data);
                        });
                    }
                }
            });
        }
    }

    // Handle overlay data from Qt
    handleOverlayData(data) {
        console.log('Web player received overlay data:', data);
        this.overlayData = data || {};

        // Update UI with overlay data
        this.updateUIFromOverlayData();

        // Check for video play command
        if (data && data.videoPath) {
            this.playVideo(data.videoPath);
        }
    }

    // Update UI from overlay data
    updateUIFromOverlayData() {
        // Update title, message, etc. from overlay data
        if (this.overlayData.title) {
            const titleElement = document.getElementById('messageTitle');
            if (titleElement) {
                titleElement.textContent = this.overlayData.title;
            }
        }

        if (this.overlayData.message) {
            const messageElement = document.getElementById('messageContent');
            if (messageElement) {
                messageElement.textContent = this.overlayData.message;
            }
        }

        // Update fighter names and venue if available
        if (this.overlayData.fighterNames) {
            const fighterNamesElement = document.getElementById('fighterNames');
            if (fighterNamesElement) {
                fighterNamesElement.textContent = this.overlayData.fighterNames;
            }
        }

        if (this.overlayData.venueInfo) {
            const venueInfoElement = document.getElementById('venueInfo');
            if (venueInfoElement) {
                venueInfoElement.textContent = this.overlayData.venueInfo;
            }
        }
    }

    // Play video
    playVideo(filePath) {
        if (!this.videoElement) {
            console.error('Video element not available');
            return;
        }

        console.log('Playing video:', filePath);
        this.currentVideoPath = filePath;

        // CRITICAL FIX: Ensure video element is always visible and properly configured
        // This fixes the grey background issue
        this.videoElement.style.display = 'block';
        this.videoElement.style.visibility = 'visible';
        this.videoElement.style.opacity = '1';
        this.videoElement.style.width = '100%';
        this.videoElement.style.height = '100%';
        this.videoElement.style.objectFit = 'contain';
        this.videoElement.style.backgroundColor = 'black';
        this.videoElement.style.position = 'absolute';
        this.videoElement.style.top = '0';
        this.videoElement.style.left = '0';
        this.videoElement.style.zIndex = '100';

        // Force video container to be visible
        const videoContainer = document.querySelector('.video-container');
        if (videoContainer) {
            videoContainer.style.display = 'block';
            videoContainer.style.visibility = 'visible';
            videoContainer.style.opacity = '1';
            videoContainer.style.zIndex = '100';
            videoContainer.style.backgroundColor = 'black';
            videoContainer.style.position = 'absolute';
            videoContainer.style.top = '0';
            videoContainer.style.left = '0';
            videoContainer.style.width = '100%';
            videoContainer.style.height = '100%';
        }

        // Set video source and load
        this.videoElement.src = filePath;

        // Add event listeners for better debugging
        this.videoElement.onloadeddata = () => {
            console.log('Video data loaded, attempting to play...');
            this._attemptPlayback();
        };

        this.videoElement.oncanplay = () => {
            console.log('Video can play, attempting to play...');
            this._attemptPlayback();
        };

        this.videoElement.onerror = (e) => {
            console.error('Video element error:', e);
            this.sendPlayerError('load_failed', 'Failed to load video source');
        };

        // Load the video
        this.videoElement.load();

        // Debug video element state
        console.log('Video element after load:', this.videoElement);
        console.log('Video readyState:', this.videoElement.readyState);
        console.log('Video networkState:', this.videoElement.networkState);
        console.log('Video error:', this.videoElement.error);
    }

    // Attempt playback with autoplay policy handling
    _attemptPlayback() {
        if (!this.videoElement) return;

        // Check if video is ready to play
        if (this.videoElement.readyState >= HTMLMediaElement.HAVE_FUTURE_DATA) {
            console.log('Video ready, attempting playback...');

            // CRITICAL FIX: Always ensure video is visible before playback
            this.videoElement.style.display = 'block';
            this.videoElement.style.visibility = 'visible';
            this.videoElement.style.opacity = '1';

            // CRITICAL FIX: Set muted to true to allow autoplay in modern browsers
            this.videoElement.muted = true;

            // Try to play with autoplay policy handling
            this.videoElement.play().then(() => {
                console.log('Playback started successfully');
                // Ensure video is visible after successful playback
                this.videoElement.style.display = 'block';
                this.videoElement.style.visibility = 'visible';
                this.videoElement.style.opacity = '1';
            }).catch(e => {
                console.error('Playback failed (likely due to autoplay policy):', e);

                // If autoplay is blocked, show controls and let user interact
                this.videoElement.controls = true;
                this.videoElement.muted = true; // Muted videos can often autoplay

                // CRITICAL FIX: Always ensure video is visible even if autoplay fails
                this.videoElement.style.display = 'block';
                this.videoElement.style.visibility = 'visible';
                this.videoElement.style.opacity = '1';

                // Try again with muted
                this.videoElement.play().catch(e2 => {
                    console.error('Muted playback also failed:', e2);
                    this.sendPlayerError('autoplay_blocked', 'Autoplay blocked by browser policy');
                });
            });
        } else {
            console.log('Video not ready yet, waiting for more data...');

            // CRITICAL FIX: Always ensure video is visible even while waiting
            this.videoElement.style.display = 'block';
            this.videoElement.style.visibility = 'visible';
            this.videoElement.style.opacity = '1';
        }
    }

    // Start timer
    startTimer() {
        this.stopTimer(); // Clear any existing timer

        this.currentTime = 0;
        this.updateTimerDisplay();

        this.timerInterval = setInterval(() => {
            this.currentTime++;
            this.updateTimerDisplay();
        }, 1000);
    }

    // Stop timer
    stopTimer() {
        if (this.timerInterval) {
            clearInterval(this.timerInterval);
            this.timerInterval = null;
        }
    }

    // Update timer display
    updateTimerDisplay() {
        const timerElement = document.getElementById('matchTimer');
        if (timerElement) {
            const minutes = Math.floor(this.currentTime / 60);
            const seconds = this.currentTime % 60;
            const timeString = `${minutes.toString().padStart(2, '0')}:${seconds.toString().padStart(2, '0')}`;
            timerElement.textContent = timeString;
        }
    }

    // Update timer based on video position
    updateTimer() {
        if (this.videoElement && !isNaN(this.videoElement.duration) && this.videoElement.duration > 0) {
            this.currentTime = Math.floor(this.videoElement.currentTime);
            this.updateTimerDisplay();
        }
    }

    // Send player status to Qt
    sendPlayerStatus(status) {
        if (this.webChannel && this.overlayChannel && this.overlayChannel.sendPlayerStatus) {
            this.overlayChannel.sendPlayerStatus(status);
        }
    }

    // Send video info to Qt
    sendVideoInfo() {
        if (this.videoElement && this.webChannel && this.overlayChannel && this.overlayChannel.sendVideoInfo) {
            const videoInfo = {
                duration: this.videoElement.duration,
                width: this.videoElement.videoWidth,
                height: this.videoElement.videoHeight,
                currentTime: this.videoElement.currentTime
            };
            this.overlayChannel.sendVideoInfo(videoInfo);
        }
    }

    // Send player error to Qt
    sendPlayerError(errorType, errorMessage) {
        if (this.webChannel && this.overlayChannel && this.overlayChannel.sendPlayerError) {
            this.overlayChannel.sendPlayerError(errorType, errorMessage);
        }
    }

    // Get current player state
    getPlayerState() {
        if (!this.videoElement) return null;

        return {
            playing: !this.videoElement.paused,
            currentTime: this.videoElement.currentTime,
            duration: this.videoElement.duration,
            volume: this.videoElement.volume,
            muted: this.videoElement.muted,
            videoPath: this.currentVideoPath
        };
    }
}

// Initialize the web player when DOM is loaded
document.addEventListener('DOMContentLoaded', function() {
    const webPlayer = new WebPlayerAPI();
    webPlayer.init();

    // Expose to global scope for debugging
    window.webPlayer = webPlayer;

    // Expose functions globally for WebChannel communication
    window.playVideo = function(filePath) {
        console.log('Global playVideo called with:', filePath);
        webPlayer.playVideo(filePath);
    };

    window.updateOverlayData = function(data) {
        console.log('Global updateOverlayData called with:', data);
        webPlayer.handleOverlayData(data);
    };

    console.log('Web Player API initialized and exposed to window.webPlayer');
    console.log('Global functions playVideo and updateOverlayData exposed for WebChannel');

    // Initialize debug mode if detected
    if (window.location.href.includes('debug') || window.location.href.includes('debug-player')) {
        console.log('DEBUG MODE DETECTED: Applying debug-specific video fixes');
        webPlayer._initDebugMode();
    }
});

// Add debug mode initialization method to WebPlayerAPI prototype
WebPlayerAPI.prototype._initDebugMode = function() {
    console.log('Initializing debug mode for web player');

    // Ensure video element is visible in debug mode
    if (this.videoElement) {
        this.videoElement.style.display = 'block';
        this.videoElement.style.visibility = 'visible';
        this.videoElement.style.opacity = '1';
        this.videoElement.style.width = '100%';
        this.videoElement.style.height = '100%';
        this.videoElement.style.objectFit = 'contain';
        this.videoElement.muted = true; // Ensure muted for autoplay in debug mode

        // Force video container to be visible
        const videoContainer = document.querySelector('.video-container');
        if (videoContainer) {
            videoContainer.style.display = 'block';
            videoContainer.style.visibility = 'visible';
            videoContainer.style.opacity = '1';
            videoContainer.style.zIndex = '100';
        }

        // Add debug overlay to indicate debug mode
        const debugOverlay = document.createElement('div');
        debugOverlay.style.position = 'absolute';
        debugOverlay.style.top = '10px';
        debugOverlay.style.right = '10px';
        debugOverlay.style.backgroundColor = 'rgba(255, 0, 0, 0.7)';
        debugOverlay.style.color = 'white';
        debugOverlay.style.padding = '5px 10px';
        debugOverlay.style.borderRadius = '5px';
        debugOverlay.style.fontFamily = 'Arial, sans-serif';
        debugOverlay.style.fontSize = '14px';
        debugOverlay.style.zIndex = '1000';
        debugOverlay.textContent = 'DEBUG MODE - Video Player';
        document.body.appendChild(debugOverlay);

        console.log('Debug mode initialization completed');
    } else {
        console.error('DEBUG MODE: Video element not available for debug initialization');
    }
};