#!/usr/bin/env python3
"""
Script to check recent bets in the database
"""

import sys
from pathlib import Path

# Add the project root to Python path
project_root = Path(__file__).parent
sys.path.insert(0, str(project_root))

from mbetterclient.database.models import BetModel, BetDetailModel
from mbetterclient.database.manager import DatabaseManager
from mbetterclient.config.settings import get_user_data_dir

def check_recent_bets():
    """Check recent bets in the database"""

    # Use the default database path
    db_path = get_user_data_dir() / "mbetterclient.db"
    db_manager = DatabaseManager(str(db_path))

    if not db_manager.initialize():
        print("Failed to initialize database")
        return False

    session = db_manager.get_session()
    try:
        # Get recent bets (last 10)
        recent_bets = session.query(BetModel).order_by(BetModel.bet_datetime.desc()).limit(10).all()
        print(f'Found {len(recent_bets)} recent bets:')
        for bet in recent_bets:
            details = session.query(BetDetailModel).filter_by(bet_id=bet.uuid).all()
            print(f'  Bet {bet.uuid[:8]}... - {len(details)} details - {bet.bet_datetime}')
            for detail in details:
                print(f'    * {detail.outcome} - ${detail.amount}')

        return True

    except Exception as e:
        print(f"Error checking bets: {e}")
        return False
    finally:
        session.close()
        db_manager.close()

if __name__ == "__main__":
    check_recent_bets()